#!/usr/local/bin/python
#

"htmldoc.py - Create web pages for BIRCH documentation from birchdb database"

# 
# Version 4/ 2/08
#
# Synopsis: htmldoc.py


import sys
import os
import string
import re
import shutil

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class Parameters :
      "Wrapper class for global parameters"

      def __init__(self) :
      
          # Get the location of the BIRCH home directory, BIRCHPATH 
          # Preferrably, we get it from the BIRCH.properties file,
          # but if that isn't found, we get it from the BIRCH
          # environment variable. The one catch is that since
          # we don't already know the location of BIRCHPATH,
          # the former will only work if the current working
          # directory is install-birch. Normally, this would
          # only be used when installing or updating BIRCH
          # Otherwise, the environment variable will be read.          
          FN = '../local/admin/BIRCH.properties'
          if  os.path.exists(FN) :
             print "htmldoc.py: Reading $BIRCH from BIRCH.properties"
             self.BIRCHPATH = ""
	     self.BIRCH_PLATFORM = ""
             FILE = open(FN,'r')
             LINE = FILE.readline()
             while (LINE != '') :
                   TOKENS = LINE.split("=") 
                   if TOKENS[0] == 'BirchProps.homedir' :
                      self.BIRCHPATH = TOKENS[1].strip()
                   if TOKENS[0] == 'BirchProps.platform' :
                      self.BIRCH_PLATFORM = TOKENS[1].strip()
                   LINE = FILE.readline() 
             FILE.close()                        
          else :
             print "htmldoc.py: Reading $BIRCH and $BIRCH_PLATFORM environment variables."                     
             self.BIRCHPATH = os.environ['BIRCH'] 

          # If BIRCH_PLATFORM is set in the environment, we want that
          # to supersede the value in BIRCH.properties
	  plat = ""
	  try:
	     plat = os.environ['BIRCH_PLATFORM']
	  except:
	     pass
	  if plat != "":
             self.BIRCH_PLATFORM = plat  
             
          # Everything else depends on BIRCHPATH 
	  print "htmldoc.py: BIRCHPATH set to: " + self.BIRCHPATH 
	  print "htmldoc.py: BIRCH_PLATFORM set to: " + self.BIRCH_PLATFORM 
	  self.BIRCHWEBPATH = self.BIRCHPATH + '/public_html'
	  self.BIRCHDBPATH = self.BIRCHPATH + '/public_html/birchdb'
	  self.BIRCHLOCALDBPATH = self.BIRCHPATH + '/local/public_html/birchdb'
          self.BFN = "" # birchdb.ace
	  self.LFN = "" # birchdb.local.ace
	  self.INFILE = ""
	  self.OFN = ""
	  self.OUTFILE = ""
	  # default prefix for http path to birch documentation
          self.DOCPREFIX = 'file://' + self.BIRCHWEBPATH + '/doc'    

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def DUMPTOACE() :
    "Dump the core BIRCH and local databases to .ace files"
    os.chdir(P.BIRCHDBPATH)
    TACEDIR = P.BIRCHPATH + "/bin-" + P.BIRCH_PLATFORM
        
    # Call tace to dump the core birch data to .ace files:
    #      category.ace, package.ace, program.ace, file.ace
    COMMAND = './tbirchdb.sh ' + P.BIRCHPATH + ' ' + P.BIRCHDBPATH + ' ' + TACEDIR + ' tace.input'
    os.system(COMMAND) 
          
    # Call tace to dump the core birch data to a .ace files
    #      category.ace, package.ace, program.ace, file.ace
    if os.path.exists(P.BIRCHLOCALDBPATH+'/database/ACEDB.wrm') :
       os.chdir(P.BIRCHLOCALDBPATH)
       COMMAND = './tbirchdb.sh '  + P.BIRCHPATH + ' ' + P.BIRCHLOCALDBPATH + ' ' + TACEDIR +  ' tace.input'
       os.system(COMMAND)     

# - - - - - - - - - - - - - - - - - - - - - - - - 
def TOKENIZE(LINE) :
    "Split up input line into tokens, where one or more spaces are seperators"
    "TOKENIZE implicitly gets rid of the first token in the list"

    # Parse the line into tokens
    TOKENS = string.split(LINE)

    # Strip quotes that begin and end data values in .ace files    
    I = 1
    while I < len(TOKENS) :
          TOKENS[I] = TOKENS[I].strip('"')
	  # Get rid of \ escape characters added by ACEDB
	  TOKENS[I] = TOKENS[I].replace('\\','')
          I = I + 1
     
    return TOKENS

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def GetStr(LINE) :
    " Read the first string delimited by double quotes"
    I = LINE.find('"')
    J = LINE.rfind('"')
    LINE = LINE[I+1:J]
    # Get rid of \ escape characters added by ACEDB
    LINE = LINE.replace('\\','')
    return LINE  

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
def AddToList(L,NAME) :
    "If a list contains NAME, do nothing"
    "If it does not exist, append it to the list"
    if NAME in L :
       pass
    else:
       L.append(NAME)
                
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class Category :

      def __init__(self) :
          self.name = ""
	  self.program = []
	  self.pkg = []



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def ReadCategory(FN,CATS) :
    "Read category.ace into a list of category objects"

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    def AssignCategory(NAME) :
	"If a category called NAME already exists, return a pointer"
	"If it does not exist, create a new category object and return a pointer"
	if CATDICT.has_key(NAME) :
	   C = CATDICT[NAME]
	else:
	   C = Category()
	   CATS.append(C)
	   C.name = NAME
	return C

    if os.path.exists(FN) :
       FILE = open(FN,'r')
       LINE = FILE.readline()

       while LINE != '':
             if re.match('Category\s:\s".+"',LINE) : 
		C = AssignCategory(GetStr(LINE))
	     elif re.match('Program\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(C.program,TOKENS[1])
	     elif re.match('Package\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(C.pkg,TOKENS[1])	        
             LINE = FILE.readline()

       FILE.close()
     	   
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class Package :

      def __init__(self) :
          self.name = ""
	  self.description = ""
	  self.category = []
	  self.doc = []	  
	  self.program = []
          self.data = []  
	  self.platform = []
	  self.installation = []     
	  	  
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def ReadPackage(FN,PKGS) :
    "Read package.ace into a list of Package objects"

    # -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
    def AssignPackage(NAME) :
	"If a package called NAME already exists, return a pointer"
	"If it does not exist, create a new package object and return a pointer"
	if PKGDICT.has_key(NAME) :
	   PK = PKGDICT[NAME]
	else:
	   PK = Package()
	   PKGS.append(PK)
	   PK.name = NAME
	return PK

    if os.path.exists(FN) :	
       FILE = open(FN,'r')
       LINE = FILE.readline()

       while LINE != '':
             if re.match('Package\s:\s".+"',LINE) :
		PK = AssignPackage(GetStr(LINE))	  
	     elif re.match('Description\s+".+"',LINE) :
		PK.description = GetStr(LINE)
	     elif re.match('Category\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PK.category,TOKENS[1])	     
	     elif re.match('Documentation\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PK.doc,TOKENS[1])
	     elif re.match('Program\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PK.program,TOKENS[1])
	     elif re.match('Data\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PK.data,TOKENS[1])
	     elif re.match('Platform\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PK.platform,TOKENS[1])	        
             elif re.match('BIRCH',LINE) :	       
		AddToList(PK.installation,"BIRCH")
	     elif re.match('local',LINE) :     
		AddToList(PK.installation,"local")
             LINE = FILE.readline()

       FILE.close()   

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class Program :

      def __init__(self) :
          self.name = ""
	  self.description = ""
	  self.category = []
	  self.interface = []
	  self.package = ""
	  self.doc = []	  
	  self.data = []
          self.sampleinput = []  
          self.sampleoutput = [] 
	  self.platform = []      

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def ReadProgram(FN,PROGS) :
    "Read program.ace into a list of Program objects"

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    def AssignProgram(NAME) :
	"If a program called NAME already exists, return a pointer"
	"If it does not exist, create a new program object and return a pointer"
	if PROGDICT.has_key(NAME) :
	   PR = PROGDICT[NAME]
	else:
	   PR = Program()
	   PROGS.append(PR)
	   PR.name = NAME
	return PR

    if os.path.exists(FN) :    
       FILE = open(FN,'r')
       LINE = FILE.readline()

       while LINE != '':
             if re.match('Program\s:\s".+"',LINE) :
		PR = AssignProgram(GetStr(LINE))
	     elif re.match('Description\s+".+"',LINE) :
        	PR.description = GetStr(LINE)
	     elif re.match('Category\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.category,TOKENS[1])

	     # In ACeDB .ace files, constants the Interface field
	     # does not actually contain the tag 'Interface'
	     # It simply contains the values of the field.	  
	     elif re.match('command\s+".+"',LINE) :
		PR.interface.append(['command',GetStr(LINE)])
	     elif re.match('interactive\s+".+"',LINE) :
		PR.interface.append(['interactive',GetStr(LINE)])
	     elif re.match('gui\s+".+"',LINE) :
		PR.interface.append(['gui',GetStr(LINE)])
	     elif re.match('GDE\s+".+"',LINE) :
		PR.interface.append(['GDE',GetStr(LINE)])
	     elif re.match('dGDE\s+".+"',LINE) :
		PR.interface.append(['dGDE',GetStr(LINE)])
	     elif re.match('mGDE\s+".+"',LINE) :
		PR.interface.append(['mGDE',GetStr(LINE)])
	     elif re.match('tGDE\s+".+"',LINE) :
		PR.interface.append(['tGDE',GetStr(LINE)])
	     elif re.match('Package\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		PR.package = TOKENS[1]
	     elif re.match('Documentation\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.doc,TOKENS[1])
	     elif re.match('Data\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.data,TOKENS[1])
	     elif re.match('Sample_input\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.sampleinput,TOKENS[1])
	     elif re.match('Sample_output\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.sampleoutput,TOKENS[1])
	     elif re.match('Platform\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		AddToList(PR.platform,TOKENS[1])	        

             LINE = FILE.readline()

       FILE.close()   	      	    

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class File :

      def __init__(self) :
          self.name = ""
	  self.description = ""
	  self.command = '\"$ACE_FILE_LAUNCHER\" '
	  self.path = "" 

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def ReadDocFiles(FN,DOCFILES) :
    "Read file.ace into a list of file objects"

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    def AssignFile(NAME) :
	"If a file called NAME already exists, return a pointer"
	"If it does not exist, create a new file object and return a pointer"
	if DOCDICT.has_key(NAME) :
	   FL = DOCDICT[NAME]
	else:
	   FL = File()
	   DOCFILES.append(FL)
	   FL.name = NAME
	return FL

    if os.path.exists(FN) :
       FILE = open(FN,'r')
       LINE = FILE.readline()

       while LINE != '':
             if re.match('File\s:\s".+"',LINE) :
		FL = AssignFile(GetStr(LINE))
	     elif re.match('Description\s+".+"',LINE) :
		FL.description = GetStr(LINE)
	     elif re.match('Pick_me_to_call\s+".+"',LINE) :
		TOKENS = TOKENIZE(LINE)
		FL.path =(TOKENS[2])	        

             LINE = FILE.readline()

       FILE.close()
    
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def GETPREFIX(FN,P) :
    "Read $BIRCH/install-birch/newstr.param, to get the prefixes"
    "needed for URLs" 
    FILE = open(FN,'r')
    LINE = FILE.readline()
    I = 1
    while LINE != '':
          if I == 3 :
             P.DOCPREFIX = LINE.strip("\s")
          LINE = FILE.readline()
	  I = I + 1	  
    FILE.close()

    if (P.DOCPREFIX.find('http://') == 0) or (P.DOCPREFIX.find('file:///') == 0) :
          OKAY = True	  
    else :
       OKAY = False
       
    return OKAY 

# - - - - - - - - - - - - - - - - - - - - - - - - 
def NameToURL(NAME) :
    "Convert a Unix path to a URL"
    "If the path begins with an environment variable like $doc,"
    "assume that the name of the directory is the name of the variable"
    " ie. just delete the '$' and append the path to DOCPREFIX"
    "Otherwise, assume it is a URL of the form 'http:///'"

    if NAME[0] == '$' :
       URL = P.DOCPREFIX + '/' + NAME[1:]
    else :
       URL = NAME
    return URL
 
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def FreshDir(DIRNAME) :
    "Delete an existing directory and create an empty directory"
    "with the same name."
    if os.path.isdir(DIRNAME) :
       shutil.rmtree(DIRNAME)
    os.mkdir(DIRNAME)
    os.chmod(DIRNAME,0755)

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
class HTMLWriter :
      "Methods for writing HTML to a file"
      
      def __init__(self) :
          self.INDENTWIDTH = 3
	  self.COL = 0 # current indentation column
          self.LPAD = ""


      def INDENT(self) :
          ">>> INDENT is not currently used by HTMLWriter"
          "Decrease indent using IDENTWIDTH blank spaces"
	  self.COL = self.COL + self.INDENTWIDTH
	  self.LPAD = ' '.rjust(self.COL)
	    
      def UNDENT(self) :
          ">>> UNDENT is not currently used by HTMLWriter"
          "Decrease indent using IDENTWIDTH blank spaces"
	  self.COL = self.COL - self.INDENTWIDTH
	  if self.COL < 0  :
	     self.COL = 0
	     self.LPAD = ""
	  else :
	     self.LPAD = ' '.rjust(self.COL)

      def START(self,HTMLFILE,TAGNAME,ATTRIBUTES) :
          "Write begin tag, with attributes"
          HTMLFILE.write('<' + TAGNAME + ATTRIBUTES + '>\n')

      def END(self,HTMLFILE,TAGNAME) :
          "Write end tag"
          HTMLFILE.write('</' + TAGNAME + '>\n')

      def PAGETITLE(self,HTMLFILE,TITLE) :
          "Write Title"
          HTMLFILE.write('<TITLE>' + 'BIRCH - ' + TITLE + '</TITLE>\n')		  
	  
      def LINK(self,HTMLFILE,URL,ATTRIBUTES,TEXT) :
          "Write Hypertext Link"
	  HTMLFILE.write('<A href="' + URL + '"' + ATTRIBUTES + '>' + TEXT + '</A>')

      def STARTPAGE(self,HTMLFILE,TITLE) :
          "Information at the top of each page is the same."
	  HTMLFILE.write('<!-- This page generated automatically by htmldoc.py -->\n')
	  
          self.START(HTMLFILE,'HTML','')
          self.PAGETITLE(HTMLFILE,TITLE)
	  ATTRIBUTES = ' style ="background-color: rgb(255, 204, 0);"'
          self.START(HTMLFILE,'BODY',ATTRIBUTES)
          # Main heading, including BIRCH logo which links to BIRCH
          # home page.
          self.START(HTMLFILE,'H1','')
	  URL = '../../index.html'
          TEXT = '<img alt="BIRCH - " src="../smallbirch.gif">'
	  self.LINK(HTMLFILE,URL,'',TEXT)
	  HTMLFILE.write(' ' + TITLE)
          self.END(HTMLFILE,'H1')
	  HTMLFILE.write('<BR>')

      def INDENTTEXT(self,HTMLFILE,TEXT) :
          "Indent a line of text"
	  ATTRIBUTES = ' style="margin-left: 40px;"'
          self.START(HTMLFILE,'div',ATTRIBUTES)
	  HTMLFILE.write(TEXT)
          self.END(HTMLFILE,'div')
	  	  
      def ENDPAGE(self,HTMLFILE) :
          "HTML tags for end of page"
          self.END(HTMLFILE,'BODY')
          self.END(HTMLFILE,'HTML')

	  

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def WriteCategoryPage(CATFN) :
    "Write HTML page listing program categories."

    # -   -   -   -   -   -   -   -   -   -   -
    def WriteCat(C) :
	"Write a page for a given category"
	CATFN = C.name + '.html'
	CATPAGEFILE = open(CATFN,'w')
	H = HTMLWriter ()
	H.STARTPAGE(CATPAGEFILE,'Category: ' + C.name)
	H.START(CATPAGEFILE,'hr','')
	H.START(CATPAGEFILE,'h1','')
	CATPAGEFILE.write('Programs:')
	H.END(CATPAGEFILE,'h1')
	H.START(CATPAGEFILE,'ul','')
	C.program.sort(lambda x,y: cmp(x.lower(), y.lower()))
	for PNAME in C.program :	    
            H.START(CATPAGEFILE,'li','')
	    URL = '../program/' + PNAME + '.html'
	    TEXT = PNAME + ' - ' + PROGDICT[PNAME].description + '\n'	
	    H.LINK(CATPAGEFILE,URL,'',TEXT)
	    H.END(CATPAGEFILE,'li')
	H.END(CATPAGEFILE,'ul')
	H.ENDPAGE(CATPAGEFILE)	    
	CATPAGEFILE.close()
	os.chmod(CATFN,0644)


    CATFILE = open(CATFN,'w')    
    H = HTMLWriter ()    
    H.STARTPAGE(CATFILE,'Programs by Category')
    H.START(CATFILE,'hr','')
    H.START(CATFILE,'br','')      

    # For each category, write an HTML page containing
    # a list of programs in that category
    H.START(CATFILE,'ul','')
    CATLIST = CATDICT.keys()
    CATLIST.sort(lambda x,y: cmp(x.lower(), y.lower()))
    for CATKEY in CATLIST :
        C = CATDICT[CATKEY]
        WriteCat(C)
        H.START(CATFILE,'li','')
	URL = C.name + '.html'
	TEXT = C.name + '\n'	
	H.LINK(CATFILE,URL,'',TEXT)
	H.END(CATFILE,'li')    
    H.END(CATFILE,'ul')
    H.ENDPAGE(CATFILE)
    CATFILE.close()
    os.chmod(CATFN,0644)    
    
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def WritePackagePage(PKGFN) :
    "Write HTML page listing program packages."

    # -   -   -   -   -   -   -   -   -   -   -
    def WritePkg(PK) :
	"Write a page for a given package"
	PKGFN = PK.name + '.html'
	PKGPAGEFILE = open(PKGFN,'w')
	H = HTMLWriter ()
	H.STARTPAGE(PKGPAGEFILE,'Package: ' + PK.name)

	# Program information is in a table.
#	ATTRIBUTES = ' style="border: 1px solid";'
        ATTRIBUTES = ' style="font-family: helvetica,arial,sans-serif;"'
        H.START(PKGPAGEFILE,'table',ATTRIBUTES)
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)

        # Title row
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(51, 255, 51);"'
        H.START(PKGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'big',ATTRIBUTES)	
        PKGPAGEFILE.write(PK.name + ' - ' + PK.description)
	H.END(PKGPAGEFILE,'big')	
	H.END(PKGPAGEFILE,'td')	
	H.END(PKGPAGEFILE,'tr')


        # Documentation rows 
        DocRows(PKGPAGEFILE,H,'Documentation',PK.doc,DOCDICT)

        # Datafile rows 
        DocRows(PKGPAGEFILE,H,'Data',PK.data,DOCDICT)


        # Platform row
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PKGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Platforms'
	TEXT = 	'Platforms'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PKGPAGEFILE,URL,ATTRIBUTES,TEXT)
	PKGPAGEFILE.write(': ')
	
	H.END(PKGPAGEFILE,'big')
	I = 0	
	for PLAT in PK.platform :
	    PKGPAGEFILE.write(PLAT)
	    I = I + 1
	    if I < len(PK.platform) :
	       PKGPAGEFILE.write(', ')	       	            	
	H.END(PKGPAGEFILE,'td')	
	H.END(PKGPAGEFILE,'tr')

        # Installation row
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PKGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Installation'
	TEXT = 	'Installation'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PKGPAGEFILE,URL,ATTRIBUTES,TEXT)
	PKGPAGEFILE.write(': ')
	
	H.END(PKGPAGEFILE,'big')
	I = 0	
	for INST in PK.installation :
	    PKGPAGEFILE.write(INST)
	    I = I + 1
	    if I < len(PK.installation) :
	       PKGPAGEFILE.write(', ')	       	            	
	H.END(PKGPAGEFILE,'td')	
	H.END(PKGPAGEFILE,'tr')

        # Program rows 
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PKGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Programs'
	TEXT = 	'Programs'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PKGPAGEFILE,URL,ATTRIBUTES,TEXT)
	PKGPAGEFILE.write(': ')
	
	H.END(PKGPAGEFILE,'big')
	H.END(PKGPAGEFILE,'td')	
	H.END(PKGPAGEFILE,'tr')
	
	ATTRIBUTES = ''
        H.START(PKGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(255, 255, 255);"'
        H.START(PKGPAGEFILE,'td',ATTRIBUTES)		
	H.START(PKGPAGEFILE,'ul','')
	PK.program.sort(lambda x,y: cmp(x.lower(), y.lower()))
	for PNAME in PK.program :	    
            H.START(PKGPAGEFILE,'li','')
	    URL = '../program/' + PNAME + '.html'
	    TEXT = PNAME + ' - ' + PROGDICT[PNAME].description + '\n'	
	    H.LINK(PKGPAGEFILE,URL,'',TEXT)
	    H.END(PKGPAGEFILE,'li')
	H.END(PKGPAGEFILE,'ul')
	H.END(PKGPAGEFILE,'td')	
	H.END(PKGPAGEFILE,'tr')	
	H.END(PKGPAGEFILE,'table')	

	H.ENDPAGE(PKGPAGEFILE)	    
	PKGPAGEFILE.close()
	os.chmod(PKGFN,0644)


    PKGFILE = open(PKGFN,'w')    
    H = HTMLWriter ()    
    H.STARTPAGE(PKGFILE,'Programs by Package')
    H.START(PKGFILE,'hr','')
    H.START(PKGFILE,'br','') 

    # For each package, write an HTML page containing
    # a list of programs in that package
    H.START(PKGFILE,'ul','')
    PKGLIST = PKGDICT.keys()

    PKGLIST.sort(lambda x,y: cmp(x.lower(), y.lower()))
    for PKGKEY in PKGLIST :
        PK = PKGDICT[PKGKEY]
        WritePkg(PK)
        H.START(PKGFILE,'li','')
	URL = PK.name + '.html'
        TEXT = PK.name + ' - ' + PKGDICT[PK.name].description + '\n'		
	H.LINK(PKGFILE,URL,'',TEXT)
	H.END(PKGFILE,'li')
    
    H.END(PKGFILE,'ul')
    H.ENDPAGE(PKGFILE)
    PKGFILE.close()
    os.chmod(PKGFN,0644)    

    
# -   -   -   -   -   -   -   -   -   -   -
def DocRows(OUTFILE,H,HEADING,FILELIST,D) :
    "Write rows for documentation lines:"
    "Documentation, Data, Sample input, Sample output"
    
    # Documentation rows - title row is 1 column, other rows are 2 columns
    ATTRIBUTES = ''
    H.START(OUTFILE,'tr',ATTRIBUTES)
    ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
    H.START(OUTFILE,'td',ATTRIBUTES)		
    ATTRIBUTES = ''
    H.START(OUTFILE,'big',ATTRIBUTES)
    URL = '../Doc_definitions.html#' + HEADING.replace(' ','_')
    TEXT = HEADING
    ATTRIBUTES = 'target="FrameWindow"'
    H.LINK(OUTFILE,URL,ATTRIBUTES,TEXT)
    H.END(OUTFILE,'big')		            	
    H.END(OUTFILE,'td')	
    H.END(OUTFILE,'tr')

    for DOCNAME in FILELIST :
	ATTRIBUTES = ''
	if D.has_key(DOCNAME) :
	   DOCUMENT = D[DOCNAME]
           if len(DOCUMENT.path) > 0 :
              H.START(OUTFILE,'tr',ATTRIBUTES)
	      ATTRIBUTES = ' border="0" colspan="1" rowspan="1" style= "background-color: rgb(255, 255, 255);"'
              H.START(OUTFILE,'td',ATTRIBUTES)	    
	      DOCTYPE = DOCUMENT.description	
              H.INDENTTEXT(OUTFILE,DOCTYPE)
	      H.END(OUTFILE,'td')
	      ATTRIBUTES = ' border="0" colspan="1" rowspan="1" style= "background-color: rgb(255, 255, 255);"'
              H.START(OUTFILE,'td',ATTRIBUTES)
              URL = NameToURL(DOCUMENT.path)
	      TEXT = DOCUMENT.path	
	      H.LINK(OUTFILE,URL,'',TEXT)		            	
	      H.END(OUTFILE,'td')
	      H.END(OUTFILE,'tr')	

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
def WriteProgramPage(PROGFN) :
    "Write HTML page listing programs in alphabetical order."

    # -   -   -   -   -   -   -   -   -   -   -
    def WriteProg(PR) :
	"Write a page for a given program"
	
	PROGFN = PR.name + '.html'
	PROGPAGEFILE = open(PROGFN,'w')
	H = HTMLWriter ()
	H.STARTPAGE(PROGPAGEFILE,PR.name)
	
	# Program information is in a table.
#	ATTRIBUTES = ' style="border: 1px solid";'
        ATTRIBUTES = ' style="font-family: helvetica,arial,sans-serif;"'
        H.START(PROGPAGEFILE,'table',ATTRIBUTES)
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'tr',ATTRIBUTES)

        # Title row
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(51, 255, 51);"'
        H.START(PROGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'big',ATTRIBUTES)	
        PROGPAGEFILE.write(PR.name + ' - ' + PR.description)
	H.END(PROGPAGEFILE,'big')	
	H.END(PROGPAGEFILE,'td')	
	H.END(PROGPAGEFILE,'tr')

        # Command rows - title row is 1 column, other rows are 2 columns
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PROGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Launching'
	TEXT = 	'Launching the program'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PROGPAGEFILE,URL,ATTRIBUTES,TEXT)
	
	H.END(PROGPAGEFILE,'big')		            	
	H.END(PROGPAGEFILE,'td')	
	H.END(PROGPAGEFILE,'tr')
	
	for COMMAND in PR.interface :
	    ATTRIBUTES = ''
            H.START(PROGPAGEFILE,'tr',ATTRIBUTES)
	    ATTRIBUTES = ' border="0" colspan="1" rowspan="1" style= "background-color: rgb(255, 255, 255);"'
            H.START(PROGPAGEFILE,'td',ATTRIBUTES)	
            H.INDENTTEXT(PROGPAGEFILE,COMMAND[0])
	    H.END(PROGPAGEFILE,'td')
	    ATTRIBUTES = ' border="0" colspan="1" rowspan="1" style= "background-color: rgb(255, 255, 255);"'
            H.START(PROGPAGEFILE,'td',ATTRIBUTES)	    
	    ATTRIBUTES = ' face="Courier New,Courier"'
            H.START(PROGPAGEFILE,'font',ATTRIBUTES)	    		
	    PROGPAGEFILE.write(COMMAND[1])		            	
            H.END(PROGPAGEFILE,'font')	    
	    H.END(PROGPAGEFILE,'td')
	    H.END(PROGPAGEFILE,'tr')


        # Documentation rows 
        DocRows(PROGPAGEFILE,H,'Documentation',PR.doc,DOCDICT)

        # Datafile rows 
        DocRows(PROGPAGEFILE,H,'Data',PR.data,DOCDICT)

        # Sample input rows 
        DocRows(PROGPAGEFILE,H,'Sample input',PR.sampleinput,DOCDICT)

        # Sample output rows 
        DocRows(PROGPAGEFILE,H,'Sample output',PR.sampleoutput,DOCDICT)


        # Package row
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PROGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Package'
	TEXT = 	'Package'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PROGPAGEFILE,URL,ATTRIBUTES,TEXT)
	PROGPAGEFILE.write(': ')
	H.END(PROGPAGEFILE,'big')
	URL = '../package/' + PR.package + '.html'
	TEXT = PR.package	
	H.LINK(PROGPAGEFILE,URL,'',TEXT)
	H.END(PROGPAGEFILE,'td')	
	H.END(PROGPAGEFILE,'tr')

        # Platform row
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'tr',ATTRIBUTES)
	ATTRIBUTES = ' colspan="2" rowspan="1" style= "background-color: rgb(204, 204, 204);"'
        H.START(PROGPAGEFILE,'td',ATTRIBUTES)		
	ATTRIBUTES = ''
        H.START(PROGPAGEFILE,'big',ATTRIBUTES)
	URL = '../Doc_definitions.html#Platforms'
	TEXT = 	'Platforms'
	ATTRIBUTES = 'target="FrameWindow"'
	H.LINK(PROGPAGEFILE,URL,ATTRIBUTES,TEXT)	
	PROGPAGEFILE.write(': ')
	H.END(PROGPAGEFILE,'big')
	I = 0	
	for PLAT in PR.platform :
	    PROGPAGEFILE.write(PLAT)
	    I = I + 1
	    if I < len(PR.platform) :
	       PROGPAGEFILE.write(', ')	       	            	
	H.END(PROGPAGEFILE,'td')	
	H.END(PROGPAGEFILE,'tr')
	
	H.END(PROGPAGEFILE,'table')	
	H.ENDPAGE(PROGPAGEFILE)	    
	PROGPAGEFILE.close()
	os.chmod(PROGFN,0644)


    PROGFILE = open(PROGFN,'w')    
    H = HTMLWriter ()    
    H.STARTPAGE(PROGFILE,'Program Index')
    H.START(PROGFILE,'hr','')
    H.START(PROGFILE,'ul','')

    # For each program, write an HTML page containing
    # all information for that program
    PROGLIST = PROGDICT.keys()
    PROGLIST.sort(lambda x,y: cmp(x.lower(), y.lower()))
    for PROGKEY in PROGLIST :
        PR = PROGDICT[PROGKEY]
        WriteProg(PR)
        H.START(PROGFILE,'li','')
	URL = PR.name + '.html'
        TEXT = PR.name + ' - ' + PROGDICT[PR.name].description + '\n'		
	H.LINK(PROGFILE,URL,'',TEXT)
	H.END(PROGFILE,'li')
    
    H.END(PROGFILE,'ul')
    H.ENDPAGE(PROGFILE)
    PROGFILE.close()
    os.chmod(PROGFN,0644)    

            	  	      	      
#======================== MAIN PROCEDURE ==========================

P = Parameters()

# Use tace to generate birchdb.ace and birchdb.local.ace, which contain
# a complete dump of both the core birch documentation objects and those
# from the local implementation.

DUMPTOACE()

# Read in birchdb.ace, and then birchdb.local.ace. If an object is present
# in both files, the object from birchdb.local.ace replaces the object
# from birchdb.ace. Thus, local versions of documentation can replace
# documents from the birch core.

# Read Categories
CATS = []
CATDICT = {}
os.chdir(P.BIRCHDBPATH)
ReadCategory('category.ace',CATS)
for C in CATS :
    CATDICT[C.name] = C
if os.path.exists(P.BIRCHLOCALDBPATH) :    
   os.chdir(P.BIRCHLOCALDBPATH)
   ReadCategory('category.ace',CATS)
   for C in CATS :
       CATDICT[C.name] = C

# Read Packages
PKGS = []
PKGDICT = {}
os.chdir(P.BIRCHDBPATH)
ReadPackage('package.ace',PKGS)
for PK in PKGS :
     PKGDICT[PK.name] = PK
if os.path.exists(P.BIRCHLOCALDBPATH) : 
   os.chdir(P.BIRCHLOCALDBPATH)
   ReadPackage('package.ace',PKGS)
   for PK in PKGS :
	PKGDICT[PK.name] = PK

# Read programs
PROGS = []
PROGDICT = {}
os.chdir(P.BIRCHDBPATH)
ReadProgram('program.ace',PROGS)
for PR in PROGS :
     PROGDICT[PR.name] = PR
if os.path.exists(P.BIRCHLOCALDBPATH) : 
   os.chdir(P.BIRCHLOCALDBPATH)
   ReadProgram('program.ace',PROGS)
   for PR in PROGS :
	PROGDICT[PR.name] = PR     

# Read documentation file names
DOCFILES = []
DOCDICT = {}
os.chdir(P.BIRCHDBPATH)
ReadDocFiles('file.ace',DOCFILES)
for FL in DOCFILES :
    DOCDICT[FL.name] = FL
if os.path.exists(P.BIRCHLOCALDBPATH) : 
   os.chdir(P.BIRCHLOCALDBPATH)
   ReadDocFiles('file.ace',DOCFILES)
   for FL in DOCFILES :
       DOCDICT[FL.name] = FL


# Read in the prefix to begin URLs that will be used as links
# to documentation files.
os.chdir(P.BIRCHPATH + '/install-birch')
OKAY = GETPREFIX('newstr.param',P)

# Remove existing directories for category, program and package,
# and create new empty directories
# This ensures that old files that are no longer needed (such
# as files describing programs or packages that have been removed)
# do not clutter up the directories.
os.chdir(P.BIRCHPATH + '/public_html/birchdoc')
FreshDir('category')
FreshDir('package')
FreshDir('program')

if OKAY : 

   # - - - - - - - - - -  Output to the public_html/birchdb directory - - - - - - 

   # For each category write a category page in HTML

   os.chdir('category')
   CATFN = 'category.html'
   WriteCategoryPage(CATFN)   
   
   # For each all programs write an HTML page with an alphabetical list of programs
   os.chdir('../program')
   PROGFN = 'program.html'
   WriteProgramPage(PROGFN)   

   # For each package, write a package page in HTML
   # For each category write a category page in HTML
   os.chdir('../package')
   PKGFN = 'package.html'
   WritePackagePage(PKGFN)   



