/*
 * SeqDoc.java
 *
 * Created on September 30, 2008, 11:08 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.core.data.seqdoc;

import java.util.Arrays;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;
import org.biolegato.core.data.seqdoc.undo.UndoDelete;
import org.biolegato.core.data.seqdoc.undo.UndoInsert;
import org.biolegato.core.data.seqdoc.undo.Undoable;
import java.util.LinkedList;
import java.util.Stack;

/**
 * The internal document format for BioLegato.
 * <p>
 * This document is structured as a linked list of sequences.  Each character has an offset based on its
 * position within the list and it's position within its containing sequence.  Sequences start at 0
 * (first character in the first sequence in the list, and end with the last character in the last sequence
 * within the list.
 * </p>
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class SeqDoc implements SequenceListener {

    /**
     * The stack of undoable events.
     * <p>
     *  Every time an event occurs within the SeqDoc that is undoable, an <b>undo object</b>
     *  is created and added to this stack.  This allows one to undo performed SeqDoc actions.
     *  To undo an action, pop the top object off of the stack and call its <b>undo method</b>.
     *  This will perform the undo and the <b>undo method</b> will return a <b>redo object</b>.
     *  The resulting <b>redo object</b> can then be added to the <b>redoStack</b> stack.
     * </p>
     *
     * @see org.biolegato.core.data.undo.Undoable
     */
    protected Stack<Undoable> undoStack = new Stack<Undoable>();
    /**
     * The stack of redoable events
     * <p>
     *  Every time an event is undone, a redo object is created and added to this stack.
     *  A <b>redo object</b> is essentially an <b>undo object</b> created to "undo an undo".
     *  To perform a redo, pop the top object off the stact and call its <b>undo method</b>.
     *  This will perform the redo and the <b>undo method</b> will return an <b>undo object</b>.
     *  The resulting <b>undo object</b> can then be added to the <b>undoStack</b> stack.
     * </p>
     *
     * @see org.biolegato.core.data.undo.Undoable
     */
    protected Stack<Undoable> redoStack = new Stack<Undoable>();
    /**
     * This linked list used to store all of the lines in the document.
     * <p>
     *  Each line is stored as a linked list of sequence wrappers.
     *  Each sequence wrapper is characterized by a sequence and a start offset.
     *  The start offset is used to determine the offset of the sequence within
     *  the document.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final LinkedList<SeqWrap> lines = new LinkedList<SeqWrap>();
    /**
     * This list is used to store all of the document's sequence listeners.
     * <p>
     *  Each time any modification is done to a sequence within the document, or
     *  any sequences are added/removed from the document, all of the document's
     *  listeners are notified of the event.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final LinkedList<SeqDocListener> listeners = new LinkedList<SeqDocListener>();

    /**
     * Creates a new instance of SeqDoc
     */
    public SeqDoc () {
    }

////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////

    /**
     * Inserts a string into the document on a given line.
     *
     * @param lineNumber the line number to insert the string.
     * @param offset the offset in the document to insert the text.
     * @param string the text to insert.
     * @return true if the operation was successful, otherwise false.
     */
    public boolean insert (int lineNumber, int offset, String string) {
        boolean result = false;                                     // the result of the operation (true if successful)
        String current;                                             // the content of the current line

        if (!lines.isEmpty() && lineNumber >= 0 && lineNumber < getLineCount()) {
            current = lines.get(lineNumber).getField("sequence").toString();

            // figure out how to insert the new data
            // 1. if the offset is in the middle of the sequence
            // 2. if the offset is at the end of the sequence
            // 3. if the offset is at the beginning of the sequence
            if (offset > 0 && offset < current.length()) {
                lines.get(lineNumber).setField("sequence", current.substring(0, offset) + string + current.substring(offset));
            } else if (offset >= current.length()) {
                lines.get(lineNumber).setField("sequence", current + string);
            } else {
                lines.get(lineNumber).setField("sequence", string + current);
            }

            // update all of the line start offsets past the current line
            while (lineNumber > 0 && lineNumber < lines.size()) {
                lines.get(lineNumber).setStart(getLineEndOffset(lineNumber - 1) + 1);
                lineNumber ++;
            }

            // TODO: fix undo stack
            undoStack.push(new UndoInsert(this, offset, string.length()));
            
            // set the result to true
            result = true;
        } else {
            // report any errors
            org.biolegato.core.main.BLMain.error("Bad offset: " + offset + " (line: " + lineNumber + ")", "SeqDoc.insert(int, String)");
        }
        return result;
    }
    
    /**
     * Inserts sequences at a specific offset in the document.
     * <p>
     *  The first sequence is always merged with the sequence located at offset, unless the offset
     *  points to the end of the sequence.  The same applies to the last sequence. <br /><br />
     *  Example:<br />
     *  Sequence[0]: aaaa<br />
     *  ....<br />
     *  Sequence[last]: cccc<br />
     *  Document: ggggtttt<br />
     *  offset: 4<br />
     *  Result: ggggaaaa<br />
     *          ....<br />
     *          cccctttt
     * </p>
     *
     * @param offset the offset in the document to insert the sequences.
     * @param sequences the array of sequences to insert.
     * @return true if the operation was successful, otherwise false.
     */
    public boolean insert (int offset, Sequence[] sequences) {
        int lineNumber = getLineNumber(offset);                 // the line number to insert the sequences
        int cutpos = offset - getLineStartOffset(lineNumber);   // the position within the line to insert the sequences (i.e. where is offset within the line)
        boolean result = false;                                 // the result of the operation (true if successful)
        Sequence split = null;                                  // if the offset lands in the middle of the line, then this variable will store all of the sequence data after offset
        String splitStr = "";                                   // the current line's sequence field

        // make sure that the variables are all valid
        if (sequences != null && sequences.length > 0 && lineNumber >= 0 && lineNumber <= getLineCount()) {
            
            // get information about where to split the line
            split = (Sequence) lines.get(lineNumber).getSequence().clone();
            splitStr = split.getField("sequence").toString();

            // test if the offset lands in the middle of the line
            if (!lines.isEmpty() && cutpos > 0 && cutpos < splitStr.length()) {
                
                // test if there are multiple sequences (if so then add both sequences to the current line)
                if (sequences.length - 1 > 0) {
                    
                    // update the current line
                    lines.get(lineNumber).setField("sequence", splitStr.substring(0, cutpos) + sequences[0].getField("sequence"));
                    split.setField("sequence", sequences[sequences.length - 1].getField("sequence").toString() + splitStr.substring(cutpos));
                    
                    // add the sequence to the document
                    addSequence(lineNumber + 1, split);
                } else {
                    insert(lineNumber, cutpos, sequences[0].getField("sequence").toString());
                }
            } else {
                // if the insert is at the end of the line marked by the offset then make
                // the insert on a seperate line after the line marked by the offse
                if (cutpos >= splitStr.length()) {
                    lineNumber ++;
                }
                
                // add the sequence to the document
                addSequence(lineNumber, sequences[0]);
                if (sequences.length - 1 > 0) {
                    addSequence(lineNumber + 1, sequences[sequences.length - 1]);
                }
            }
            lineNumber ++;

            // add all of the sequences in the middle
            for (int count = 1; count < sequences.length - 1; count ++) {
                addSequence(lineNumber, sequences[count]);
                lineNumber ++;
            }

            // TODO: fix undo stack
            //undoStack.push(new UndoInsert(this, offset, length));
            result = true;
            
        } else if (sequences == null || sequences.length <= 0) {
            // report if the sequences to be inserted are invalid
            org.biolegato.core.main.BLMain.error("Invalid sequence parameter", "SeqDoc.insert(int, Sequence[])");
            
        } else if (getLineCount() <= 0) {
            // if there are no lines in the document then add the line as the first line in the document
            addSequences(sequences);
            
        } else if (lineNumber < 0 && lineNumber > getLineCount()) {
            // report if the offset is invalid
            org.biolegato.core.main.BLMain.error("Invalid offset", "SeqDoc.insert(int, Sequence[])");
        }
        return result;
    }
    
    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.  The offset and length parameters are both character based (not line based).
     * </p>
     *
     * @param offset the offset to start the deletion from.
     * @param length the number of characters to delete.
     * @return true if the operation was successful, otherwise false.
     */
    public boolean delete (int offset, int length) {
        int lineNumber = Math.max(0, getLineNumber(offset));    // the line number of the offset
        int cutpos = 0;                                         // the position within the line of the offset
        String data = "";                                       // the data for the sequence after the removal
        String deleted;                                         // the text deleted

        // make sure all of the parameters are valid
        if (!lines.isEmpty() && lines.size() > 0 && offset >= 0 && offset < getLineEndOffset(getLineCount() - 1)) {
            
            // prevent the length from exceeding the document length
            length = Math.max(0, Math.min(offset + length, getLength()) - offset);

            // obtain the text to be deleted
            deleted = getText(offset, length);

            // make all of the lines to be deleted into one line
            while (getLineEndOffset(lineNumber) < offset + length && lineNumber + 1 < lines.size()) {
                lines.get(lineNumber).setField("sequence", lines.get(lineNumber).getField("sequence").toString() + "\n"
                        + lines.get(lineNumber + 1).getField("sequence").toString());
                lines.remove(lineNumber + 1);
            }
            
            // obtain the sequence and position to substring
            data = lines.get(lineNumber).getField("sequence").toString();
            cutpos = Math.max(0, offset - getLineStartOffset(lineNumber));

            // substring the sequence (delete the data)
            lines.get(lineNumber).setField("sequence", data.substring(0, cutpos) + data.substring(Math.min(data.length(), cutpos + length)));

            // update all of the line start offsets
            while (lineNumber > 0 && lineNumber < lines.size()) {
                lines.get(lineNumber).setStart(getLineEndOffset(lineNumber - 1) + 1);
                lineNumber ++;
            }
       }
        return true;
    }

    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.
     * </p>
     *
     * @param lineNumber the line number to delete characters from.
     * @param offset the offset to start the deletion from.
     * @param length the number of characters to delete.
     * @return true if the operation was successful, otherwise false.
     */
    public boolean delete (int lineNumber, int offset, int length) {
        int cutpos = 0;                                         // the position within the line of the offset
        String data = "";                                       // the data for the sequence after the removal
        String deleted;                                         // the text deleted

        // make sure all of the parameters are valid
        if (!lines.isEmpty() && lines.size() > 0 && offset >= 0 && length >= 0&& offset < getLineLength(lineNumber)) {
            
            // obtain the text to be deleted
            deleted = getSequence(lineNumber).getField("sequence").toString().substring(offset, length);

            // substring the sequence (delete the data)
            lines.get(lineNumber).setField("sequence", data.substring(0, offset) + data.substring(offset + length));

            // update all of the line start offsets
            while (lineNumber > 0 && lineNumber < lines.size()) {
                lines.get(lineNumber).setStart(getLineEndOffset(lineNumber - 1) + 1);
                lineNumber ++;
            }

            // send event to listeners
            undoStack.push(new UndoDelete(this, lineNumber, offset, deleted));
        }
        return true;
    }

    /**
     * Adds an array of sequences to a data container at position "lineNumber".
     *
     * @param lineNumber the line number to insert the sequence.
     * @param sequenceList the array of sequences to add to the data container.
     */
    public void addSequences (int lineNumber, Sequence[] sequenceList) {
        // ensure that the line number is valid
        lineNumber = Math.max(0, Math.min(getLineCount(), lineNumber));

        int offset = (lineNumber > 0 ? getLineEndOffset(lineNumber - 1) + 1 : 0); // the offset of the inserted sequence
        int length = 0;               // the length of the inserted sequence
        SeqWrap addLine;                                                                // the sequence wrapper object added

        // if the sequence list is not null add it        
        if (sequenceList != null) {
            for (Sequence seq : sequenceList) {
                // if the current sequence is not null add it
                if (seq != null) {
                    // create the new sequence wrapper object
                    addLine = new SeqWrap(offset, seq);
                    addLine.addListener(this);
                    lines.add(lineNumber, addLine);

                    // update the length counter
                    length += seq.getField("sequence").toString().length();

                    // move on to the next line number (for updating all lines past the current)
                    lineNumber ++;
                }
                
                // send sequence added event
                for (SeqDocListener listener : listeners) {
                    listener.sequenceAdded(this, lineNumber, seq);
                }
            }
            
            // update all of the line start offsets past the current line
            while (lineNumber > 0 && lineNumber < lines.size()) {
                lines.get(lineNumber).setStart(getLineEndOffset(lineNumber - 1) + 1);
                lineNumber ++;
            }
            
            // send event to listeners
            undoStack.push(new UndoInsert(this, offset, length));
        }
    }
    
    /**
     * Removes an array of sequences from the data container.
     *
     * @param lineNumbers the line numbers to remove.
     */
    public void removeSequences (int[] lineNumbers) {
        int offset;         // the offset of the sequence removed.
        int length;         // the length of the sequence removed.
        int lineNumber;     // the current line number to remove.
        String text;        // the data of the sequence removed.
        SeqWrap removed;    // the sequence wrapper object removed.

        // sort the line numbers
        Arrays.sort(lineNumbers);
        
        // ensure that there are lines to delete
        if (!lines.isEmpty()) {
            // itterate backwards through each sequence line number and delete it (using the removeSequence method)
            for (int count = lineNumbers.length - 1; count >= 0; count--) {
                // set the current line number to delete
                lineNumber = lineNumbers[count];
                
                // check the bounds of the line number to delete
                if (lines.size() > lineNumber && lineNumber >= 0) {
                    // remove the sequence
                    removed = lines.remove(lineNumber);

                    // get information about the sequence removed
                    text = removed.getField("sequence").toString();
                    offset = removed.getStart();
                    length = text.length();

                    // send sequence removed event
                    for (SeqDocListener listener : listeners) {
                        listener.sequenceRemoved(this, lineNumber, removed.getSequence());
                    }
                }
                
                // update the start offsets of every line after the removed line
                while (lineNumber > 0 && lineNumber < lines.size()) {
                    lines.get(lineNumber).setStart(getLineEndOffset(lineNumber - 1) + 1);
                    lineNumber ++;
                }
            }

        }
    }

//////////////////////////////////
//******************************//
//* INDIRECT DATA MODIFICATION *//
//******************************//
//////////////////////////////////
    
    /**
     * Adds a sequence to the end of the data container.
     *
     * @param seq the sequence to add to the data container.
     */
    public void addSequence (Sequence seq) {
        addSequence(getLineCount(), seq);
    }
    
    /**
     * Adds a sequence to the document.
     *
     * @param lineNumber the line number to insert the sequence.
     * @param seq the sequence to add to the document.
     */
    public void addSequence (int lineNumber, Sequence seq) {
        addSequences(lineNumber, new Sequence[]{seq});
    }
    
    /**
     * Adds an array of sequences to the end of the data container.
     *
     * @param sequenceList the array of sequences to add to the data container.
     */
    public void addSequences (Sequence[] sequenceList) {
        addSequences(getLineCount(), sequenceList);
    }

    /**
     * Removes a sequences from the data container.
     *
     * @param lineNumber the line number of the sequence to remove.
     */
    public void removeSequence (int lineNumber) {
        removeSequences(new int[]{lineNumber});
    }

    /**
     * Performs an undo operation to the data container.
     *
     * @return true if the undo was successful.
     */
    public boolean undo () {
        boolean result =  ! undoStack.empty();  // the result of the undo.
        Undoable undone = null;                 // the redo object (to redo the operation).

        if (result) {
            undone = undoStack.pop().undo();
            if (undone != null) {
                redoStack.push(undone);
            }
        }
        return result;
    }

    /**
     * Performs an redo operation to the data container.
     *
     * @return true if the redo was successful.
     */
    public boolean redo () {
        boolean result =  ! redoStack.empty();  // the result of the undo.
        Undoable redone = null;                 // the undo object (to undo the redone operation).

        if (result) {
            redone = redoStack.pop().undo();
            if (redone != null) {
                undoStack.push(redone);
            }
        }
        return result;
    }

//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////

    /**
     * Returns the length of the data container's content.
     *
     * @return the length in characters of the data container's sequence content.
     */
    public int getLength () {
        // if there are no sequences the length is 0 (the bottom line is a safeguard for an empty sequence list)
        return (lines.size() > 0 ? lines.getLast().getStart() + lines.getLast().getField("sequence").toString().length() - lines.getFirst().getStart() : 0);
    }

    /**
     * Obtains the line number for a given offset.
     *
     * @param offset the offset to get the line number for (in characters).
     * @return the corresponding line number.
     */
    public int getLineNumber (int offset) {
        int high = lines.size() - 1;	// the current upper bound of the search.
        int low = 0;                    // the current lower bound of the search.
        int index = -1;                 // the current index to test (on success this will contain final result of the search).
        SeqWrap current = null;         // the Element at the current index.

        if (lines.size() > 0 && offset >= 0 && offset <= getLength()) {
            // perform a do while loop through each index, slicing in half.
            // the loop continues until the index is outside the boundaries
            // of of the list of lines or until the search offset is bound by
            // the Element at the current index's start and end offsets.
            do {
                // calculate the next index taking the midpoint of the previous
                // upper and lower boundaries.
                if (index == (low + high) / 2 && index != high) {
                    index ++;
                } else if (index == (low + high) / 2) {
                    index --;
                } else {
                    index = (low + high) / 2;
                }
                current = lines.get(index);

                // text the current Element
                if (current != null) {
                    if (current.getStart() + current.getField("sequence").toString().length() < offset) {
                        // if the Element at the current index's end offset is
                        // less than the offset we are searching for, make it
                        // the lower boundary for the next search (this value is
                        // used in calculating the next index).
                        low = index;
                    } else if (current.getStart() > offset) {
                        // if the Element at the current index's start offset is
                        // greater than the offset we are searching for, make it
                        // the upper boundary for the next search (this value is
                        // used in calculating the next index).
                        high = index;
                    }
                }
            } while (current != null && low != high && (current.getStart() + current.getField("sequence").toString().length() < offset || current.getStart() > offset));
        }
        return index;
    }

    /**
     * Returns the number of lines in the document.
     *
     * @return the number of lines in the document.
     */
    public int getLineCount () {
        return lines.size();
    }

    /**
     * Retrieves a given line of text from the document.
     *
     * @param lineNumber the line number to retrieve.
     * @return the line of text.
     */
    public String getLine (int lineNumber) {
        return (lineNumber >= 0 && lineNumber < lines.size() ? lines.get(lineNumber).getField("sequence").toString() : "");
    }

    /**
     * Retrieves the start offset of a line.
     *
     * @param lineNumber the line number to obtain the start offset for.
     * @return the start offset for the line (in characters).
     */
    public int getLineStartOffset (int lineNumber) {
        return (lineNumber >= 0 && lineNumber < lines.size() ? lines.get(lineNumber).getStart() : 0);
    }

    /**
     * Retrieves the end offset of the given line.
     *
     * @param lineNumber the line number to obtain the end offset for.
     * @return the end offset for the line (in characters).
     */
    public int getLineEndOffset (int lineNumber) {
        return (lineNumber >= 0 && lineNumber < lines.size() ? (getLineStartOffset(lineNumber) + lines.get(lineNumber).getField("sequence").toString().length()) : 0);
    }

    /**
     * Retrieves the length of a line in the document.
     *
     * @param lineNumber the line number to find the length of.
     * @return the length of the line (in characters).
     */
    public int getLineLength (int lineNumber) {
        return (lineNumber >= 0 && lineNumber < lines.size() ? getLineEndOffset(lineNumber) - getLineStartOffset(lineNumber) : 0);
    }

    /**
     * Retrieves a body of text from the data container.
     *
     * @param offset the offset to start retrieving from (in characters).
     * @param length the length of text to retrieve (in characters).
     * @return the corresponding body of text.
     */
    public String getText (int offset, int length) {
        int minOffset = Math.max(0, Math.min(offset, offset + length));                 // the start offset (in characters) of the retreival.
        int maxOffset = Math.min(getLength() - 1, Math.max(offset, offset + length));   // the end offset (in characters) of the retreival.
        int minLine = getLineNumber(minOffset);                                         // the line number of the start offset.
        int maxLine = getLineNumber(maxOffset);                                         // the line number of the end offset.
        int cutpos = Math.max(0, offset - getLineStartOffset(minLine));                 // the offset of the start position with the minLine sequence
        String result = getLine(minLine);                                               // the resulting text
        StringBuffer lineBuffer = new StringBuffer();                                   // string buffer used if the text spans multiple lines

        // make sure there is at least on line within the data contianer.
        if (lines.size() > 0) {
            // NOTE: this is faster than + or concat operators
            lineBuffer.append(result);
            
            // obtain all lines after the first
            // NOTE: append is faster than + or concat operators
            for (int count = minLine + 1; count <= maxLine; count ++) {
                lineBuffer.append(getLine(count));
            }
            result = lineBuffer.toString();

            // handle if the line is empty
            if (cutpos >= result.length() || length <= 0) {
                result = "";
            } else {
                // cut the initial part of the line
                if (cutpos > 0) {
                    result = result.substring(cutpos);
                }
                
                // handle the length parameter
                if (length < result.length()) {
                    result = result.substring(0, length);
                }
            }
        }
        return result;
    }

    /**
     * Retrieves a sequence object from the SeqDoc specified by its line number.
     *
     * @param lineNumber the line number to retreive the sequence.
     * @return the sequence.
     */
    public Sequence getSequence (int lineNumber) {
        return (lineNumber >= 0 && lineNumber < lines.size() ? lines.get(lineNumber).getSequence() : null);
    }

    /**
     * Calls getSequence multiple times for multiple line numbers.
     *
     * @param lineNumbers the line numbers to send to get sequence.
     * @return the resulting sequences (in an array).
     */
    public Sequence[] getSequences (int[] lineNumbers) {
        Sequence seqAdd = null;                                     // the result of the current call to getSequence
        LinkedList<Sequence> seqList = new LinkedList<Sequence>();  // the list of all resulting calls to getSequence

        // itterate through the line numbers
        for (int line : lineNumbers) {
            seqAdd = getSequence(line);
            
            // ensure that we do not add any null results
            if (seqAdd != null) {
                seqList.add(seqAdd);
            }
        }
        return seqList.toArray(new Sequence[0]);
    }

/////////////////
//*************//
//* LISTENERS *//
//*************//
/////////////////
    /**
     * Adds a listener object to the data container.
     *
     * @param listener the listener to add.
     */
    public void addListener (SeqDocListener listener) {
        listeners.add(listener);
    }

    /**
     * Returns the length of the longest line in the data container
     *
     * @return the length of the longest line
     */
    public int getLongestLineLength() {
        int lineLength = 0;
        
        for (int count = 0; count < getLineCount(); count++) {
            lineLength = Math.max(lineLength, getLineLength(count));
        }
        return lineLength;
    }

    /**
     * Called when a field in a sequence is modified.
     *
     * @param sequence the sequence modified.
     * @param key the key of the modified field in the sequence.
     */
    public void sequenceChanged(Sequence sequence, String key) {
        for (SeqDocListener listener : listeners) {
            listener.sequenceChanged(this, lines.indexOf(sequence), sequence, key);
        }
    }
}
