/*
 * SeqWrap.java
 *
 * Created on June 9, 2009, 3:01 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.biolegato.core.data.seqdoc;

import java.util.LinkedList;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;

/**
 * Used to wrap sequences; this object is used to give sequences start offsets,
 * so they may be ordered within a SeqDoc structure.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class SeqWrap implements SequenceListener {
    /**
     * The start offset of the sequence (all character positions are calculated based off of this number)
     */
    private int start = 0;
    /**
     * The previous length of the sequence (used for updating the start offset for each line after the current).
     */
    private int length = 0;
    /**
     * The wrapped sequence object.
     */
    private Sequence seq;

    /**
     * Creates a new instance of SeqWrap.
     *
     * @param offset the start offset of the sequence wrapper.
     * @param seq the sequence to wrap.
     */
    public SeqWrap (int offset, Sequence seq) {
        // ensure that the sequence is not null
        if (seq == null) {
            throw new NullPointerException("Cannot call SeqWrap(" + start + "," + seq + ")");
        }
        
        // set the internal fields of the sequence wrapper
        this.start = offset;
        this.seq = seq;
        this.length = seq.getField("sequence").toString().length();
        
        // add the sequence wrapper as a listener to the sequence
        seq.addListener(this);
    }

    /**
     * Retrieves the start offset for the sequence wrapper object.
     *
     * @return the start offset of the sequence wrapper object.
     */
    public int getStart () {
        return start;
    }

    /**
     * Returns the content of a field within the wrapped sequence.
     *
     * @param key the key of the field to retrieve.
     * @return the current value of the field.
     */
    public Object getField (String key) {
        return seq.getField(key);
    }

    /**
     * Alters the content of a field within the wrapped sequence.
     *
     * @param key the key of the field to alter.
     * @param value the new value of the field.
     */
    public void setField (String key, Object value) {
        seq.setField(key, value);
    }

    /**
     * Extracts a portion of the wrapper's sequence.
     *
     * @param start the position to start the extraction.
     * @return the extracted sequence.
     */
    public Sequence subseq (int start) {
        return seq.subseq(start);
    }

    /**
     * Extracts a portion of the wrapper's sequence.
     *
     * @param start the position to start the extraction.
     * @param end the position to end the extraction.
     * @return the extracted sequence.
     */
    public Sequence subseq (int start, int end) {
        return seq.subseq(start, end);
    }

    /**
     * Changes the start offset of the sequence.
     *
     * @param offset the new start offset of the sequence.
     */
    public void setStart (int offset) {
        this.start = offset;
    }

    /**
     * Retrieves the wrapped sequence object.
     *
     * @return the sequence object.
     */
    public Sequence getSequence () {
        return seq;
    }

    /**
     * Adds a listener to the wrapped sequence.
     *
     * @param listener the listener to add.
     */
    public void addListener (SequenceListener listener) {
        seq.addListener(listener);
    }
    
    /**
     * Handles changes to the inner sequence.
     *
     * @param sequence the sequence modified.
     * @param key the key modified.
     */
    public void sequenceChanged(Sequence sequence, String key) {
    }
    
    /**
     * Removes the sequence wrapper as a sequence listener.
     */
    protected void finalize() {
        seq.removeListener(this);
    }

    /**
     * Tests equality of the inner sequence.
     * NOTE: The two objects are equal if<br />
     * 1. "test" is a sequence wrapper then if the wrapped sequences are equal
     * 2. "test" is equal to the wrapped sequence (sequence.equals method)
     *
     * @param test the object to test for equality.
     */
    public boolean equals(Object test) {
        return ((test instanceof SeqWrap) ? seq.equals(((SeqWrap)test).seq) : seq.equals(test));
    }
}
