package org.biolegato.gdesupport.canvas;

/*
 * BLTextArea.java
 *
 * Created on August 18, 2008, 11:10 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import javax.swing.JComponent;
import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.core.data.seqdoc.SeqDocListener;
import org.biolegato.gdesupport.canvas.colour.ColourMap;
import org.biolegato.gdesupport.canvas.colour.MonochromeColourMap;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import org.biolegato.gdesupport.canvas.selections.SelectionListener;
import org.biolegato.gdesupport.canvas.selections.LineSelection;
import org.biolegato.gdesupport.canvas.selections.Selection;
import org.biolegato.gdesupport.canvas.selections.TextLines;
import org.biolegato.gdesupport.canvas.selections.TextRectangle;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.main.ProgramWindow;

/**
 *
 * @author alvare
 */
public class BLTextArea extends JComponent implements FocusListener, SelectionListener, Editable, UndoRedoable,SeqDocListener, KeyListener, MouseMotionListener, MouseListener {

    /**
     * The biolegato instance to use to obtain data from
     */
    protected ProgramWindow instance = null;
    /**
     * If true new text will overwrite existing text.
     */
    protected boolean insertMode = false;
    /**
     * Used to track the shift key.
     * If this is false, cancel the current selection on moving using the direction arrows.
     * If this is true, stretch the current selection on moving using the direction arrows.
     */
    protected boolean selectionMove = false;
    /**
     * Used for mouse dragging.
     * If this is true, dragging the mouse stretches the current selection;
     * otherwise dragging the mouse starts a new selection.
     */
    protected boolean selectionMouse = false;
    /**
     * Used to track the CTRL key.
     * If true, selections should be the shape of a box; otherwise, selections
     * should be in the default form.
     */
    protected boolean selectionBox = false;
    /**
     * The height of each row in the textarea
     */
    protected int rowHeight = 1;
    /**
     * The width of each column in the textarea
     */
    protected int columnWidth = 1;
    /**
     * The current row of the caret
     */
    protected int row = 0;
    /**
     * The current column of the caret.
     */
    protected int column = 0;
    /**
     * The current text selection
     */
    protected Selection selection = new TextLines();
    /**
     * The colour map to use when the textarea is greyed out.
     */
    protected ColourMap greyedOutMap = new MonochromeColourMap(Color.BLACK, Color.LIGHT_GRAY, Color.DARK_GRAY, Color.WHITE);
    /**
     * Handles printing text
     */
    protected ColourMap normalMap = new MonochromeColourMap(Color.BLACK, Color.WHITE, Color.WHITE, Color.BLUE);
    /**
     * Handles printing text
     */
    protected ColourMap colourMap = normalMap;
    /**
     * The right click menu for the text area.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Linked list to store all cursor listeners
     */
    protected LinkedList<CursorListener> cursorListeners = new LinkedList<CursorListener>();
    /**
     * Linked list to store all mode change listeners
     */
    protected LinkedList<ModeListener> modeListeners = new LinkedList<ModeListener>();
    /**
     * The colour of invalid areas in the textarea.
     */
    protected final Color invalidzone = Color.RED;
    /**
     * Self-reference for inner classes.
     */
    public final BLTextArea blTextAreaSelf = this;
    /**
     * This constant is used for serialization purposes.
     */
    public static final long serialVersionUID = 7526472295622777004L;

    /**
     * Creates a new instance of BLTextArea
     *
     * @param instance the BioLegat instance to attach the textarea to
     * @param data the document to use for the textarea
     */
    public BLTextArea (ProgramWindow instance) {
        // initialize variables
        this.instance = instance;

        // add the listeners
        instance.getSeqDoc().addListener(this);
        selection.addListener(this);
        addKeyListener(this);
        addFocusListener(this);
	addMouseListener(this);
        addMouseMotionListener(this);

        // create popup menu
        addPopupMenuItem(new JMenuItem(new AbstractAction("Cut") {

            private static final long serialVersionUID = 7526472295622777029L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.cut();
            }

        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Copy") {

            private static final long serialVersionUID = 7526472295622777030L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.copy();
            }

        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Paste") {

            private static final long serialVersionUID = 7526472295622777031L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.paste();
            }

        }));

        // set display
        setFont(ProgramWindow.DEFAULT_FONT);
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
        setFocusTraversalKeysEnabled(false);
        refreshSize();
        setDoubleBuffered(true);
        repaint();
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addCursorListener (CursorListener listener) {
        cursorListeners.add(listener);
    }

    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addModeListener (ModeListener listener) {
        modeListeners.add(listener);
    }

    /**
     * Updates/moves the cursor to the new position.
     *
     * @param newColumn the column co-ordinate of the new position.
     * @param newRow the row co-ordinate of the new position.
     */
    public void changePosition (int newColumn, int newRow) {
        final int oldRow = row;
        final int oldColumn = column;
        
        // ensure that the new row and column are valid
        newRow = Math.max(0, Math.min(newRow, (instance.getSeqDoc().getLineCount() - 1)));
        newColumn = Math.max(0, Math.min(newColumn, (instance.getSeqDoc().getLineLength(newRow))));

        // update the row and column
        this.row = newRow;
        this.column = newColumn;

        // repaint the new and old cursor areas
        repaint(column2X(oldColumn), row2Y(oldRow), column2X(oldColumn), row2Y(oldRow + 1));
        repaint(column2X(column), row2Y(row), column2X(column), row2Y(row + 1));
        
        // update any cursor listeners
        for (CursorListener listener : cursorListeners) {
            listener.cursorChange(this, column, row);
        }
    }

    /**
     * Returns the SeqDoc associated with the BLTextArea
     *
     * @return the SeqDoc object.
     */
    public SeqDoc getSeqDoc () {
        return instance.getSeqDoc();
    }

///////////////////////
//*******************//
//* DISPLAY METHODS *//
//*******************//
///////////////////////
    /**
     * Changes the colour map for unselected text.
     *
     * @param c the colour map to use
     */
    public void setColourMap (ColourMap c) {
        colourMap = c;
	normalMap = colourMap;
        c.setBackground(getBackground());
        repaint();
    }

    /**
     * Changes the background of the canvas.
     *
     * @param c the new colour for the canvas background.
     */
    @Override
    public void setBackground (Color c) {
        super.setBackground(c);
        colourMap.setBackground(c);
    }

    /**
     * Obtains the sequence text currently selected.
     *
     * @return the text currently selected in the textarea.
     */
    public Sequence[] getData () {
        // TODO: getSelectedText
        return selection.getSelectionSequences(instance.getSeqDoc());
    }

    /**
     * Paints the current textarea
     *
     * @param gfx the graphics instance to paint the window to.
     */
    @Override
    public void paintComponent (Graphics gfx) {
        Rectangle area = gfx.getClipBounds();   // get the area to paint

        paint(gfx, area.x, area.y, area.width, area.height);
    }
    
    /**
     * Paints the current textarea
     *
     * @param gfx the graphics instance to paint the window to.
     * @param x the x-co-ordinate to start painting.
     * @param y the y-co-ordinate to start painting.
     * @param width the width of the area to be painted.
     * @param height the height of the area to be painted.
     */
    public void paint (Graphics gfx, final int x, final int y, final int width, final int height) {
        Sequence currentLine;
        LineSelection intersect;
        final int startrow = Y2Row(y);
        final int endrow = Math.min(instance.getSeqDoc().getLineCount(), startrow + Y2Row(height) + 1);
        final int startcol = X2Column(x);
        final int endcol = startcol + X2Column(width);
        final int xstart = column2X(startcol);
        final int ystart = row2Y(startrow);

        if (gfx != null) {
            // print the background
            gfx.setColor(getBackground());
            gfx.fillRect(xstart, ystart, width, height);

            // print each sequence
            for (int count = startrow; count < endrow; count ++) {
                // get the data to print
                currentLine = instance.getSeqDoc().getSequence(count);

                // determine whether ther is a selection
                intersect = selection.intersect(count, currentLine.getField("sequence").toString().length());

                // substring the data to fit the area to print
                currentLine = currentLine.subseq(startcol, endcol);
                
                // fix start and end positions of selection based on substring
                if (intersect != null) {
                    intersect = new LineSelection(intersect.getStart() - startcol, intersect.getEnd() - startcol);
                }

                // print the sequence
                colourMap.drawString(this, gfx, xstart, row2Y(count), currentLine, intersect);
            }

            // draw the cursor.
            if (hasFocus() && column2X(column) >= x && column2X(column) <= x + width
                    && row2Y(row) >= y && row2Y(row + 1) <= y + height) {
                gfx.setColor(getForeground());
                gfx.drawLine(column2X(column), row2Y(row), column2X(column), row2Y(row + 1));
            }
        }
    }
////////////////////////
//********************//
//* KEYBOARD METHODS *//
//********************//
////////////////////////

    /**
     * Processes the typing of keys within the text area
     *
     * @param event the KeyEvent for the key typed
     */
    public void keyTyped (KeyEvent event) {
        boolean canInsert = true;
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                case KeyEvent.VK_DELETE:
                    break;
                case KeyEvent.VK_ENTER:
                    // now only changes the row (same effect as downkey, except clears selection)
                    if (row + 1 < instance.getSeqDoc().getLineCount()) {
                        changePosition(Math.min(instance.getSeqDoc().getLineLength(row + 1), column), row + 1);
                        selection.reset();
                    }
                    break;
                default:
                    if ( ! selection.isEmpty()) {
                        deleteSelection();
                    } else if (insertMode && instance.getSeqDoc().getLength() > 0) {
                        canInsert = delete(getDocPosition(), 1);
                    }
                    if (canInsert && insert(event.getKeyChar())) {
                        changePosition(column + 1, row);
                    }
                break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    /**
     * Processes key presses within the text area
     *
     * @param event the KeyEvent for the key pressed
     */
    public void keyPressed (KeyEvent event) {
        int backspacecol = 0;
        boolean moved = false;  // used for updating selections

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                    if ( ! selection.isEmpty()) {
                        deleteSelection();
                    } else if (row > 0 || column > 0) {
                        if (column <= 0 && row > 0) {
                            backspacecol = instance.getSeqDoc().getLineLength(row - 1);
                        }
                        if (delete(getDocPosition() - 1, 1)) {
                            if (column <= 0 && row > 0) {
                                changePosition(backspacecol, row - 1);
                            } else {
                                changePosition(column - 1, row);
                            }
                        }
                    }
                    break;
                case KeyEvent.VK_DELETE:
                    if ( ! selection.isEmpty()) {
                        deleteSelection();
                    } else if (instance.getSeqDoc().getLength() > 0) {
                        delete(getDocPosition(), 1);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            if (selection.isEmpty()) {
                                selection.move(column, row);
                                selection.stretch(column, row);
                            }
                            selectionMove = true;
                            break;
                        case KeyEvent.VK_CONTROL:
                            selectionBox = true;
                            break;
                        case KeyEvent.VK_LEFT:
                            if (column > 0) {
                                changePosition(column - 1, row);
                                moved = true;
                            } else if (row > 0) {
                                changePosition(instance.getSeqDoc().getLineLength(row), row - 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_RIGHT:
                            if (column + 1 <= instance.getSeqDoc().getLineLength(row)) {
                                changePosition(column + 1, row);
                                moved = true;
                            } else if (row + 1 < instance.getSeqDoc().getLineCount()) {
                                changePosition(0, row + 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_UP:
                            if (row > 0) {
                                changePosition(Math.min(instance.getSeqDoc().getLineLength(row - 1), column), row - 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_DOWN:
                            if (row + 1 < instance.getSeqDoc().getLineCount()) {
                                changePosition(Math.min(instance.getSeqDoc().getLineLength(row + 1), column), row + 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_HOME:
                            changePosition(0, row);
                            moved = true;
                            break;
                        case KeyEvent.VK_END:
                            changePosition(instance.getSeqDoc().getLineLength(row), row);
                            moved = true;
                            break;
                        case KeyEvent.VK_COPY:
                            copy();
                            break;
                        case KeyEvent.VK_CUT:
                            cut();
                            break;
                        case KeyEvent.VK_PASTE:
                            paste();
                            break;
                        case KeyEvent.VK_INSERT:
                            insertMode =  ! insertMode;
                            for (ModeListener listener : modeListeners) {
                                listener.insertionMode(insertMode);
                            }
                            break;
                        default:
                            if (instance.getProperty("debug").equalsIgnoreCase("true")) {
                                org.biolegato.core.main.BLMain.warning("Unhandled key pressed --- getKeyChar() = " + ((int) event.getKeyChar()) + "\tgetKeyCode() = " + event.getKeyCode(), "BLTextArea");
                            }
                            break;
                    }
                    if (column >= instance.getSeqDoc().getLineLength(row)) {
                        changePosition(instance.getSeqDoc().getLineLength(row), row);
                    }
                    if (moved) {
                        if (selectionMove) {
                            if (selectionBox) {
                                selectRectangle();
                            } else {
                                selectNormal();
                            }
                            selection.stretch(column, row);
                        } else if (!selection.isEmpty()) {
                            selection.reset();
                        }
                        moved = false;
                    }
                    break;
                default:
                    deleteSelection();
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    /**
     * Processes key releases within the text area
     *
     * @param event the KeyEvent for the key released
     */
    public void keyReleased (KeyEvent event) {
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = false;
                            break;
                        case KeyEvent.VK_CONTROL:
                            selectionBox = false;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

/////////////////////
//*****************//
//* MOUSE METHODS *//
//*****************//
/////////////////////
    /**
     * Handles mouse clicks.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set, in which case it stretches the selection).
     *
     * @param event the MouseEvent object corresponding to the click.
     */
    public void mouseClicked (MouseEvent event) {
        if (!event.isPopupTrigger()) {
            // calculate the row and column numbers from the X and Y co-ordinates.
            changePosition(X2Column(event.getX()), Y2Row(event.getY()));

            // update the selection (if necessary)
            if ( ! selectionMove) {
                selection.reset();
            } else {
                selection.stretch(column, row);
            }
            
            // request focus within the current window
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse button presses.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set).
     *
     * @param event the MouseEvent object corresponding to the press.
     */
    public void mousePressed (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(this, event.getX(), event.getY());
        } else if ( ! selectionMove) {
            selection.reset();
        }
    }

    /**
     * Handles mouse button releases.
     * This method sets selectionMouse to false
     *
     * @param event the MouseEvent object corresponding to the release.
     */
    public void mouseReleased (MouseEvent event) {
        selectionMouse = false;
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX(), event.getY());
        }
    }

    /**
     * Handles the mouse entering the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the enter.
     */
    public void mouseEntered (MouseEvent event) {
    }

    /**
     * Handles the mouse exiting the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the exit.
     */
    public void mouseExited (MouseEvent event) {
    }

    /**
     * Handles mouse drags.
     * Updates the selection and caret on drags.
     *
     * @param event the MouseEvent object corresponding to the drag.
     */
    public void mouseDragged (MouseEvent event) {
        // calculate the row and column numbers from the X and Y co-ordinates and update the position
        changePosition(X2Column(event.getX()), Y2Row(event.getY()));
        
        if (selectionBox &&  ! (selection instanceof TextRectangle)) {
            selectRectangle();
        } else if ( ! selectionBox && selection instanceof TextRectangle) {
            selectNormal();
        }
        if ( ! selectionMouse) {
            selection.move(column, row);
            selectionMouse = true;
        } else {
            selection.stretch(column, row);
        }
        requestFocus();
        requestFocusInWindow();
    }

    /**
     * Handles mouse movements.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the movement.
     */
    public void mouseMoved (MouseEvent event) {
    }
////////////////////
//****************//
//* MATH METHODS *//
//****************//
////////////////////

    /**
     * Converts a row number into a Y co-ordinate
     *
     * @param r the row number
     * @return the corresponding Y value
     */
    public int row2Y (int r) {
        return (r * rowSize());
    }

    /**
     * Converts a column number into a X co-ordinate
     *
     * @param c the column number
     * @return the corresponding X value
     */
    public int column2X (int c) {
        return (c * columnSize());
    }
    
    /**
     * Returns the size of one row of the textarea.
     *
     * @return the row size of textarea.
     */
    public int rowSize() {
        return rowHeight;
    }
    
    /**
     * Returns the size of one column of the textarea.
     *
     * @return the column size of textarea.
     */
    public int columnSize() {
        return columnWidth;
    }

    /**
     * Converts a Y co-ordinate into a row number
     *
     * @param Y the Y value
     * @return the corresponding row number
     */
    public int Y2Row (int Y) {
        return Math.round(Y / rowSize());
    }

    /**
     * Converts a X co-ordinate into a column number
     *
     * @param X the X value
     * @return the corresponding column number
     */
    public int X2Column (int X) {
        return Math.round(X/ columnSize());
    }

    /**
     * Returns the current maximum number of visible rows
     *
     * @return the number of visible rows
     */
    public int maxRows () {
        return (getHeight() / rowSize());
    }

    /**
     * Returns the current maximum number of visible columns
     *
     * @return the number of visible columns
     */
    public int maxColumns () {
        return (getWidth() / columnSize());
    }

    /**
     * Used to calculation the insert value for the current row/column
     *
     * @return the current position to insert the text at.
     */
    public int getDocPosition () {
        changePosition(column, Math.max(0, Math.min(row, instance.getSeqDoc().getLineCount() - 1)));
        changePosition(Math.max(0, Math.min(column, instance.getSeqDoc().getLineEndOffset(row))), row);
        return Math.max(0, Math.min(instance.getSeqDoc().getLineEndOffset(row), instance.getSeqDoc().getLineStartOffset(row) + column));
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     *
     * @return whether or not the deletion was performed.
     */
    protected boolean deleteSelection () {
        boolean result = false;

        if ( ! selection.isEmpty()) {
            changePosition(selection.getMinX(), selection.getMinY());
            result = selection.delete(this);
            selection.reset();
        }
        return result;
    }

    /**
     * Adds an item to the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to add.
     */
    public void addPopupMenuItem (JMenuItem act) {
        popup.add(act);
    }

    /**
     * Removes an item from the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to remove.
     */
    public void removePopupMenuItem (JMenuItem act) {
        popup.remove(act);
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont(Font font) {
        super.setFont(font);
        columnWidth = 0;
        for (int fontWidth : getFontMetrics(getFont()).getWidths()) {
            columnWidth = Math.max(columnWidth, fontWidth);
        }
        rowHeight = getFontMetrics(getFont()).getHeight();
        refreshSize();
        repaint();
    }

    /**
     * Copies content from the current Editable object.
     */
    public void copy () {
        if ( ! selection.isEmpty()) {
            instance.setClipboard(selection.getSelectionSequences(instance.getSeqDoc()));
        }
    }

    /**
     * Cuts content from the current Editable object.
     */
    public void cut () {
        if ( ! selection.isEmpty()) {
            copy();
            deleteSelection();
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste () {
        deleteSelection();
        insert(instance.getClipboard());
    }

    /**
     * Selects all the sequences within the textarea.
     */
    public void selectAll() {
        selectNormal();
        changePosition(0, 0);
        selection.move(0, 0);
        selection.reset();
        changePosition(column, instance.getSeqDoc().getLineCount() - 1);
        changePosition(instance.getSeqDoc().getLineLength(row), row);
        selection.stretch(column, row);
    }

    /**
     * Undoes data modification.
     *
     * @return whether or not the undo was successful.
     */
    public boolean undo () {
        return instance.getSeqDoc().undo();
    }

    /**
     * Redoes data modification.
     *
     * @return whether or not the redo was successful.
     */
    public boolean redo () {
        return instance.getSeqDoc().redo();
    }

    /**
     * Inserts a character into the textarea's underlying SeqDoc
     *
     * @param character the character to insert.
     * @return whether or not the insertion was successful.
     */
    protected boolean insert (char character) {
        return instance.getSeqDoc().insert(row, column, "" + character);
    }

    /**
     * Inserts some sequences into the textarea's underlying SeqDoc
     *
     * @param paste the sequences to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert (Sequence[] paste) {
        return instance.getSeqDoc().insert(getDocPosition(), paste);
    }

    /**
     * Deletes characters from the textarea's underlying SeqDoc
     *
     * @param offset the offset to start the deletion from.
     * @param length the number of characters to delete.
     * @return whether or not the eletion was successful.
     */
    public boolean delete (int offset, int length) {
        return instance.getSeqDoc().delete(Math.max(0, offset), length);
    }

    /**
     * Returns the insertion mode status of the textarea.
     *
     * Insertion mode is invoked by hitting the "INSERT" key.
     * While in insertion mode, if you type a character key in front of some text
     * the character key inserted will replace the first character of the text in front
     * of the insertion point.
     *
     * @return whether or not insertion mode is on.
     */
    public boolean getInsertMode () {
        return insertMode;
    }

    /**
     * Refreshes the size of the textarea (for scroll size purposes)
     */
    protected void refreshSize() {
        Dimension size = new Dimension((instance.getSeqDoc().getLongestLineLength() + 1) * columnSize(), rowSize() * instance.getSeqDoc().getLineCount());
        setSize(size);
        setPreferredSize(size);
    }

    /**
     * Called when a sequence is added to a SeqDoc.
     *
     * @param index the location the sequence was added.
     * @param sequence the sequence added.
     */
    public void sequenceAdded(SeqDoc source, final int index, Sequence sequence) {
        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(index);                 // the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight();  // the height of the area to repaint
        refreshSize();                              // update the size of the textarea
        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Called when a field in a sequence is modified.
     *
     * @param index the location of the sequence.
     * @param sequence the sequence modified.
     * @param key the value modified.
     */
    public void sequenceChanged(SeqDoc source, int index, Sequence sequence, String key) {
        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(index);                 // the Y co-ordinate to start repainting at
        final int h = rowSize();                    // the height of the area to repaint
        
        refreshSize();                              // update the size of the textarea
        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Called when a sequence is removed from a SeqDoc.
     *
     * @param index the location (line number) where the sequence was removed from.
     * @param sequence the sequence removed from the SeqDoc.
     */
    public void sequenceRemoved(SeqDoc source, int index, Sequence sequence) {
        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(index);                 // the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight();  // the height of the area to repaint
        
        refreshSize();                              // update the size of the textarea
        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Notifies a listener that the selection start has been moved.
     *
     * @param oldx the old X co-ordinate to start the selection from.
     * @param oldy the old Y co-ordinate to start the selection from.
     * @param newx the new X co-ordinate the selection has been moved to.
     * @param newy the new Y co-ordinate the selection has been moved to.
     */
    public void selectionMoved(int oldx, int oldy, int newx, int newy) {
        final int x = 0;                                                        // the X co-ordinate to start repainting at
        final int y = row2Y(Math.min(oldy, selection.getMinY()) - 1);           // the width of the area to repaint
        final int w = (int)getSize().getWidth();                                // the Y co-ordinate to start repainting at
        final int h = row2Y(Math.max(oldy, selection.getMaxY()) + 1);           // the height of the area to repaint
        
        repaint(x, Math.max(0, y), w, Math.min((int)getSize().getHeight(), h)); // repaint the modified area of the textarea
    }
    
    /**
     * Notifies a listener that the selection stretched.
     *
     * @param oldx the old X co-ordinate the selection was stretched to.
     * @param oldy the old Y co-ordinate the selection was stretched to.
     * @param newx the new X co-ordinate the selection has been stretched to.
     * @param newy the new Y co-ordinate the selection has been stretched to.
     */
    public void selectionStretched(int oldx, int oldy, int newx, int newy) {
        final int x = 0;                                                        // the X co-ordinate to start repainting at
        final int y = row2Y(Math.min(oldy, selection.getMinY()) - 1);           // the width of the area to repaint
        final int w = (int)getSize().getWidth();                                // the Y co-ordinate to start repainting at
        final int h = row2Y(Math.max(oldy, selection.getMaxY()) + 1);           // the height of the area to repaint
        
        repaint(x, Math.max(0, y), w, Math.min((int)getSize().getHeight(), h)); // repaint the modified area of the textarea
    }

    /**
     * Handles activating the text area (changing colours from greyed out to normal).
     *
     * @param e this parameter is currently ignored
     */
    public void focusGained(FocusEvent e) {
        colourMap = normalMap;
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
	repaint();
    }

    /**
     * Handles deactivating the text area (changing colours from normal to greyed out).
     *
     * @param e this parameter is currently ignored
     */
    public void focusLost(FocusEvent e) {
        colourMap = greyedOutMap;
        setForeground(Color.BLACK);
        setBackground(Color.LIGHT_GRAY);
	repaint();
    }

    /**
     * Changes the selection shape to a rectangle select
     */
    public void selectRectangle () {
        selection = new TextRectangle(selection);
        repaint();
    }
    
    /**
     * Changes the selection shape to a the normal selection shape
     */
    public void selectNormal() {
        selection = new TextLines(selection);
        repaint();
    }
}
