/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.selections;

import java.util.LinkedList;
import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.gdesupport.canvas.BLTextArea;
import org.biolegato.core.data.sequence.Sequence;

    /**
     * Used to handle selections of text in the textarea
     */
    public abstract class Selection {

        /**
         * The start column of the selection
         */
        private int x1 = -1;
        /**
         * The start row of the selection
         */
        private int y1 = -1;
        /**
         * The end column of the selection
         */
        private int x2 = -1;
        /**
         * The end row of the selection
         */
        private int y2 = -1;
        /**
         * The list containing all selection listeners for the current object.
         */
        private LinkedList<SelectionListener> selectionListeners = new LinkedList<SelectionListener>();
                
        /**
         * Creates a new Selection object (with the parameters -1,-1,-1,-1).
         */
        public Selection () {
        }
        
        /**
         * Deep copies a selection
         *
         * @param s the section to obtain x1,y1,x2,y2 from
         */
        public Selection (Selection s) {
            this.selectionListeners = s.selectionListeners;
            move(s.x1, s.y1);
            stretch(s.x2, s.y2);
        }

////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////
        /**
         * Moves the start of the selection.
         *
         * @param newx the new column to start the selection from
         * @param newy the new row to start the selection from
         */
        public void move (int newx, int newy) {
            final int oldx = x1;
            final int oldy = y1;
            
            this.x1 = newx;
            this.y1 = newy;
            
            // send information to all listener objects
            for (SelectionListener l : selectionListeners) {
                l.selectionMoved(oldx, oldy, newx, newy);
            }
        }

        /**
         * Moves the end of the selection.
         *
         * @param newx the new column to end the selection at
         * @param newy the new row to end the selection at
         */
        public void stretch (int newx, int newy) {
            final int oldx = x2;
            final int oldy = y2;

            this.x2 = newx;
            this.y2 = newy;

            // send information to all listener objects
            for (SelectionListener l : selectionListeners) {
                l.selectionStretched(oldx, oldy, newx, newy);
            }
        }

//////////////////////////////////
//******************************//
//* INDIRECT DATA MODIFICATION *//
//******************************//
//////////////////////////////////
       /**
         * Resets the selection to an invalid empty value.
         */
        public void reset () {
            move(-1,-1);
            stretch(-1,-1);
        }
        
//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////
       /**
         * Used to obtain the minimum intersection values
         *  of two selection shapes on a single axis.
         *
         * @param t1x1 point #1's starting x value
         * @param t1x2 point #1's ending x value
         * @param t2x1 point #2's starting x value
         * @param t2x2 point #2's ending x value
         * @return the intersection point on the axis (returns null if the points don't intersect)
         */
        protected Integer axisIntersect (int t1x1, int t1x2, int t2x1, int t2x2) {
            Integer xint = null;
            if ((t2x1 <= t2x2) && ((t1x1 <= t2x1 && t2x1 <= t1x2) || (t1x2 <= t2x1 && t2x1 <= t1x1))) {
                xint = t2x1;
            } else if ((t1x1 <= t1x2) && ((t2x1 <= t1x1 && t1x1 <= t2x2) || (t2x2 <= t1x1 && t1x1 <= t2x1))) {
                xint = t1x1;
            } else if ((t2x1 >= t2x2) && ((t1x1 <= t2x2 && t2x2 <= t1x2) || (t1x2 <= t2x2 && t2x2 <= t1x1))) {
                xint = t2x2;
            } else if ((t1x1 >= t1x2) && ((t2x1 <= t1x2 && t1x2 <= t2x2) || (t2x2 <= t1x2 && t1x2 <= t2x1))) {
                xint = t1x2;
            }
            return xint;
        }

        /**
         * Returns the current value of X1
         * (the staring column of the selection)
         *
         * @return the value of x1
         */
        public int getX1 () {
            return x1;
        }

        /**
         * Returns the current value of X2
         * (the ending column of the selection)
         *
         * @return the value of x2
         */
        public int getX2 () {
            return x2;
        }

        /**
         * Returns the current value of Y1
         * (the staring row of the selection)
         *
         * @return the value of y1
         */
        public int getY1 () {
            return y1;
        }

        /**
         * Returns the current value of Y2
         * (the ending row of the selection)
         *
         * @return the value of y2
         */
        public int getY2 () {
            return y2;
        }

        /**
         * Gets the minimum of getX1() and getX2()
         *
         * @return Math.min(x1, x2)
         */
        public int getMinX () {
            return Math.min(x1, x2);
        }

        /**
         * Gets the maximum of getX1() and getX2()
         *
         * @return Math.max(x1, x2)
         */
        public int getMaxX () {
            return Math.max(x1, x2);
        }

        /**
         * Gets the minimum of getY1() and getY2()
         *
         * @return Math.min(y1, y2)
         */
        public int getMinY () {
            return Math.min(y1, y2);
        }

        /**
         * Gets the maximum of getY1() and getY2()
         *
         * @return Math.max(y1, y2)
         */
        public int getMaxY () {
            return Math.max(y1, y2);
        }

        /**
         * Tests if the selection shape is empty
         *
         * @return true if the selection is empty
         */
        public boolean isEmpty () {
            return (x1 == x2 && y1 == y2);
        }

        /**
         * Adds a selection listener to the selection object.
         *
         * @param listener the selection listener to add.
         */
        public void addListener(SelectionListener listener) {
            selectionListeners.add(listener);
        }
        
        /**
         * Used to obtain all of the text selected within a document according
         * to the current selection shape object.
         *
         * @param document the document to obtain the selection from
         * @return the text selected
         */
        public String getSelectionText (SeqDoc document) {
            StringBuffer result = new StringBuffer();
            Sequence[] sequences = getSelectionSequences(document);

            // NOTE: append is faster than + or concat operators
            for (int count = 0; count < sequences.length; count ++) {
                if (count > 0) {
                    result.append("\n");
                }
                result.append(sequences[count].getField("sequence").toString());
            }
            return result.toString();
        }
        
////////////////////////
//********************//
//* ABSTRACT METHODS *//
//********************//
////////////////////////

        /**
         * Returns the intersection between a line of text and the current object
         *
         * @param line the line number of the line to test
         * @param length the total length of the line
         * @return the interstection (null if the line does not intersect)
         */
        public abstract LineSelection intersect (int line, int length);

        /**
         * Used to obtain all of the text selected within a document according
         * to the current selection shape object.
         *
         * @param document the document to obtain the selection from
         * @return the text selected
         */
        public abstract Sequence[] getSelectionSequences (SeqDoc document);

        /**
         * Used to delete all of the text selected within a document according
         * to the current selection shape object.
         *
         * @param textarea the textarea to delete the selection from
         * @return whether the deletion was successful
        s		 */
        public abstract boolean delete (BLTextArea textarea);

    }
