/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.selections;

import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.gdesupport.canvas.BLTextArea;
import java.util.LinkedList;

/**
 * Used to handle standard selections (i.e. all lines and fragments
 * of text located inbetween a start and end point within a document).
 */
public class TextLines extends Selection {

    /**
     * Creates a new TextRectangle object (with default parameters).
     */
    public TextLines () {
        super();
    }
    
    /**
     * Deep copies a selection
     *
     * @param s the section to obtain x1,y1,x2,y2 from
     */
    public TextLines (Selection s) {
        super(s);
    }

    /**
     * Returns the intersection between a line of text and the current object
     *
     * @param line the line number of the line to test
     * @param length the total length of the line
     * @return the interstection (null if the line does not intersect)
     */
    public LineSelection intersect (int line, int length) {
        LineSelection intersect = null;

        if (line > getMinY() && line < getMaxY()) {
            intersect = new LineSelection(0, length);
        } else if (line == getY1() && line == getY2()) {
            intersect = new LineSelection(getMinX(), getMaxX());
        } else if (line == getY1()) {
            if (getY1() == getMinY()) {
                intersect = new LineSelection(getX1(), length);
            } else {
                intersect = new LineSelection(0, getX1());
            }
        } else if (line == getY2()) {
            if (getY2() == getMinY()) {
                intersect = new LineSelection(getX2(), length);
            } else {
                intersect = new LineSelection(0, getX2());
            }
        }
        return intersect;
    }

    /**
     * Used to obtain all of the text selected within a document according
     * to the current selection shape object.
     *
     * @param document the document to obtain the selection from
     * @return the text selected
     */
    public Sequence[] getSelectionSequences (SeqDoc document) {
        int length = 0;
        int startLine = 0;
        int startColumn = 0;
        int endLine = 0;
        int endColumn = 0;
        String currentSequence = null;
        Sequence current = null;
        LinkedList<Sequence> result = new LinkedList<Sequence>();

        if ( ! isEmpty()) {
            if (getY1() < getY2()) {
                startLine = getY1();
                startColumn = getX1();
                endLine = getY2();
                endColumn = getX2();
            } else if (getY1() == getY2()) {
                startLine = getY1();
                startColumn = getMinX();
                endLine = getY2();
                endColumn = getMaxX();
            } else {
                startLine = getY2();
                startColumn = getX2();
                endLine = getY1();
                endColumn = getX1();
            }
            
            // prevent invalid start and end line values
            startLine = Math.max(0, Math.min(startLine, document.getLineCount() - 1));
            endLine = Math.max(0, Math.min(endLine, document.getLineCount() - 1));
            
            for (int count = startLine; count <= endLine; count ++) {
                length = document.getLineLength(count);
                current = (Sequence) document.getSequence(count).clone();
                currentSequence = current.getField("sequence").toString();
                if (count == endLine && endColumn < length) {
                    currentSequence = currentSequence.substring(0, endColumn);
                }
		if (count == startLine && startColumn > 0) {
                    currentSequence = currentSequence.substring(startColumn);
                }
		if (!currentSequence.equals(current.getField("sequence").toString())) {
		    current.setField("sequence", currentSequence);
		}
                result.add(current);
            }
        }
        return result.toArray(new Sequence[0]);
    }

    /**
     * Used to delete all of the text selected within a document according
     * to the current selection shape object.
     *
     * @param textarea the textarea to delete the selection from
     * @return whether the deletion was successful
     */
    public boolean delete (BLTextArea textarea) {
        int pos1 = textarea.getSeqDoc().getLineStartOffset(getY1()) + getX1();
        int pos2 = textarea.getSeqDoc().getLineStartOffset(getY2()) + getX2();
        boolean result = false;

        if ( ! isEmpty()) {
            result = textarea.delete(Math.min(pos1, pos2), Math.abs(pos1 - pos2));
        }
        return result;
    }

}
