package org.biolegato.gdesupport.files;

/*
 * GDEFile.java
 *
 * Created on January 30, 2008, 11:58 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.Sequence.Direction;
import org.biolegato.core.data.sequence.Sequence.Strandedness;
import org.biolegato.core.data.sequence.Sequence.Topology;
import org.biolegato.core.data.sequence.Sequence.Type;
import org.biolegato.core.plugintypes.DataFormat;
import java.io.File;
import java.io.Reader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;
import java.util.LinkedList;

/**
 *
 * @author alvare
 */
public class GDEFile extends DataFormat {

    /** Creates a new instance of GDEFile */
    public GDEFile () {
    }

    /**
     * Translates a sequence to the GDE file format
     * @param seq the sequence to convert
     * @return the resulting string
     */
    public String translateTo (Sequence seq) {
        StringBuffer result = new StringBuffer();

        // translate the data
        // NOTE: append is faster than + or concat operators
        if (seq != null) {
            result.append("{\nname            \"").append(quote(seq.getField("name").toString())).append("\"\n").append("longname        \"").append(quote(seq.getField("name").toString())).append("\"\n").append("sequence-ID     \"").append(quote(seq.getField("accession").toString())).append("\"\n").append("creation-date   \"").append((new SimpleDateFormat("MM/dd/yy kk:mm:ss")).format(new Date())).append("\"\n").append("direction       ").append((seq.getField("direction").equals(Direction.FROM5TO3) ? "1" : "-1")).append("\n").append("strandedness    ").append((seq.getField("strandedness").equals(Strandedness.SINGLE) ? "1" : "2")).append("\n").append("type            ").append(blTypetoGDE((Type) seq.getField("type"))).append("\n").append((seq.getField("topology").equals(Topology.CIRCULAR) ? "circular        1\n" : "")).append("offset          0\n").append("groupID         0\n").append("creator         \"").append(quote("")).append("\"\n").append("descrip         \"").append(quote(seq.getField("description").toString())).append("\"\n").append("comments        \"").append(quote(seq.getField("comments").toString())).append("\"\n").append("sequence        \"").append(quote(seq.getField("sequence").toString())).append("\"\n}\n");
        }
        return result.toString();
    }

    /**
     * Translates a string from the given file format into the BioLegato internal format
     * @param data  the string to convert
     * @return  the resulting string
     * @throws java.io.IOException 
     */
    public Sequence[] translateFrom (java.io.BufferedReader data) throws
            java.io.IOException {
        String line = "";
        String fname = "";
        String fvalue = "";
        StringBuffer fvalueBuffer;                                              // used for reading multiple lines
        LinkedList<Sequence> result = new LinkedList<Sequence>();
        Hashtable<String, Object> sequence = new Hashtable<String, Object>();

        // translate the data
        while ((line = data.readLine()) != null) {
            line = line.trim();
            if ( ! line.equals("")) {
                if ((line.contains("}") || line.contains("{")) &&
                     ! sequence.isEmpty() && sequence.containsKey("sequence")) {
                    result.add(new Sequence(sequence));
                    sequence.clear();
                    line = line.substring(line.indexOf('{') + 1).trim();
                }
                if (line.contains(" ") && ( ! line.contains("\t") ||
                                           line.indexOf(' ') <
                                           line.indexOf('\t'))) {
                    fname = line.substring(0, line.indexOf(' ')).trim();
                    fvalue = line.substring(line.indexOf(' ') + 1).trim();
                } else if (line.contains("\t")) {
                    fname = line.substring(0, line.indexOf('\t')).trim();
                    fvalue = line.substring(line.indexOf('\t') + 1).trim();
                } else {
                    fname = "";
                    fvalue = "";
                }
                if (fvalue.startsWith("\"")) {
                    fvalueBuffer = new StringBuffer();
                    fvalue = fvalue.substring(1);
                    fvalueBuffer.append(fvalue);
                    while (fvalue != null && ! fvalue.contains("\"")) {
                        // NOTE: append is faster than + or concat operators
                        fvalue = data.readLine();
                        fvalueBuffer.append(fvalue);
                    }
                    fvalue = fvalueBuffer.toString();
                    if (fvalue.contains("\"")) {
                        fvalue = fvalue.substring(0, fvalue.indexOf('\"'));
                    }
                }
                if (fname.equals("") && fvalue.equals("")) {
                } else if (fname.equals("type")) {
                    if (fvalue.equalsIgnoreCase("RNA")) {
                        sequence.put("type", Type.RNA);
                    } else if (fvalue.equalsIgnoreCase("PROTEIN")) {
                        sequence.put("type", Type.PROTEIN);
                    } else if (fvalue.equalsIgnoreCase("TEXT")) {
                        sequence.put("type", Type.TEXT);
                    } else if (fvalue.equalsIgnoreCase("MASK")) {
                        sequence.put("type", Type.MASK);
                    } else {
                        sequence.put("type", Type.DNA);
                    }
                } else if (fname.equals("circular")) {
                    if (fvalue.equals("0")) {
                        sequence.put("topology", Topology.LINEAR);
                    } else {
                        sequence.put("topology", Topology.CIRCULAR);
                    }
                } else if (fname.equals("direction")) {
                    if (fvalue.equals("-1")) {
                        sequence.put("direction", Direction.FROM3TO5);
                    } else {
                        sequence.put("direction", Direction.FROM5TO3);
                    }
                } else if (fname.equals("strandedness")) {
                    if (fvalue.equals("1")) {
                        sequence.put("strandedness", Strandedness.SINGLE);
                    } else {
                        sequence.put("strandedness", Strandedness.DOUBLE);
                    }
                } else if (fname.equals("creation-date")) {
                    sequence.put("creation-date", fvalue);
                } else if (fname.equals("creator")) {
                    sequence.put("creator", fvalue);
                } else if (fname.equals("sequence-ID")) {
                    sequence.put("accession", fvalue);
                } else if (fname.equals("name")) {
                    sequence.put("name", fvalue);
                } else if (fname.equals("longname")) {
                    sequence.put("longname", fvalue);
                } else if (fname.equals("descrip")) {
                    sequence.put("description", fvalue);
                } else if (fname.equals("comments")) {
                    sequence.put("comments", fvalue);
                } else if (fname.equals("sequence")) {
                    sequence.put("sequence", fvalue);
                } else {
                    org.biolegato.core.main.BLMain.warning("Unsupported attribute (" +
                                                           fname + " = " +
                                                           fvalue + " )",
                                                           "GDEFile plugin");
                }
            }
        }
        if ( ! sequence.isEmpty() && sequence.containsKey("sequence")) {
            result.add(new Sequence(sequence));
        }
        return result.toArray(new Sequence[0]);
    }

    /**
     * Determines whether or not the file should be accepted by the filter
     * @param	file	the file to test
     * @return	whether or not the file is accepted
     * @see javax.swing.filechooser.FileFilter#accept
     */
    public boolean accept (File file) {
        return (file.isDirectory() || file.getAbsolutePath().toLowerCase().
                endsWith(".gde"));
    }

    /**
     * The file description to display inside the JFileChooser
     * @return	the string description of the DataFormat
     * @see javax.swing.filechooser.FileFilter#getDescription
     */
    public String getDescription () {
        return "GDE file (*.gde)";
    }

    /**
     * Obtains the name of the file format for File Chooser purposes
     * @return  the name of the file format
     */
    @Override
    public String getName () {
        return "gde";
    }

    /**
     * Quotes a GDE string (replaces " with ' and {} with [])
     * @param	input	the string to quote
     * @return	the quoted string
     */
    public static String quote (String input) {
        return input.replaceAll("\"", "\'").replaceAll("\\{", "\\[").replaceAll(
                "\\}", "\\]");
    }

    /**
     * Converts the BioLegato internal type representation to GDE
     * @param	type	the type to convert from BioLegato to GDE
     * @return	the GDE equivilent
     */
    private static String blTypetoGDE (Type type) {
        String result = "DNA";
        switch (type) {
            case RNA:
                result = "RNA";
                break;
            case PROTEIN:
                result = "PROTEIN";
                break;
            case MASK:
                result = "MASK";
                break;
            case TEXT:
                result = "TEXT";
                break;
        }
        return result;
    }

    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the string to test
     * @return whether or not the format is correct
     */
    @Override
    public boolean isFormat (Reader test) {
        int check = ' ';
        try {
            while (check == ' ' || check == '\t' || check == '\n' || check ==
                                                                     '\r') {
                test.mark(2);
                check = test.read();
            }
            test.reset();
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return (check == '{');
    }

}
