/*
 * Plugin.java
 *
 * Created on October 22, 2008, 1:56 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.core.plugins;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

/**
 * This class is used to handle plugins.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class PluginLoader {

    /**
     * Stores all of the plugins lodaed by BioLegato.<br />
     * NOTE: plugins are only loaded once at the start of the program.
     */
    private static Hashtable<String, PluginWrapper> plugins = new Hashtable<String, PluginWrapper>();

    /**
     * Maintains a list with all of the plugins loaded by BioLegato.
     *
     * @return a list containing all of the plugins loaded
     */
    public static Map<String, PluginWrapper> getPluginHash () {
        return new Hashtable<String, PluginWrapper>(plugins);
    }

    /**
     * Returns a list with all of the plugins loaded by BioLegato that are
     * instances of the class type indicated.
     *
     * @param type the plugin type to find
     * @return a list containing all of the plugins loaded
     */
    public static Collection<PluginWrapper> getPlugins (Class<?> type) {
        ArrayList<PluginWrapper> list = new ArrayList<PluginWrapper>();   // the list of plugins to return

        // itterate through the plugins and extract those which are instances ot "type"
        for (PluginWrapper plugin : plugins.values()) {
            if (plugin != null && plugin.isA(type)) {
                list.add(plugin);
            }
        }
        return list;
    }

    /**
     * Retrieves a plugin by name.
     * This can be used for plugins dependant on other plugins
     *
     * @param string the plugin to retrieve
     * @return the plugin requested (null on failure)
     */
    public static PluginWrapper getPlugin (String string) {
        return plugins.get(string);
    }

    /**
     * Loads an the plugins into BioLegato.
     *
     * @param directory the directory to load the plugins from
     */
    public static void loadPlugins (String directory) {
        String className = "";			    // the name of the current class
        File pluginDirectory = new File(directory); // the current directory to load from

        // check that the directory exists, and iterate through its files
        if (pluginDirectory.exists() && pluginDirectory.isDirectory()) {
            for (File file : pluginDirectory.listFiles()) {
                try {
		    // branch based on whether the current entry is a class file, a jar file or a directory.
                    if (file.exists() && file.canRead() && file.isFile() && file.getName().toLowerCase().endsWith(".class")) {
			// if the entry is a class file, read the file's comprising classes.
                        className = file.getName().substring(0, file.getName().length() - 6);                                   /* get the class name by removing the extension from the filename */
                        loadClass(pluginDirectory.toURL(), className);
                    } else if (file.exists() && file.canRead() && file.isFile() && file.getName().endsWith(".jar")) {
			// if the entry is a jar file, read the file's contained classes.
                        loadJar(file);
                    } else if (file.exists() && file.canRead() && file.isDirectory()) {
			// if the entry is a directory, traverse the directory for files.
                        loadPlugins(file.getAbsolutePath());
                    }
                } catch (Throwable th) {
                    org.biolegato.core.main.BLMain.error("Error loading the file: " + file.getAbsolutePath(), "loadPlugins");
                    th.printStackTrace();
                }
            }
        }
    }

    /**
     * Loads all plugin classes within a jar file.
     * 
     * @param file the jar file to read classes from.
     */
    private static void loadJar (File file) throws IOException {
        Enumeration<JarEntry> entries = null;	    // the current list of entries in the jar file
        JarFile jarfile = new JarFile(file);	    // the jar file object to read
        JarEntry entry = null;			    // the current entry being parsed
        String className = "";			    // the name of the current class

        entries = jarfile.entries();
        while (entries.hasMoreElements()) {
            entry = entries.nextElement();
            if ( ! entry.isDirectory() && entry.getName().toLowerCase().endsWith(".class")) {
                // get the class name by removing the extension from the filename
                className = entry.getName().substring(0, entry.getName().length() - 6);
                className = className.replace("/", ".");
                loadClass(file.toURL(), className);
            }
        }
    }

    /**
     * Loads a class and its subclasses for a given url.
     *
     * @param url the url to load the classes in
     * @param name the name of the class to load
     * @param file the file that contains the classes
     */
    private static void loadClass (URL url, String name) {
        try {
            // load the class
            Class currentClass = new URLClassLoader(new URL[] {url}).loadClass(name);
            if (currentClass != null) {
                // insert the class into the hashtable
                plugins.put(name, new PluginWrapper(name, currentClass));

                // load any subclasses
                for (Class c : currentClass.getDeclaredClasses()) {
                    loadClass(url, c.getName());
                }
            }
        } catch (Throwable th) {
	    // on failure print an error message
            org.biolegato.core.main.BLMain.error("Error loading the class: " + url.getPath() + " (" + name + ")");
            th.printStackTrace();
        }
    }

}
