package org.biolegato.ezmenu.variables;

/*
 * GDEFileVariable.java
 *
 * Created on November 20, 2007, 4:06 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import org.biolegato.core.data.sequence.Sequence;
import java.io.File;
import org.biolegato.core.main.ProgramWindow;
import org.biolegato.core.plugintypes.DataFormat;

/**
 * Stores information about the file fields used by the command
 * 
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDETempFile extends GDEVariable {

    /**
     * The currently opened file handle.
     */
    private File currentFile = null;
    /**
     * Whether or not to add the contents of the window to the file before execution
     * (whether the temporary file is input for a program).
     */
    private boolean input = false;
    /**
     * Whether or not to add the contents of the file to the window after execution.
     * (whether the temporary file is output for a program).
     */
    private boolean output = false;
    /**
     * Determines whether or not to delete the file after execution.
     */
    private boolean save = false;
    /**
     * Corresponds to GDE's inmask directive (unknown usage).
     */
    private boolean mask = false;
    /**
     * Determines whether or not to overwrite the file if it already exists.
     */
    private boolean overwrite = false;
    /**
     * The window to obtain the canvas data from.
     */
    private ProgramWindow window = null;
    /**
     * Stores the file format of the file (used for translation).
     */
    private DataFormat format = null;

    /**
     * Creates a new instance of GDETempFile.
     *
     * @param data the data to create the temporary file variable with.
     */
    public GDETempFile (java.util.Map data) {
        super(data);

        if (data.containsKey("window") && data.get("window") instanceof org.biolegato.core.main.ProgramWindow) {
            window = ((org.biolegato.core.main.ProgramWindow) data.get("window"));
        }
        if (data.containsKey("format") && data.get("format") instanceof org.biolegato.core.plugintypes.DataFormat) {
            format = ((org.biolegato.core.plugintypes.DataFormat) data.get("format"));
            if (format == null) {
                format = org.biolegato.core.plugintypes.DataFormat.getFormat("raw");
                if (format == null) {
                    org.biolegato.core.main.BLMain.error("missing raw file type", "TempFileVariable");
                }
            }
        }
        if (data.containsKey("save") && Boolean.TRUE.equals(data.get("save"))) {
            save = true;
        }
        if (data.containsKey("overwrite") && Boolean.TRUE.equals(data.get("overwrite"))) {
            overwrite = true;
        }
        if (data.containsKey("mask") && Boolean.TRUE.equals(data.get("mask"))) {
            mask = true;
        }
        if (data.containsKey("input") && Boolean.TRUE.equals(data.get("input"))) {
            input = true;
        }
        if (data.containsKey("output") && Boolean.TRUE.equals(data.get("output"))) {
            output = true;
        }
    }

    /**
     * Gets the current filename corresponding to this variable
     * If the file does not exists, then this function creates a file to
     * use for IO and writes any initial contents
     *
     * @return the filename to use for command line substitution.
     */
    public Object getValue () {
        String result = "";	// the filename to use
        String data = "";	// the data to insert into the file

        if (currentFile == null) {
            try {
                // create the file
                currentFile = File.createTempFile("bio", null, new File(org.biolegato.core.main.ProgramWindow.CURRENT_DIR));
                if (!save) {
                    currentFile.deleteOnExit();
                }

                if (input && format != null) {
                    // write the file
                    format.writeFile(currentFile, window.getData());
                } else if (format == null) {
                    org.biolegato.core.main.BLMain.error("Invalid file format", "GDETempFile");
                }
            } catch (Throwable ex) {
                ex.printStackTrace();
            }
        }
        if (currentFile != null) {
            result = currentFile.getName();
        }
        return (result != null &&  ! "".equals(result.trim()) ? result : getName());
    }

    /**
     * Releases any files currently in use by this class
     */
    public void close () {
        Sequence[] dataIn;
        
        // releases all of the files
        if (currentFile != null && currentFile.exists()) {
            if (output) {
                if (format != null) {
                    try {
                        dataIn = format.readFile(currentFile);
                        window.addData(dataIn);
                    } catch (Throwable ex) {
                        ex.printStackTrace();
                    }
                } else {
                    org.biolegato.core.main.BLMain.error("Invalid file format", "GDETempFile");
                }
            }
            if (!save) {
                currentFile.delete();
            }
        }
        currentFile = null;
    }

    /**
     * Returns the type name
     * @return "tempfile"
     */
    public static String getType () {
        return "tempfile";
    }

}