/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas;

import java.awt.Color;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Arrays;
import javax.swing.AbstractAction;
import javax.swing.DefaultListModel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;
import org.biolegato.gdesupport.canvas.Editable;
import org.biolegato.core.main.ProgramWindow;

/**
 * The JList of sequences in the GDE canvas.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEList extends JList implements MouseListener, FocusListener, Editable {
    /**
     * The last time the mouse was clicked (used for double click detection)
     */
    private long lastClickTime = -1000;
    /**
     * The entry selected on the last click
     */
    private int lastClickEntry = -1;
    /**
     * The biolegato instance to associate with this list
     */
    protected ProgramWindow program = null;
    /**
     * Self reference.
     */
    private final GDEList listSelf = this;
    /**
     * The right click menu for the GDE sequence list.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * The menu item "Get info..."
     */
    public final JMenuItem getInfoAction = new JMenuItem(new AbstractAction("Get info...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_G));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                getInfo();
            }

        });
    /**
     * The menu item "Cut"
     */
    private final JMenuItem cutAction = new JMenuItem(new AbstractAction("Cut") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.cut();
            }

        });
    /**
     * The menu item "Copy"
     */
    private final JMenuItem copyAction = new JMenuItem(new AbstractAction("Copy") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.copy();
            }

        });
    /**
     * The menu item "Paste"
     */
    private final JMenuItem pasteAction = new JMenuItem(new AbstractAction("Paste") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.paste();
            }

        });
    /**
     * The menu item "Select All"
     */
    private final JMenuItem selectAllAction = new JMenuItem(new AbstractAction("Select All") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.selectAll();
            }

        });
    /**
     * Used in serialization.
     */
    private static final long serialVersionUID = 7526472295622777009L;

    /**
     * Constructs a new GDEList
     *
     * @param program the instance of BioLegato to attach the list to.
     */
    public GDEList (ProgramWindow program) {
        super(new SequenceListModel(program.getSeqDoc()));

        // initialize variables
        this.program = program;

        // add listeners
        addFocusListener(this);
        addMouseListener(this);

        // build list popupmenu
        addPopupMenuItem(cutAction);
        addPopupMenuItem(copyAction);
        addPopupMenuItem(pasteAction);
        addPopupMenuItem(selectAllAction);
        addPopupMenuItem(getInfoAction);

        // update the list content
        //updateList();
        
        // set display properties
        setForeground(Color.BLACK);
        setBackground(Color.LIGHT_GRAY);
        setSelectionBackground(Color.WHITE);
        setSelectionForeground(Color.GRAY);
        setSize(getPreferredSize());
    }

    /**
     * Checks for double clicks.  On a double click, this method opens
     * up a sequence properties window.
     *
     * @param event the mouse event to check for the double click.
     */
    public void mouseClicked (MouseEvent event) {
        if (event.getClickCount() > 2 || (getSelectedIndex() == lastClickEntry
                && event.getWhen() - lastClickTime < org.biolegato.core.main.ProgramWindow.DOUBLE_CLICK_TIME)) {
            getInfo();
        }
        lastClickTime = event.getWhen();
        lastClickEntry = getSelectedIndex();
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mousePressed (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mouseReleased (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseEntered (MouseEvent event) {
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseExited (MouseEvent event) {
    }

    /**
     * Converts a row number into a Y co-ordinate
     *
     * @param r the row number
     * @return the corresponding Y value
     */
    public int row2Y (int r) {
        return (r * getFontMetrics(getFont()).getHeight());
    }

    /**
     * Converts a column number into a X co-ordinate
     *
     * @param c the column number
     * @return the corresponding X value
     */
    public int column2X (int c) {
        return (c * getFontMetrics(getFont()).charWidth('a'));
    }

    /**
     * Converts a Y co-ordinate into a row number
     *
     * @param Y the Y value
     * @return the corresponding row number
     */
    public int Y2Row (int Y) {
        return (Y / getFontMetrics(getFont()).getHeight());
    }

    /**
     * Converts a X co-ordinate into a column number
     *
     * @param X the X value
     * @return the corresponding column number
     */
    public int X2Column (int X) {
        return (X / getFontMetrics(getFont()).charWidth('a'));
    }

    /**
     * Copies content from the current Editable object.
     */
    public void copy() {
        program.setClipboard(program.getSeqDoc().getSequences(getSelectedIndices()));
    }

    /**
     * Cuts content from the current Editable object.
     */
    public void cut() {
        copy();
        program.getSeqDoc().removeSequences(getSelectedIndices());
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste() {
        int position = program.getSeqDoc().getLineCount();

        if (getSelectedIndex() >= 0) {
            position = getSelectedIndex() + 1;
        }
        program.getSeqDoc().addSequences(position, program.getClipboard());
    }

    /**
     * Selects all the sequences within the list.
     */
    public void selectAll () {
        setSelectionInterval(0, getModel().getSize());
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont (Font font) {
        super.setFont(font);
        setFixedCellHeight(getFontMetrics(font).getHeight());
        repaint();
    }

    /**
     * Adds an item to the GDEList's popup menu
     *
     * @param item the menu item to add.
     */
    public void addPopupMenuItem (JMenuItem item) {
        popup.add(item);
    }

    /**
     * Displays a window to edit the currently selected sequence's properties
     */
    public void getInfo() {
        if (getSelectedIndex() >= 0 && getSelectedIndex() < program.getSeqDoc().getLineCount()) {
            new GDESequenceWindow(getSelectedIndex(), program, program.getSeqDoc().getSequence(getSelectedIndex()));
        }
    }
    
    /**
     * Used to control list display (when focused the list is colourful)
     *
     * @param e the focus event object.
     */
    public void focusGained (FocusEvent e) {
        setBackground(new Color(255, 255, 240));
        setSelectionBackground(Color.BLUE);
        setSelectionForeground(Color.WHITE);
        repaint();
    }

    /**
     * Used to control list display (when not focused the list is greyed out)
     *
     * @param e the focus event object.
     */
    public void focusLost (FocusEvent e) {
        setBackground(Color.LIGHT_GRAY);
        setSelectionBackground(Color.WHITE);
        setSelectionForeground(Color.GRAY);
        repaint();
    }

    /**
     * Returns the current list model.
     *
     * @return the current sequence list model.
     */
    public SequenceListModel getListModel () {
        return (SequenceListModel) super.getModel();
    }
}
