/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.colour;

import org.biolegato.gdesupport.canvas.BLTextArea;
import org.biolegato.gdesupport.canvas.selections.LineSelection;
import java.awt.Color;
import java.awt.Graphics;
import org.biolegato.core.data.sequence.Sequence;

/**
 * The superclass of all colour maps.
 *
 * Colour maps are used to provide different colour schemas to canvases.
 * To implement a new schema, create a subclass that implements all of the methods
 * you wish to override of the default schema.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public abstract class ColourMap {

    /**
     * The current default foreground colour of normal unselected text
     */
    protected Color foreground = Color.BLACK;
    /**
     * The current default background colour of normal unselected text
     */
    protected Color background = Color.WHITE;
    /**
     * The current default foreground colour of normal selected text
     */
    protected Color foregroundSelect = Color.WHITE;
    /**
     * The current default background colour of normal selected text
     */
    protected Color backgroundSelect = Color.BLUE;
    
////////////////////////////////////
//********************************//
//* STANDARD CONTRASTING COLOURS *//
//********************************//
////////////////////////////////////
    /**
     * Standard colour #1
     */
    public static final Color colour1 = new Color(255, 0, 0);           // Red
    /**
     * Standard colour #2
     */
    public static final Color colour2 = new Color(240, 180, 20);        // Orange
    /**
     * Standard colour #3
     */
    public static final Color colour3 = new Color(0, 0, 255);           // Blue
    /**
     * Standard colour #4
     */
    public static final Color colour4 = new Color(0, 0, 0);		// Black
    /**
     * Standard colour #5
     */
    public static final Color colour5 = new Color(0, 200, 0);           // Green
    /**
     * Standard colour #6
     */
    public static final Color colour6 = new Color(255, 0, 160);         // Pink
    /**
     * Standard colour #7
     */
    public static final Color colour7 = new Color(0, 160, 200);         // Tirquoise
    /**
     * Standard colour #8
     */
    public static final Color colour8 = new Color(0, 127, 255);         // Cyan
    /**
     * Standard colour #9
     */
    public static final Color colour9 = new Color(105, 139, 30);        // Olive 2
    /**
     * Standard colour #10
     */
    public static final Color colour10 = new Color(128, 0, 255);        // Purple
    /**
     * Standard colour #11
     */
    public static final Color colour11 = new Color(125, 158, 192);      // Blue-grey
    /**
     * Standard colour #12
     */
    public static final Color colour12 = new Color(205, 205, 0);        // Dark yellow
    /**
     * Standard colour #13
     */
    public static final Color colour13 = new Color(139, 10, 80);        // Deep pink
    /**
     * Standard colour #14
     */
    public static final Color colour14 = new Color(233, 116, 81);       // Burnt sienna
    /**
     * Standard colour #15
     */
    public static final Color colour15 = new Color(0, 180, 127);        // Medium tirquoise
    /**
     * Standard colour #16
     */
    public static final Color colour16 = new Color(127, 230, 0);        // Charteuse (med)
    /**
     * Standard neutral colour
     */
    public static final Color neutralColour = new Color(64, 64, 64);    // MediumGrey

    /**
     * The abstract constructor for colour maps.
     */
    public ColourMap () {
    }

    /**
     * Draws a sequence mapping the appropriate colours to each character rendered.
     *
     * This class will appropriately call regularDrawString and selectDrawString
     * to handle all of the actual drawing.  This allows anyone who wishes to create
     * their own colour map to have to only create a method that handles characters
     * they already know are selected or unselected (i.e., all subclasses should only
     * have to implement/override regularDrawString and selectDrawString for drawing
     * functionality.
     *
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param seq the sequence to draw.
     * @param intersect the text that is currently selected
     */
    public void drawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, Sequence seq, LineSelection intersect) {
        int startSelect = -1;
        int endSelect = -1;
        String string = seq.getField("sequence").toString();
        Sequence first = null;
        Sequence selected = null;
        Sequence tail = null;

        if (intersect != null &&  ! intersect.isEmpty()) {
            // get the selection start position
            startSelect = Math.max(0, intersect.getStart());
            endSelect = Math.min(intersect.getEnd(), string.length() - 1);
        }

        if (startSelect < string.length() && endSelect >= 0) {
            // create the sequences for each draw method
            first = (Sequence) seq.clone();
            first.setField("sequence", string.substring(0, startSelect));
            selected = (Sequence) seq.clone();
            selected.setField("sequence", string.substring(startSelect, endSelect));
            tail = (Sequence) seq.clone();
            tail.setField("sequence", string.substring(endSelect));

            // draw the sequences
            regularDrawString(ta, gfx, xstart, ystart, first);
            selectDrawString(ta, gfx, xstart + Math.round(ta.columnSize() * startSelect), ystart, selected);
            regularDrawString(ta, gfx, xstart + Math.round(ta.columnSize() * endSelect), ystart, tail);
        } else {
            regularDrawString(ta, gfx, xstart, ystart, seq);
        }
    }

    /**
     * Draws an unselected sequence while mapping the appropriate colours to each character rendered.
     *
     * The sequence passed to this function need not be modified, as all of the math and string
     * manipulation required for handling selected text is done in the drawString function.
     *
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param seq the sequence to draw.
     */
    public abstract void regularDrawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, Sequence seq);

    /**
     * Draws a selected sequence while mapping the appropriate colours to each character rendered.
     *
     * The sequence passed to this function need not be modified, as all of the math and string
     * manipulation required for handling selected text is done in the drawString function.
     * 
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param seq the sequence to draw.
     */
    public void selectDrawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, Sequence seq) {
        drawString(ta, gfx, xstart, ystart, seq.getField("sequence").toString(), foregroundSelect, backgroundSelect);
    }

    /**
     * This is a function to help ensure proper spacing.
     * All of the drawing should be done through this function.
     *
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param string the string to draw.
     * @param foreground the foreground colour of the string.
     * @param background the background colour of the string.
     */
    protected void drawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, String string, Color foreground, Color background) {
        int xstartMod = 0;                          // the amount to offset xstart so each printed character is centered.
        char[] drawArray = string.toCharArray();    // the current character to draw
        
        gfx.setColor(background);
        gfx.fillRect(xstart, ystart, string.length() * ta.columnSize(), ta.rowSize());
        gfx.setFont(ta.getFont());
        gfx.setColor(foreground);
        for (int column = 0; column < drawArray.length; column++) {
            xstartMod = (ta.columnSize() - ta.getFontMetrics(ta.getFont()).charWidth(drawArray[column])) / 2;
            gfx.drawChars( drawArray, column, 1, xstart + xstartMod, ystart + ta.rowSize() - 5);
            xstart += ta.columnSize();
        }
    }


    /**
     * Returns the current default foreground colour for painting portions of the canvas without text.
     *
     * @return the background colour.
     */
    public Color getForeground () {
        return background;
    }

    /**
     * Changes the current default foreground colour for painting portions of the canvas without text.
     *
     * @param c the new background colour.
     */
    public void setForeground (Color c) {
        background = c;
    }

    /**
     * Returns the current default background colour for painting portions of the canvas without text.
     *
     * @return the background colour.
     */
    public Color getBackground () {
        return background;
    }

    /**
     * Changes the current default background colour for painting portions of the canvas without text.
     *
     * @param c the new background colour.
     */
    public void setBackground (Color c) {
        background = c;
    }

}
