package org.biolegato.gdesupport.menu;

/*
 * GDEMenu.java
 *
 * Created on November 9, 2007, 10:18 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import org.biolegato.ezmenu.EZMenuType;
import org.biolegato.ezmenu.RunWindow;
import org.biolegato.ezmenu.variables.GDETempFile;
import org.biolegato.ezmenu.variables.GDEVariable;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.File;
import java.util.HashMap;
import java.util.LinkedList;
import javax.swing.JMenuItem;
import org.biolegato.core.main.ProgramWindow;
import org.biolegato.core.plugintypes.DataFormat;

/**
 * GDEMenu is used to read and parse GDE menu files into into BioLegato
 * 
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEMenu extends EZMenuType {
    /**
     * Reads in the default .GDEMenus files.
     *
     * @param program the instance of BioLegato to read the menu into.
     */
    public static void loadMenu (ProgramWindow program) {
        // check the settings before reading in the menus
        if (! "true".equalsIgnoreCase(program.getProperty("GDE.menu.disable"))) {
            // read in menus from the GDE directory
            if (System.getenv("GDE_HELP_DIR") != null) {
                readGDEMenuFile(program, new File(System.getenv("GDE_HELP_DIR") + File.separator + ".GDEmenus"));
            } else {
                org.biolegato.core.main.BLMain.warning("GDE_HELP_DIR not set!", "GDEMenu plugin");
            }

            // read in menus from the current working directory
            readGDEMenuFile(program, new File(".GDEmenus"));

            try {
                // read in menus from the home directory
                if (System.getProperty("user.home") != null) {
                    readGDEMenuFile(program, new File(System.getProperty("user.home") + File.separator + ".GDEmenus"));
                } else {
                    org.biolegato.core.main.BLMain.warning("HOME not set!", "GDEMenu plugin");
                }
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * This function loads a GDE formatted menu file into the program.
     *
     * @param program the instance of BioLegato to load the menu into.
     * @param file the file to read.
     */
    public static void readGDEMenuFile (final ProgramWindow program, File file) {
        String line = "";						    // the current line in the file
        String menuName = "File";					    // the current menu
        String fname = "";						    // The current field name being parsed
        String fvalue = "";						    // The current field value
        String itemName = "";						    // The current menu item's name
        BufferedReader reader = null;					    // the BufferedReader for reading the GDE menu
        JMenuItem jmitemp = null;					    // temporary vairble for adding action listeners to the JMenuItem
        RunWindow currentItem = null;					    // the current menu item being processed
        LinkedList<String> orderChoices = new LinkedList<String>();         // used for default values
        HashMap<String, String> choices = new HashMap<String, String>();    // Used for storing variable choices
        HashMap<String, Object> infile = new HashMap<String, Object>();	    // The data for the current argument
        HashMap<String, Object> outfile = new HashMap<String, Object>();    // The data for the current argument
        HashMap<String, Object> arg = new HashMap<String, Object>();	    // The data for the current argument

        if (file.exists() && file.isFile() && file.canRead()) {
            try {
                // open the file
                reader = new BufferedReader(new FileReader(file));
                do {
                    // read the next line
                    fname = "";
                    fvalue = "";
                    line = reader.readLine();
                    
                    // skip comments and blank lines and make sure that there is a :
                    if ( line != null && ! "".equals(line) && line.indexOf('#') != 0 && line.indexOf(':') >= 0) {
                        // the first parameter is the field name
                        // the second parameter is the field data
                        fname = line.substring(0, line.indexOf(':')).trim().toLowerCase();
                        fvalue = line.substring(line.indexOf(':') + 1).trim();
                    }   

//////////////////////////////////////////////////
//**********************************************//
//* add any pending arguments to the menu item *//
//**********************************************//
//////////////////////////////////////////////////
                    if (currentItem != null) {
                        if ((line == null || "item".equals(fname) || "arg".equals(fname) || "in".equals(fname) || "menu".equals(fname)) && infile.containsKey("name")) {
                            infile.put("window", program);
                            infile.put("input", Boolean.TRUE);
                            currentItem.addVariable(new GDETempFile(infile));
                            infile.clear();
                        }
                        if ((line == null || "item".equals(fname) || "arg".equals(fname) || "out".equals(fname) || "menu".equals(fname)) && outfile.containsKey("name")) {
                            outfile.put("window", program);
                            outfile.put("output", Boolean.TRUE);
                            currentItem.addVariable(new GDETempFile(outfile));
                            outfile.clear();
                        }
                        if ((line == null || "item".equals(fname) || "arg".equals(fname) || "menu".equals(fname)) && arg.containsKey("name")) {
                            arg.put("window", program);
                            arg.put("choices", choices);
                            arg.put("choiceorder", orderChoices);
                            // this is necessary before doing any other processing
                            if (arg.containsKey("name") && arg.containsKey("type")) {
                                currentItem.addVariable(GDEVariable.createVariable(arg));
                            }
                            orderChoices.clear();
                            choices.clear();
                            arg.clear();
                        }
                    }

                    // only do processing if the line is not null
                    if (!"".equals(fname)) {
/////////////////////////////////
//*****************************//
//* create a new menu heading *//
//*****************************//
/////////////////////////////////
                        if ("menu".equals(fname)) {
                            menuName = fvalue;
                            program.addMenuHeading(menuName);
                        }
//////////////////////////////
//**************************//
//* create a new menu item *//
//**************************//
//////////////////////////////
                        else if ("item".equals(fname)) {
                            itemName = fvalue;
                            currentItem = new RunWindow(program, itemName);
                            jmitemp = new JMenuItem(itemName);
                            jmitemp.addActionListener(currentItem);
                            program.addMenuItem(menuName, jmitemp);
                        } else if (currentItem != null) {

////////////////////////////////////
//********************************//
//* do menu item data processing *//
//********************************//
////////////////////////////////////
// this is optional for improving the usability of the menu item
                            if ("itemmeta".equals(fname) && fvalue.length() >= 1) {
                                // generate the mnemonic for the menu item
                                // this method allows only mnemonics of length 1
                                // (menu mnemonics in java are all of length 1, except complex ones involving shift/ctrl/etc.)
                                if ((fvalue.charAt(0) >= 'a' && fvalue.charAt(0) <= 'z') || (fvalue.charAt(0) >= 'A' && fvalue.charAt(0) <= 'Z')) {
                                    //currentItem.putValue(javax.swing.Action.MNEMONIC_KEY, new Integer(javax.swing.KeyStroke.getKeyStroke(fieldValue.charAt(0)).getKeyCode()));
                                }
                            } else if ("itemlabel".equals(fname)) {
                            } else if ("itemopen".equals(fname)) {
                            } else if ("itemhelp".equals(fname)) {
                                final String helpCommand = program.getProperty("GDE.help.viewer") + " " + program.getProperty("GDE.help.path") + File.separator + fvalue;
                                // obtain the help command for the menu item
                                currentItem.addButton(
                                        new RunWindow.ActionThread("Help", new Runnable() {

                                    public void run () {
                                        try {
                                            program.shellCommand(helpCommand, "");
                                        } catch (Throwable e) {
                                            e.printStackTrace();
                                        }
                                    }

                                }));

                            } else if ("itemmethod".equals(fname)) {
                                // get the command corresponding to the menu item
                                currentItem.setCommand(fvalue);
                            } ////////////////////////////////////////
                            //************************************//
                            //* do argument parameter processing *//
                            //************************************//
                            ////////////////////////////////////////
                            // this is used to allow for parameter passing for running commands
                            else if ("arg".equals(fname)) {
                                // create a new argument parameter
                                arg.put("name", fvalue);
                            } else if ("argtype".equals(fname)) {
                                // changes the argument type
                                arg.put("type", fvalue);
                            } else if ("arglabel".equals(fname)) {
                                // changes the description of the command
                                arg.put("description", fvalue);
                            } else if ("argmin".equals(fname)) {
                                // changes the minimum value for the argument
                                // this is applicable only to number choosers
                                try {
                                    arg.put("minimum", Integer.parseInt(fvalue));
                                } catch (NumberFormatException nfe) {
                                    org.biolegato.core.main.BLMain.error("Invalid GDE field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                                }
                            } else if ("argmax".equals(fname)) {
                                // changes the maximum value for the argument
                                // this is applicable only to number choosers
                                try {
                                    arg.put("maximum", Integer.parseInt(fvalue));
                                } catch (NumberFormatException nfe) {
                                    org.biolegato.core.main.BLMain.error("Invalid GDE field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                                }
                            } else if ("argvalue".equals(fname) || "argtext".equals(fname)) {
                                // changes the default value for the argument
                                arg.put("default", fvalue);
                            } else if ("argchoice".equals(fname)) {
                                if (fvalue.indexOf(':') >= 0) {
                                    // adds a new choice for the argument
                                    // this is applicable only to selection widgets
                                    choices.put(fvalue.substring(0, fvalue.indexOf(':')),
                                                fvalue.substring(fvalue.indexOf(':') + 1));
                                    orderChoices.add(fvalue.substring(0, fvalue.indexOf(':')));
                                } else {
                                    choices.put(fvalue, "");
                                    orderChoices.add(fvalue);
                                    org.biolegato.core.main.BLMain.warning("Badly formed argchoice field: " + fvalue, "GDEMenu");
                                }
                            } ///////////////////////////////////
                            //*******************************//
                            //* do I/O parameter processing *//
                            //*******************************//
                            ///////////////////////////////////
                            //***********************
                            //* Handle Input fields *
                            //***********************
                            else if ("in".equals(fname)) {
                                infile.put("name", fvalue);
                            } else if ("informat".equals(fname)) {
                                infile.put("format", DataFormat.getFormat(fvalue));
                            } else if ("inmask".equals(fname)) {
                                infile.put("mask", Boolean.TRUE);
                            } else if ("insave".equals(fname)) {
                                infile.put("save", Boolean.TRUE);
                            } //************************
                            //* Handle Output fields *
                            //************************
                            else if ("out".equals(fname)) {
                                outfile.put("name", fvalue);
                            } else if ("outformat".equals(fname)) {
                                outfile.put("format", DataFormat.getFormat(fvalue));
                            } else if ("outsave".equals(fname)) {
                                outfile.put("save", Boolean.TRUE);
                            } else if ("outoverwrite".equals(fname)) {
                                outfile.put("overwrite", Boolean.TRUE);
                            } else {
                                org.biolegato.core.main.BLMain.error("Invalid GDE field (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                            }
                        } else {
                            org.biolegato.core.main.BLMain.error("Invalid GDE location or field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                        }
                    }
                } while (line != null);

                // close the file
                reader.close();
            } catch (java.io.IOException e) {
                e.printStackTrace();
            }
        }
    }
}