//=====================================================================
// File:    ScoreFunction.java
// Class:   ScoreFunction
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * This class represents functions used to score a bin. For example, 
 * a function could be designed so that the bin was labeled with
 * A's and B's for a segregating population. The function should also
 * be able to give additional information once all of the lanes have been
 * scored, such as the number scored in a certain way. However, the overall
 * information should always be applicable to the case being handled.
 * Note: the first option should be the number of levels expected by
 * the function + 1. This corresponds to the number of different labels,
 * which is a more intuitive value for the user.
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public abstract class ScoreFunction extends Operation implements Cloneable
{
  /**
   * Should provide overall information about a bin scored with
   * this method. The labels will contain how each lane was scored and
   * the peaks will be the ones of interest in the bin. The string will
   * contain overall information, such as the number scored each way or
   * means and standard deviations.
   *
   * @param labels  the list of how each lane was scored.
   * @param peaks   all of the peaks of interest in the bin.
   *
   * @see ScoreFunction#score
   */
  public abstract String[] getInfo(String[] labels, DataList peaks);

  /**
   * Gives a label for a lane with the specified cutoff and the specified
   * peaks. For example, with a population of two parents, A and B,
   * the score method would return either an "A" or a "B" depending on
   * how the peaks related to the cutoff. If parent A was above the cutoff,
   * an A should be returned. Note that this case is only an example, and
   * many other possiblities exist.
   *
   * @param cutoff   the cutoff to be used when scoring
   * @param peaks    the peaks in the region of interest
   */
  public abstract String score(Cutoff cutoff, DataList peaks);

  /**
   * Gives the possible values that <code>score</code> will return.
   * These could differ based on the number of levels in the cutoff
   *
   * @param numLevels  the number of levels that the score function needs
   *                   to accomodate.
   */
  public abstract String[] getChoices(int numLevels);

  /**
   * Changes the score method so that the scoring is opposite of the
   * normal scoring, for example, A becomes B and B becomes A.
   */
  public abstract void invert();

 /**
   * Gives an object where every field has been copied from this object
   * to the new object.
   *
   * @return a copy of this object.
   */
  public Object clone()
  {
    try
      {
	return super.clone();
      }
    catch(CloneNotSupportedException e)
      {
	System.err.println("Could not clone Bin Score Function. " + 
			   e.getMessage());
      }
    return null;
  }

  /**
   * Writes all of the information this class needs to store in order
   * to be recreated. This will be used for things like storing the
   * class in a file. Therefore, the write should output enough information
   * so that <code>read</code> can recreate the essential properties of this
   * class.
   *
   * @param out  the destination to write the data to.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void write(DataOutputStream out) throws IOException;

  /**
   * Reads in the properties of this class from the specified input stream.
   * The stream data should have been created by <code>write</code>. This
   * will retrieve this classes state from the input stream. All of the
   * current data in this class will be replaced by the data from the 
   * stream.
   *
   * @param in  the input stream with the data for the class.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void read(DataInputStream in) throws IOException;
}
