//=====================================================================
// File:    SizeStandard.java
// Class:   SizeStandard
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

/**
 * This class is used to define a size standard. As far as this program is
 * concerned, a size standard only contains the size in bp of the expected
 * values. For example, something like: 50, 75, 100, 150, etc... The
 * program can then use these to tell which values read in from a file
 * are valid points. Size standards are read in from the file "standards.cfg"
 * and stored in this class.
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public class SizeStandard extends Operation
{
  private DataList values;

  /**
   * Create a new size standard defintion with the specified name.
   *
   * @param name  the name of the size standard.
   */
  SizeStandard(String name)
  {
    this.name = name;
    values = new DataList();
  }

  /**
   * Gives the name of this size standard.
   *
   * @return the name.
   */
  public String getName()
  {
    return name;
  }

  /**
   * Tells whether or not the specified value is included in this size
   * standard definition.
   * 
   * @param value  the size in bp
   *
   * @return <code>true</code> if it contains the value.
   */
  public boolean contains(double value)
  {
    return (values.find(value).location != -1);
  }

  /**
   * Adds the specified value to the defintion.
   *
   * @param value  the value to add, in bp
   */
  public void add(double value)
  {
    values.addData(new Peak(value, 0, 0));
  }

  /**
   * Defined to satisfy abstract parent class, but not really applicable in
   * this case.
   */
  public String getDescription()
  {
    return null;
  }

  /**
   * Defined to satisfy abstract parent class, but not really applicable in
   * this case.
   */
  public String getHelpFile()
  {
    return null;
  }

  /**
   * Defined to satisfy abstract parent class, but not really applicable in
   * this case.
   */
  public Option[] getOptions()
  {
    return null;
  }

  /**
   * Defined to satisfy abstract parent class, but not really applicable in
   * this case.
   */
  public void setOptions(Option[] opts)
  {
  }
  
  /**
   * Returns the number of elements conained within the datalist
   */
  public int getSize()
  {
	  return values.size();
  }
  
  /**
   * Returns the location of a peak at a given index
   */
  public double getPeakLocation(int peakIndex)
  {
	  return ((Peak)values.dataAt(peakIndex)).getLocation();
  }
}
