//=====================================================================
// File:    ButtonBar.java
// Class:   ButtonBar
// Package: AFLPgui
//
// Author:  James J. Benham
// Date:    August 11, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPgui;

import java.awt.Button;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import AFLPcore.ProgOptions;


/**
 * This class extends the bar class and adds four buttons to a bar.
 * The buttons have the standard icons for "new", "open", "save", and
 * "print." However, the functionality must be added by another class.
 * Objects can register themselves as <code>ActionListeners</code> by
 * calling the <code>sendActionEventsTo</code> method. This class
 * also provides a way to retrieve an image from a file.
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 11, 1998
 */

public class ButtonBar extends Bar
{
  private static String RELATIVE_PATH = "";
  private static String NEW_FILE   = "new.gif";
  private static String OPEN_FILE  = "open.gif";
  private static String SAVE_FILE  = "save.gif";
  private static String PRINT_FILE = "print.gif";

  protected static int BUTTON_WIDTH  = 23;
  protected static int BUTTON_HEIGHT = 22;
  protected static int HORZ_INSET    =  5;
  protected static int HORZ_SPACE    =  0;
  protected static int VERT_INSET    =  5;

  private ImgButton newButton;
  private ImgButton openButton;
  private ImgButton saveButton;
  private ImgButton printButton;

  private int buttonCount;
  private int adjustments;

  /**
   * Create a new button bar with both top and bottom border lines.
   */
  public ButtonBar()
  {
    this(true, true);
  }

  /**
   * Create a new button bar with the specified top and bottom borders.
   *
   * @param drawTop specifies if the top seperating line should be drawn.
   *   <code>true</code> for the line to show up.
   * @param drawBottom specifies if the bottom line should be drawn.
   */
  public ButtonBar(boolean drawTop, boolean drawBottom)
  {
    // set the borders for the bar
    setBorders(drawTop, drawBottom);

    buttonCount = 4;
    adjustments = 0;

    Toolkit tk = Toolkit.getDefaultToolkit();

    Image tempImg = tk.getImage(ProgOptions.homePath + RELATIVE_PATH
				+ NEW_FILE);
    newButton = new ImgButton(tempImg);

    tempImg = tk.getImage(ProgOptions.homePath + RELATIVE_PATH + OPEN_FILE);
    openButton = new ImgButton(tempImg);

    tempImg = tk.getImage(ProgOptions.homePath + RELATIVE_PATH + SAVE_FILE);
    saveButton = new ImgButton(tempImg);

    tempImg = tk.getImage(ProgOptions.homePath + RELATIVE_PATH + PRINT_FILE);
    printButton = new ImgButton(tempImg);

    setLayout(null);
    add(newButton);
    newButton.setBounds(HORZ_INSET, VERT_INSET, BUTTON_WIDTH, BUTTON_HEIGHT);
    add(openButton);
    openButton.setBounds(HORZ_INSET + HORZ_SPACE + BUTTON_WIDTH, VERT_INSET,
			 BUTTON_WIDTH, BUTTON_HEIGHT);
    add(saveButton);
    saveButton.setBounds(HORZ_INSET + 2*(HORZ_SPACE + BUTTON_WIDTH),
			 VERT_INSET, BUTTON_WIDTH, BUTTON_HEIGHT);
    add(printButton);
    printButton.setBounds(HORZ_INSET + 3*(HORZ_SPACE + BUTTON_WIDTH),
			  VERT_INSET, BUTTON_WIDTH, BUTTON_HEIGHT);
  }

  /**
   * Gives the location of the first space not occupied by a button.
   *
   * @return  the position of the right edge of the rightmost button.
   */
  public int getFreeHorzPos()
  {
    return (buttonCount*(HORZ_SPACE+BUTTON_WIDTH) + HORZ_INSET + adjustments);
  }

  /**
   * Get an image from the specified file. The image should be a GIF or PNG.
   *
   * @param filename  the full name, including path, of the desired image.
   *
   * @return   an image object, which will contain the picture once it
   *           has been loaded. Note: this method can return before the
   *           image is actually retrieved into the <code>Image</code> obect.
   */
  public static Image retrieveImage(String filename)
  {
    Toolkit tk = Toolkit.getDefaultToolkit();
    return tk.getImage(filename);
  }

  /**
   * Sets the specified <code>target</code> so that it recieves action
   * events from the four buttons.
   *
   * @param target  an object that wants to handle action events from the
   *                button.
   */
  public void sendActionEventsTo(ActionListener target)
  {
    newButton.addActionListener(target);
    openButton.addActionListener(target);
    saveButton.addActionListener(target);
    printButton.addActionListener(target);
  }

  /**
   * Gives the "new" button on the bar.
   */
  public Button getNewButton()
  {
    return newButton;
  }

  /**
   * Gives the "open" button on the bar.
   */
  public Button getOpenButton()
  {
    return openButton;
  }

  /**
   * Gives the "save" button on the bar.
   */
  public Button getSaveButton()
  {
    return saveButton;
  }

  /**
   * Gives the "print" button on the bar.
   */
  public Button getPrintButton()
  {
    return printButton;
  }
}
