#!/bin/csh

#Version  12/ 3/2007
# Run discrete data distance programs as a command
#Synopsis: restdist.csh infile model transratio sitelen method bseed replicates percent\
#             tconmeth power subrep global negbranch outgroup jumble jseed numjum\
#             termout printdata outfile treefile

#Convert arguments to variables
set INFILE        = $1
set MODEL         = $2
set TRANSRATIO    = $3
set SITELEN       = $4
set METHOD        = $5
set BSEED         = $6
set REPLICATES    = $7
set PERCENT       = $8
set TCONMETH      = $9
set POWER         = $10
set SUBREP        = $11
set GLOBAL        = $12
set NEGBRANCH     = $13
set OUTGROUP      = $14
set JUMBLE        = $15
set JSEED         = $16
set NUMJUM        = $17
set TERMOUT       = $18
set PRINTDATA     = $19
set OUTFILE       = $20
set TREEFILE      = $21

# RM_CMD - command to be used for removing files and directories
if (-e /usr/bin/rm) then
   set RM_CMD = /usr/bin/rm
else
   if (-e /bin/rm) then
      set RM_CMD = /bin/rm
   else
      set RM_CMD = rm
   endif
endif


# Make a temporary directory in which to run the program
set TEMPDIR = RESTDIST.$$
mkdir $TEMPDIR
cp $INFILE $TEMPDIR/infile.temp
cd $TEMPDIR

#----------------- generate keyboard input to send to RESTDIST -----

echo "Discrete Data Distance Matrix Phylogeny Methods" > MSGFILE
echo "" >> MSGFILE
echo "---------------------  DISTANCE MATRIX ---------------------" >> MSGFILE
echo "" >> MSGFILE

# Choose method for constructing distance matrix
switch ($MODEL)
  case "S": 
     echo "Markers represent alleles at polymorphic sites" >> MSGFILE
     echo "Distance matrix constructed using modified Nei/Li model" >> MSGFILE
    breaksw
  case "s": 
     echo "Markers represent alleles at polymorphic sites" >> MSGFILE
     echo "Distance matrix constructed using original Nei/Li model" >> MSGFILE
     echo n >> MATRIXPARAMFILE
    breaksw
  case "R": 
     echo "Markers represent restriction fragments" >> MSGFILE
     echo "Modified Nei/Li change model" >> MSGFILE
     echo "Distance matrix constructed modified Nei/Li model" >> MSGFILE
     echo r >> MATRIXPARAMFILE
    breaksw       
  case "r": 
     echo "Markers represent restriction fragments" >> MSGFILE
     echo "Distance matrix constructed original Nei/Li model" >> MSGFILE
     echo r >> MATRIXPARAMFILE
     echo n >> MATRIXPARAMFILE
    breaksw   
  default: 
     echo "Markers represent alleles at polymorphic sites" >> MSGFILE
     echo "Distance matrix constructed using modified Nei/Li model" >> MSGFILE
    breaksw
endsw

if (($MODEL == 'S') || ($MODEL == 'R')) then
   # Transition/transversion ratio
   if ($TRANSRATIO != 2.0) then
      echo t >> MATRIXPARAMFILE
      echo $TRANSRATIO >> MATRIXPARAMFILE
      endif
   echo "Transition/Transversion ratio = "$TRANSRATIO >> MSGFILE
endif

# set site length
if ($SITELEN > 0) then
   echo s >> PARAMFILE
   echo $SITELEN >> PARAMFILE
endif

#----------------- generated resampled datasets, if specified  -----
# Choose resampling method

# Note: For bootstrapping, as of Phylip 3.63, restdist cannot utilize the weights file generated by
# seqboot. Therefore, complete datasets must be generated by seqboot, which
# in turn will be used to create distance matrices.


# Random seeds, odd, of the form 4n + 1
@ tempbseed = ( ( ( $BSEED / 4 ) * 4 ) + 1 )

switch ($METHOD)
  case "n":
    echo " " >> MSGFILE
    cp infile.temp infile
    breaksw
  case "b":
    echo RESAMPLING\: Bootstrap,  $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    if ( $PERCENT < 100 ) echo 'Partial Resampling: ' $PERCENT  'percent of sites sampled' >> MSGFILE
    #cp infile.temp infile
    seqboot.csh infile.temp r $tempbseed b $REPLICATES $PERCENT 1 no 0 infile
    echo m >> MATRIXPARAMFILE
    echo $REPLICATES >> MATRIXPARAMFILE
    breaksw
  case "d" :
    echo RESAMPLING\: Delete-half Jacknifing, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    if ( $PERCENT < 100 ) echo 'Partial Resampling: ' $PERCENT  'percent of sites sampled' >> MSGFILE
    #cp infile.temp infile
    seqboot.csh infile.temp r $tempbseed d $REPLICATES $PERCENT 1 no 0 infile
    echo m >> MATRIXPARAMFILE
    echo $REPLICATES >> MATRIXPARAMFILE
    breaksw
  case "ps":
    echo RESAMPLING\: Permute species for each character, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp r $tempbseed ps $REPLICATES $PERCENT 1 no 0 infile
    echo m >> MATRIXPARAMFILE
    echo $REPLICATES >> MATRIXPARAMFILE
    breaksw
  case "po":
    echo RESAMPLING\: Permute character order, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp r $tempbseed po $REPLICATES $PERCENT 1 no 0 infile
    echo m >> MATRIXPARAMFILE
    echo $REPLICATES >> MATRIXPARAMFILE
    breaksw
  case "pw":
    echo RESAMPLING\: Permute within species, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp r $tempbseed pw $REPLICATES $PERCENT 1 no 0 infile
    echo m >> MATRIXPARAMFILE
    echo $REPLICATES >> MATRIXPARAMFILE
    breaksw        
endsw

#accept current settings and do the analysis
echo y  >> MATRIXPARAMFILE


#----------------- Run RESTDIST   -----

nice restdist < MATRIXPARAMFILE
mv outfile infile

echo "---------------------  CONSTRUCTING TREE(S) ---------------------" >> MSGFILE
echo "" >> MSGFILE

#----------------- generate keyboard input to send to distance tree program -----

# Choose method for constructing distance matrix
switch ($TCONMETH)
  case "F": # FITCH, Fitch-Margoliash method
    set PROGRAM = fitch
    breaksw
  case "f": # FITCH, Minimum evolution
    set PROGRAM = fitch
    echo d >> TREEPARAMFILE
    breaksw
  case "K": # KITSCH, Fitch-Margoliash method
    set PROGRAM = kitsch
    breaksw
  case "k": # KITSCH, Minimum evolution method
    set PROGRAM = kitsch
    echo d >> TREEPARAMFILE
    breaksw
  case "N": # Neighbor-joining
    set PROGRAM = neighbor
    breaksw
  case "U": # Neighbor-joining
    set PROGRAM = neighbor
    echo n >> TREEPARAMFILE
    breaksw
  default : # FITCH, Fitch-Margoliash method
    set PROGRAM = fitch
    breaksw
endsw

switch ($METHOD)
  case "n":
    breaksw
  case "b":
    echo m >> TREEPARAMFILE
#    echo w >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "d" :
    echo m >> TREEPARAMFILE
#    echo w >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "ps":
    echo m >> TREEPARAMFILE
#    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "po":
    echo m >> TREEPARAMFILE
#    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "pw":
    echo m >> TREEPARAMFILE
#    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw        
endsw


# Jumble - When multiple datasets are analyzed, DNAPARS automatically
# jumbles, and prompts for a random number seed for jumbling. Othersise,
# jumbling must be explicitly set.

# Random seed, odd, of the form 4n + 1
@ tempjseed = ( ( ( $JSEED / 4 ) * 4 ) + 1 )

if ($METHOD == "n") then
   if ($JUMBLE == J) then 
      echo JUMBLING SEQUENCE ORDER $NUMJUM ITERATIONS, SEED\=$tempjseed >> MSGFILE
      echo  j >> TREEPARAMFILE
      echo $tempjseed  >> TREEPARAMFILE
      echo $NUMJUM >> TREEPARAMFILE
   endif
else
   echo JUMBLING SEQUENCE ORDER $NUMJUM ITERATIONS, SEED\=$tempjseed >> MSGFILE
   echo $tempjseed  >> TREEPARAMFILE
   echo $NUMJUM >> TREEPARAMFILE
endif

# Subreplicates
if ($SUBREP == 'n') echo 's' >> TREEPARAMFILE

# Global rearrangements
if ($GLOBAL == 'y') echo 'g' >> TREEPARAMFILE
	
# Negative branch lengths
if ($NEGBRANCH == 'y') echo '-' >> TREEPARAMFILE

# Outgroup
set FIRSTLINE = `head -1 infile`
set NUMSEQ = $FIRSTLINE[1]
if (($OUTGROUP > 1) && ($OUTGROUP <= $NUMSEQ)) then
   echo o >> TREEPARAMFILE
   echo $OUTGROUP >> TREEPARAMFILE
else
   set OUTGROUP = 1
endif


# Should sequence data be printed?
if ($PRINTDATA ==  y) echo 1 >> TREEPARAMFILE

# When resampling, turn off printing trees to outfile
if (($METHOD == 'b') || ($METHOD == 'd') || ($METHOD == 'ps') || ($METHOD == 'po') || ($METHOD == 'pw')) then
   echo 3 >> TREEPARAMFILE
endif

#accept current settings and do the analysis
echo y  >> TREEPARAMFILE

#----------------- Run FITCH, KITCH or NEIGHBOR   -----

echo `time nice +8 $PROGRAM < TREEPARAMFILE > $TERMOUT` > TIMEFILE
echo Execution times on `hostname`\: `cat TIMEFILE` >> outfile


#----------- Return results to calling directory----------------
# When using resampling, filter the treefile through
# consense to generate an unrooted consensus tree.
if (($METHOD == 'b') || ($METHOD == 'd') || ($METHOD == 'ps') || ($METHOD == 'po') || ($METHOD == 'pw')) then
	echo " " >> outfile
	echo '-------------------------------------------' >> outfile
	consense.csh outtree e 1 $OUTGROUP n outfile.consense ../$TREEFILE
	cat MSGFILE outfile outfile.consense  > ../$OUTFILE
        echo '' >> ../$OUTFILE
        echo '' >> ../$OUTFILE
        echo '>>>> THIS TREEFILE IS A CONSENSUS TREE' >> ../$OUTFILE
        echo '>>>> TO GENERATE BRANCH LENGTHS' >> ../$OUTFILE
        echo '>>>> USE TREE FILE AS INPUT FOR DNAML OR OTHER PROGRAM' >> ../$OUTFILE
        echo '>>>> USING THE USERTREE OPTION' >> ../$OUTFILE
else
        mv outtree ../$TREEFILE
	cat MSGFILE outfile > ../$OUTFILE
endif

cd ..
$RM_CMD -rf $TEMPDIR

echo $PROGRAM completed.





