package embl.ebi.trace;

import java.awt.*;
import java.io.*;
/**
A subclass for handling SCF format DNA sequence trace file parsing.
@see Chromatogram
@see ABIChromatogram
*/

public class SCFChromatogram extends Chromatogram {

	// public variables
	public static final int MagicNum = (((((int)'.'<<8)+(int)'s'<<8)+(int)'c'<<8)+(int)'f');
	
	// Internal temporary parsing variables
	private int sampleOffset, basesOffset, commentsOffset, commentsSize, sampleSize;
	private int bytesRead, baseNum, codeSet, sampleNum;
	private String versionString;
	private float versionNum;
	private DataInputStream filterIN;

	public SCFChromatogram(InputStream in) throws IOException {
		super();
		read(in);
	}

	public void write(OutputStream OUT) {
		// yet to be written
	}

	public void read(InputStream in) throws IOException{
		filterIN = new DataInputStream(in);
		bytesRead = 0;
		filterIN.skipBytes(4);
		sampleNum = filterIN.readInt();
		sampleOffset = filterIN.readInt();
		baseNum = filterIN.readInt();
		filterIN.skip(8);
		basesOffset = filterIN.readInt();
		commentsSize = filterIN.readInt();
		commentsOffset = filterIN.readInt();
		int i;
		versionString = "";
		for (i=0;i<4;i++) {
			versionString += (char) filterIN.readUnsignedByte();
			}
		versionNum = Float.valueOf(versionString).floatValue();
		sampleSize = filterIN.readInt();
		codeSet = filterIN.readInt();
		bytesRead = 48;
		
		// determine order of file components and then read them
		if ((basesOffset < sampleOffset) && (basesOffset < commentsOffset)) {
			readBases();
			if (sampleOffset < commentsOffset) {
				readSample();
				readComments();
			}
			else {
				readComments();
				readSample();
			}
		}
		else if (sampleOffset < commentsOffset) {
			readSample();
			if (basesOffset < commentsOffset) {
				readBases();
				readComments();
			}
			else {
				readComments();
				readBases();
			}
		}
		else {
			readComments();
			if (sampleOffset < basesOffset) {
				readSample();
				readBases();
			}
			else {
				readBases();
				readSample();
			}
		}
	}	

	private void readBases() throws IOException{
		filterIN.skipBytes(basesOffset - bytesRead);
		int i;
		basePosition = new int[baseNum];
		quality = new byte[sampleNum];
		base = new char[baseNum];
		if (versionNum < 3.0) {
			for (i=0;i<baseNum;i++) {
				basePosition[i] = filterIN.readInt();
				filterIN.skipBytes(4);
				base[i] = (char) filterIN.readUnsignedByte();
				filterIN.skipBytes(3);
			}
		}
		else {
			for (i=0;i<baseNum;i++) {
				basePosition[i] = filterIN.readInt();
			}
			filterIN.skipBytes(4 * baseNum);
			for (i=0;i<baseNum;i++) {
				base[i] = (char) filterIN.readUnsignedByte();
			}
			filterIN.skipBytes(3 * baseNum);
		}
		bytesRead = basesOffset + baseNum * 12;
	}

	private void readSample() throws IOException {
		System.err.println("sampleSize= " + sampleSize + " versionNum= " + versionNum + " sampleNum= " + sampleNum);
		filterIN.skipBytes(sampleOffset - bytesRead);
		A = new int[sampleNum];
		C = new int[sampleNum];
		G = new int[sampleNum];
		T = new int[sampleNum];
		int i;
		if ((sampleSize == 2) && (versionNum < 3.0)) {
		for (i=0;i<sampleNum;i++) {
			A[i] = filterIN.readUnsignedShort();
			C[i] = filterIN.readUnsignedShort();
			G[i] = filterIN.readUnsignedShort();
			T[i] = filterIN.readUnsignedShort();
		}
	}
	else if ((sampleSize == 1) && (versionNum < 3.0)) {
		for (i=0;i<sampleNum;i++) {
			A[i] = filterIN.readUnsignedByte();
			C[i] = filterIN.readUnsignedByte();
			G[i] = filterIN.readUnsignedByte();
			T[i] = filterIN.readUnsignedByte();
		}
	}
	else if ((sampleSize == 2) && (versionNum >= 3.0)) {
		for (i=0;i<sampleNum;i++) {
			A[i] = filterIN.readShort();
		}
		A = deltaDelta(A);
		for (i=0;i<sampleNum;i++) {
			C[i] = filterIN.readShort();
		}
		C = deltaDelta(C);
		for (i=0;i<sampleNum;i++) {
			G[i] = filterIN.readShort();
		}
		G = deltaDelta(G);
		for (i=0;i<sampleNum;i++) {
			T[i] = filterIN.readShort();
		}
		T = deltaDelta(T);
		
	}
	else if ((sampleSize == 1) && (versionNum >= 3.0)) {
		for (i=0;i<sampleNum;i++) {
			A[i] = filterIN.readByte();
		}
		A = deltaDelta(A);
		for (i=0;i<sampleNum;i++) {
			C[i] = filterIN.readByte();
		}
		C = deltaDelta(C);
		for (i=0;i<sampleNum;i++) {
			G[i] = filterIN.readByte();
		}
		G = deltaDelta(G);
		for (i=0;i<sampleNum;i++) {
			T[i] = filterIN.readByte();
			}
		T = deltaDelta(T);
		}
		bytesRead = sampleOffset + (sampleNum * sampleSize * 4);
		}
	
	private void readComments() throws IOException {
		int i;
		char[] cArray = new char[commentsSize];
		filterIN.skipBytes(commentsOffset - bytesRead);
		for (i=0;i<commentsSize;i++) {
			cArray[i] = (char) filterIN.readUnsignedByte();
		}
		comments = String.valueOf(cArray);
		bytesRead = commentsOffset + commentsSize;
		}
	
	private int[] deltaDelta(int[] theArray)	{
		int	i;
		int	mask;
		int	pSample;

		/**
		The deltas are calculated assuming appropriate overflow arithmatic
		for the sample size. I fake the over/underflows here to get one
		routine that works for both one and two byte smaples.
		*/
		if (sampleSize == 1){
			// mask = Byte.MAX_VALUE;
			mask = 255;
			}
		else	{
			// mask = Short.MAX_VALUE;
			mask = 65535;
			}
		pSample = 0;
		for (i=0;i<theArray.length-1;i++) {
			theArray[i] = (theArray[i] + pSample) & mask;
			pSample = theArray[i];
		}
		pSample = 0;
		for (i=0;i<theArray.length-1;i++) {
			theArray[i] = (theArray[i] + pSample) & mask;
			pSample = theArray[i];
		}
		return theArray;
	}
				
	public static boolean isSCF(File aFile) throws FileNotFoundException, IOException{
			FileInputStream fileIN = new FileInputStream(aFile);
			DataInputStream filtIN = new DataInputStream(fileIN);
			if (filtIN.readInt() == MagicNum) return true;
			else return false;
	}
	
	public int getCodeSet() {
		return codeSet;
	}
	
	public int getBaseNumber() {
		return (int) base.length;
	}

	public int getTraceLength() {
		return sampleNum;
	}
	
	public String getComments() {
		return comments;
	}
	
	public String getVersion() {
		return versionString;
	}

}
