/* The main server routine

*/
package embl.ebi.trace;
import java.io.*;
import java.util.*;
import org.omg.CORBA.*;
import org.omg.CosNaming.*;

/**
TraceServer is the topmost class in an object hierarchy used to send DNA sequence
chromatograms (traces) down CORBA connections to clients for analysis or display.

1) The TraceServer class has a main method which gets the name of a configuration
file from the command line.
2) A CORBA BOA connection is established
3) Naming service is discovered and an object reference to it is
written to a specified file where it can be retrieved via http.
4) From the config file, one TraceStore object is started for
each separate trace set.

The methods promised via IDL are implemented in a separate class.
@see TraceStoreImpl

@author Jeremy D. Parsons
*/
/*************************************************************************

Copyright (c) 1998 by European Bioinformatics Institute, Cambridge, UK.


You are hereby granted permission to use, copy, modify, and distribute 
this software and its documentation for any purpose and without fee,
provided that the above copyright notices appear in all copies and that 
both the above copyright notices and this permission notice appear in 
supporting documentation, and that neither the name of the European 
Molecular Biology Laboratory nor European Bioinformatics Institute are 
used in advertising or publicity pertaining to distribution of the 
software without specific written prior permission. You HEREBY AGREE 
THAT SINCE THE FOREGOING LICENCE IS GRANTED WITHOUT CHARGE, IT IS 
REASONABLE THAT THIS SOFTWARE IS PROVIDED "AS IS" AND THAT THE 
EUROPEAN MOLECULAR BIOLOGY LABORATORY AND EUROPEAN BIOINFORMATICS 
INSTITUTE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF 
SATISFACTORY QUALITY OR FITNESS FOR A PARTICULAR PURPOSE, AND IN NO 
EVENT SHALL THE EUROPEAN MOLECULAR BIOLOGY LABORATORY OR EUROPEAN 
BIOINFORMATICS INSTITUTE BE LIABLE FOR ANY SPECIAL, INDIRECT OR 
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF 
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT 
(INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN 
CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

*************************************************************************/


public class TraceServer{

	public static final char	COMMENT_LINE_CHAR = '#';
	public static final String	FIELD_SEPARATOR_CHARS = ":";

	// public final static String USAGE_LINE = "Usage: TraceServer Config_filename IOR_filename(in WWW visible place)";
	public final static String USAGE_LINE = "Usage: TraceServer Config_filename -ORBnaming `cat naming.ior`";

	/**
	@param Config_filename is the name in the local filesystem of a file
	containing the names, locations, and descriptions of the various
	different trace sets to be served.
	@param IOR_filename Once the name server is located, its object reference needs to
	be published to enable access to all the different TraceStores
	*/
	public static void main(String[] args){
		NamingContext rootContext = null;

		if (args.length < 2){
			System.err.println(USAGE_LINE);
			System.exit(1);
			}
		String configFilename = args[0];
		// String IORFilename = args[1];

		try	{
			ORB orb = org.omg.CORBA.ORB.init(args, System.getProperties());
			BOA boa = orb.BOA_init(args, System.getProperties());
			try	{
				org.omg.CORBA.Object objRef = orb.resolve_initial_references("NameService");
				rootContext = NamingContextHelper.narrow(objRef);
				}
			catch (Exception ex) {
				System.err.println("Cannot connect to the naming service");
				System.exit (-1);
				}

			// Create naming context for TraceStore IORs and read configFilename
			NamingContext storeContext = publishStores(configFilename, rootContext, boa);
			/*
			try     {
				String ref = orb.object_to_string (rootContext);
				FileWriter fileW = new FileWriter(IORFilename);
				fileW.write(ref, 0, ref.length());
				fileW.close();
				}
			catch (IOException e) {
				System.err.println ("Can't save IOR reference to file " + e);
				System.exit (-1);
				}
			*/

			// boa.obj_is_ready((org.omg.CORBA.Object)server1);
			System.err.println("The server is running.");

			boa.impl_is_ready(null);
			}
		catch (Exception Ex){
			System.err.println(Ex);
			}
		}
	
	/**
	Read the configuration file, instantiate all the TraceStores, and
	leave an object reference for each one in the nameService.
	*/
	static NamingContext publishStores(String configFilename, NamingContext rootContext, BOA boa){
		NamingContext	storeContext = null;
		NameComponent	storeNameComponent[];
		
		// First establish the naming context: build an array of components and add them all at once
		try	{
			storeContext = rootContext.bind_new_context(TraceServerNaming.NAME_PATH
);
			}
		catch (org.omg.CosNaming.NamingContextPackage.AlreadyBound boundEx){
			try	{
				org.omg.CORBA.Object objRef= rootContext.resolve(TraceServerNaming.NAME_PATH);
				storeContext = NamingContextHelper.narrow(objRef);
				}
			catch (Exception Ex){
				System.err.println ("Problem resolving supposedly known name " + Ex);
				System.exit(-1);
				}
			}
		catch (Exception Ex){
			System.err.println ("Problem creating naming context: " + Ex);
			System.exit(-1);
			}

		// Open the config file and parse the non-comment lines
		FileReader	fileR;
		BufferedReader	fileBR;
		StringTokenizer tokenizer;
		String 		line;
		String		storeName;
		String		baseDir;
		String		description;
		TraceStoreImpl	store;

		try	{
			fileR = new FileReader(configFilename);
			fileBR = new BufferedReader(fileR);
			line = fileBR.readLine();
			while (line != null){
				// Ignore comment lines: anything beginning with COMMENT_LINE_CHAR
				// Use short cut boolean evaluation to save trouble
				while ((line != null) && ((line.length() == 0) || (line.charAt(0) == COMMENT_LINE_CHAR))){
					line = fileBR.readLine();
					}
				
				if (line == null) break;
				tokenizer = new StringTokenizer(line, FIELD_SEPARATOR_CHARS);
				if (tokenizer.countTokens() != 3){
					System.err.println ("Unable to parse the line below because it had "
						+ tokenizer.countTokens()
						+ " fields\n"
						+ line);
					System.exit(-1);
					}
				storeName = tokenizer.nextToken();
				baseDir = tokenizer.nextToken();
				description = tokenizer.nextToken();
	
				System.out.println("Adding database: " + storeName);
				store = null;
				try	{
					store = new TraceStoreImpl(storeName, baseDir, description);
					}
				catch (Exception ex) {
					System.err.println ("Cannot create a new trace server for: "
						+ storeName + ":"
						+ baseDir + ":"
						+ description
						);
					System.exit(-1);
					}
				//boa.obj_is_ready((org.omg.CORBA.Object)store);
				// Add the new store to the name server context
				storeNameComponent = new NameComponent[1];
				// System.err.println("Got here1" + storeName);
				storeNameComponent[0] = new NameComponent(storeName, TraceServerNaming.CONTENT_TYPE);
				try	{
					storeContext.rebind(storeNameComponent, store);
					}
				catch (Exception ex){
					System.err.println ("Unable to bind server with name: " + storeName + " Reason: " + ex);
					System.exit(-1);
					}

				line = fileBR.readLine();
				}
			
			}
		catch (IOException IoEx){
			System.err.println(IoEx);
			}
		
		return(storeContext);
		}
}

