package embl.ebi.trace;

import java.awt.*;
import java.awt.image.*;
import java.applet.*;

/**
This canvas draws typical views of DNA sequence traces with a top base call
line and a base position scale. The base calls are now drawn first to allow
them to be seen when scrolling. Base peaks should not intrude into the
base call lines. There is no vertical scaling provided, only horizontal
scrolling.
@see ChromatogramViewer
*/
public class ChromatogramCanvas extends Canvas {

	private Chromatogram	chromData; 
	private int[]		A,C,G,T;
	private int		maxValue;
	private int		offset = 0;
	private Image offScreenImage;
	private Graphics offScreenGC;
	private Color colorA = Color.green;
	private Color colorC = Color.blue;
	private Color colorG = Color.black;
	private Color colorT = Color.red;
	private Color colorU = Color.red;
	private Color colorUnknown = Color.gray;

	private	int canvasHeight = 100;
	private	int canvasWidth = 100;
	private	float scale = 100.0F;
	private	float externalScale = 1.0F;
	
	public ChromatogramCanvas() {
		super();
		setBackground(Color.white);
		emptyTrace();
		}

	public void emptyTrace(){
		chromData = null;
		A = C = G = T = null;
		repaint();
		}

	public void loadTrace(Chromatogram chromatogram){
		this.chromData = chromatogram;
		A = chromData.getATrace();
		C = chromData.getCTrace();
		G = chromData.getGTrace();
		T = chromData.getTTrace();
		maxValue = 0;
		for (int i = 0; i < chromData.getTraceLength(); i++) {
			if (A[i] > maxValue) maxValue = A[i];
			if (C[i] > maxValue) maxValue = C[i];
			if (G[i] > maxValue) maxValue = G[i];
			if (T[i] > maxValue) maxValue = T[i];
			}
		repaint();
		}
	
	public void setOffset(int newOffset){
		this.offset = newOffset;
		}
	
	public int getOffset(){
		return(this.offset);
		}

	public void setMagnifier(float magnifier) {
		this.externalScale = magnifier;
		}

	public float getMagnifier() {
		return(this.externalScale);
		}

	public void update(Graphics grph){
		if ((offScreenImage == null)
			|| (offScreenImage.getWidth(this) != getSize().width)
			&& (offScreenImage.getHeight(this) != getSize().height))
			{
			offScreenImage = createImage(getSize().width, getSize().height);
			offScreenGC = offScreenImage.getGraphics();
			}
		offScreenGC.setColor(getBackground());
		offScreenGC.fillRect(0, 0, getSize().width, getSize().height);
		paint(offScreenGC);
		grph.drawImage(offScreenImage, 0, 0, this);
		}
	
	public void paint(Graphics grph) {	
		if (chromData == null){ // Nothing to do without a trace !
			return;
			}

		FontMetrics	baseFontMetrics = getFontMetrics(grph.getFont());
		int topMargin = baseFontMetrics.getMaxDescent() + baseFontMetrics.getMaxAscent();

		canvasHeight = getSize().height;
		canvasWidth = getSize().width;
		scale = ((float)maxValue / (float)(canvasHeight - topMargin)) ;
		int i = 0; 
		
		// Draw in the base calls and the base position scale
		while ((i < chromData.getBaseNumber()) && (chromData.getBasePosition(i) < offset)){
			i++;
			}
		int	rightMostSample = offset + canvasWidth;
		while ((i < chromData.getBaseNumber()) && (chromData.getBasePosition(i) < rightMostSample)){
			switch (chromData.getBase(i)) {
				case 'A':
				grph.setColor(colorA);
				break;
				case 'C':
				grph.setColor(colorC);
				break;
				case 'G':
				grph.setColor(colorG);
				break;
				case 'T':
				grph.setColor(colorT);
				break;
				case 'U':
				grph.setColor(colorU);
				break;
				default:
				grph.setColor(colorUnknown);
				}
			int cmid = baseFontMetrics.charWidth(chromData.getBase(i)) / 2;
			grph.drawString(String.valueOf(chromData.getBase(i)),chromData.getBasePosition(i) - offset - cmid, 15);
			if ((i+1) % 10 == 0) {
				grph.setColor(Color.black);
				int nmid = baseFontMetrics.stringWidth(String.valueOf(i+1)) / 2;
				grph.drawString(String.valueOf(i+1),chromData.getBasePosition(i) - offset - nmid,25);
				}
			i++;
			}

		int[]	xPoints = new int[canvasWidth];
		int[]	yPoints = new int[canvasWidth];
		for (int j = 0; j < canvasWidth; j++){
			xPoints[j] = j;
			}
		fillYPoints(yPoints, A, offset);
		grph.setColor(colorA);
		grph.drawPolyline(xPoints, yPoints, xPoints.length);
		fillYPoints(yPoints, C, offset);
		grph.setColor(colorC);
		grph.drawPolyline(xPoints, yPoints, xPoints.length);
		fillYPoints(yPoints, G, offset);
		grph.setColor(colorG);
		grph.drawPolyline(xPoints, yPoints, xPoints.length);
		fillYPoints(yPoints, T, offset);
		grph.setColor(colorT);
		grph.drawPolyline(xPoints, yPoints, xPoints.length);
		}
		
	void fillYPoints(int[] yPoints, int[] intensities, int offset){
		int count = yPoints.length;
		for (int i = 0; i < count; i++){
			yPoints[i] = canvasHeight - (int)((float)(intensities[i + offset]) * externalScale / scale);
			}
		}
}
