package embl.ebi.trace;

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.zip.*;
import org.omg.CORBA.*;
import org.omg.CosNaming.*;     // Uses Name Service


/********************************************************************

Copyright (c) 1998 by European Bioinformatics Institute, Cambridge, UK.


You are hereby granted permission to use, copy, modify, and distribute 
this software and its documentation for any purpose and without fee,
provided that the above copyright notices appear in all copies and that 
both the above copyright notices and this permission notice appear in 
supporting documentation, and that neither the name of the European 
Molecular Biology Laboratory nor European Bioinformatics Institute are 
used in advertising or publicity pertaining to distribution of the 
software without specific written prior permission. You HEREBY AGREE 
THAT SINCE THE FOREGOING LICENCE IS GRANTED WITHOUT CHARGE, IT IS 
REASONABLE THAT THIS SOFTWARE IS PROVIDED "AS IS" AND THAT THE 
EUROPEAN MOLECULAR BIOLOGY LABORATORY AND EUROPEAN BIOINFORMATICS 
INSTITUTE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD TO 
THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED WARRANTIES OF 
SATISFACTORY QUALITY OR FITNESS FOR A PARTICULAR PURPOSE, AND IN NO 
EVENT SHALL THE EUROPEAN MOLECULAR BIOLOGY LABORATORY OR EUROPEAN 
BIOINFORMATICS INSTITUTE BE LIABLE FOR ANY SPECIAL, INDIRECT OR 
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF 
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT 
(INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN 
CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

*******************************************************************/
/** 
This class implements much of the CORBA and event message conversions
listening to requests from the GUI, looking up CORBA object 
references, and calling methods on the remote trace servers to
pull down trace data.
This component can be instantiated directly from within some other
object with only one constructor parameter: the naming service
root context used by the remote server.
The TraceRequestListener interface can be used to pass requests
for trace display.
This version displays a separate window showing which trace servers
are known at the remote naming service. 
@see CorbaChromatogramApplet
@see ChromatogramViewer
@see CorbaChromatogramWrapper
@see TraceStore
@author Eugen Buehler buehler@agave.humgen.upenn.edu
@author Jeremy D. Parsons jparsons@ebi.ac.uk
*/

public class CorbaChromatogramWrapper extends Panel implements TraceRequestListener{
	private ChromatogramViewer	myView;
	public Frame			serverSummaryFrame = null;
	private TraceStore		currentServer = null;

	private java.util.Hashtable	knownServers = new java.util.Hashtable();
	private NamingContext		rootContext = null;
	private NamingContext		traceDirContext = null;
	
	public CorbaChromatogramWrapper(){
		// Super call here
		// Needed empty constructor for Beans
		}
		
	public void setRootContext(NamingContext rootContext){
		myView = new ChromatogramViewer();
		this.setLayout(new BorderLayout());
		this.add("Center", myView);

		// From the root make a specific path to the server "directory"/context
		// Also a quick test that the path exists
		try	{
			org.omg.CORBA.Object objRef = rootContext.resolve(TraceServerNaming.NAME_PATH);
			traceDirContext = NamingContextHelper.narrow(objRef);
			}
		catch (Exception ex){
			UserComms.warn("Could not get to the trace server names");
			if (UserComms.applet == null){
				System.exit(-1);
				}
			return;
			}

		// Create the list of known servers
		if (serverSummaryFrame != null){
			serverSummaryFrame.dispose();
			}
		serverSummaryFrame = new Frame("DB Servers Available from Here");
		serverSummaryFrame.add("North", new Label("DB name"));
		serverSummaryFrame.add("Center", createKnownList());
		serverSummaryFrame.pack();
		WindowListener winListener = new WindowAdapter(){
			public void windowClosing(WindowEvent e) {serverSummaryFrame.dispose();}
                        };
                serverSummaryFrame.addWindowListener(winListener);

		serverSummaryFrame.setVisible(true);
		}

	public NamingContext getRootContext(NamingContext rootContext){
		 return(rootContext);
		}
	
	/**
	Ask the naming service for a list of all the known trace servers.
	In the future, this method should also ask each of the trace
	servers for their desciptions.
	*/

	List createKnownList() {
		// Get just the names of all currently known trace servers and display in a List
		List		serversList = new List(10);
		int		chunkSize = 100; // How many names per call
		int		foundCount;
		Binding[]	nameList = null;
		try	{
			BindingListHolder	tempHolder = new BindingListHolder();
			BindingIteratorHolder	nameIterator = new BindingIteratorHolder();
			traceDirContext.list(chunkSize, tempHolder, nameIterator);
			nameList = tempHolder.value;
			foundCount = nameList.length;
			if (foundCount >= chunkSize){
				UserComms.warn("Ignoring trace servers after first " + chunkSize);
				nameIterator.value.destroy();
				}
			for (int i = 0; i < foundCount; i++){
				if (nameList[i].binding_type == BindingType.nobject){
					// Found what should be a trace server
					NameComponent[] tempName = nameList[i].binding_name;
					try	{
						org.omg.CORBA.Object objRef = traceDirContext.resolve(tempName);
						knownServers.put(tempName[0].id, TraceStoreHelper.narrow(objRef));
						// Need to follow reference to get details but for now just name
						serversList.add(tempName[0].id);
						}
					catch (Exception ex){
						UserComms.warn("The database " + tempName[0].id + " is not available. : " + ex);
						}
					}
				}
			}
		catch (org.omg.CORBA.SystemException ex){
			UserComms.warn("CORBA problem: " + ex);
			}
		catch (Exception ex){
			UserComms.warn("Unknown problem : " + ex);
			}

		return(serversList);
		}

	public java.util.Hashtable getKnownServers(){
		return(knownServers);
		}

	/**
	@param The request event can be generated from below by the 
	GUI, or from above, by some other component that needs a trace
	displayed.
	*/
        public void traceRequested(TraceRequestEvent reqE){
                // Child control panel has asked for a new sequence
                reloadTrace(reqE.getStore(), reqE.getAccession());
                }

	void reloadTrace(String database, String traceName) {
		byte[]			givenByteArr = null;

		UserComms.inform("Requested " + database + ":" + traceName);

		// Which trace server has been requested ?
		if (knownServers.contains(database)) {
			currentServer = (TraceStore)knownServers.get(database);
			}
		else	{
			// Need to get the new object reference
			try	{
				NameComponent serverName[] = new NameComponent[1];
				serverName[0] = new NameComponent(database, TraceServerNaming.CONTENT_TYPE);
				org.omg.CORBA.Object objRef = traceDirContext.resolve(serverName);
				currentServer = TraceStoreHelper.narrow(objRef);
				}
			catch (org.omg.CORBA.SystemException ex){
				UserComms.warn("CORBA problem: " + ex);
				}
			catch (Exception ex){
				UserComms.warn("The database " + database + " is not available. : " + ex);
				}

			if (currentServer != null){
				knownServers.put(database, currentServer);
				}
			else	{
				return;
				}
			}

		try	{
			givenByteArr = currentServer.getGZIPFile(traceName);
			/* Spare testing code
			givenByteArr = currentServer.getFile(traceName);
			try	{
				FileOutputStream checkFile = new FileOutputStream("checkFile");
				checkFile.write(givenByteArr);
				}
			catch (IOException ex){
				UserComms.warn(ex + " Problem writing the checkfile");
				}
			*/
			}
		catch (InvalidID ex){
			UserComms.warn("Unknown name problem: " + ex);
			}
		catch (org.omg.CORBA.SystemException ex){
			UserComms.warn("CORBA problem: " + ex);
			}
	
		Chromatogram		chromData = null;
		ByteArrayInputStream	bufferIn;
		byte[]			unCompByteArr;

		UserComms.inform("Requested " + database + ":" + traceName);
		if (givenByteArr != null){
			try	{
				unCompByteArr = TraceUtils.gzipStream2bytes(new ByteArrayInputStream(givenByteArr));
				bufferIn = new ByteArrayInputStream(unCompByteArr);
				chromData = Chromatogram.createChromatogramSubType(bufferIn);
				bufferIn.close();
				}
			catch (IOException ex) {
				System.err.println("IO ERROR: " + ex);
				}

			// Get the trace display refreshed
			chromData.setDatabaseName(database);
			chromData.setAccession(traceName);
			myView.loadTrace(chromData);	
			}
		}
}	
