package embl.ebi.trace;
/**
  This class does all the implementation dependent name resolution
  to find the required EST trace

  Most of these methods will need to be overridden at other
  sites using different filing conventions, or DBMSs to store
  traces and qualities.
*/

import java.io.*;
import java.util.*;
import java.util.zip.*;

public class FindEST {
	/**
	Some pathname string manipulation routines
	Use standard UNIX filenames and let File class handle other architectures.
	
	This class will need to be overridden according to local file
	conventions/database access methods.
	*/

	// final static String ROOT_DIR = "/home1/watson/ftp/gsc1/est/";
	// final static String ROOT_DIR = "/home3/ftp/pub/gsc1/est/";
	public static String THIS_TRACE_SET_NAME = "WUGSC";
	public static String QUAL_EXTENSION = ".qual"; 
	public static String GZIP_EXTENSION = ".gz"; 
	public static String COMPRESS_EXTENSION = ".Z"; // Not usable easily by Java

	/**
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	static String name2path(String rootDir, String readName, String suffix) throws InvalidID {
		// Basic name checks
		nameCheck(readName);

		// Add a directory separator character if not already given 
		StringBuffer path = new StringBuffer(rootDir);
		if (! rootDir.endsWith(java.io.File.separator)){
			path.append(java.io.File.separatorChar);
			}
			
		// Add on the separate parts to find this EST in the local filesystem
		// yc17d12.s1 -> yc/yc17/yc17d12/yc17d12.s1
		path.append(readName.substring(0,2));
		path.append(java.io.File.separatorChar);
		path.append(readName.substring(0,4));
		path.append(java.io.File.separatorChar);
		path.append(readName.substring(0,7));
		path.append(java.io.File.separatorChar);
		path.append(readName);
		path.append(suffix);

		return (path.toString());
		}

	/**
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	static void nameCheck(String readName) throws InvalidID {
		
		if (readName.length() != 10){
			System.err.println("Incorrect number of characters: " + readName);
			throw(new InvalidID("Incorrect number of characters: " + readName));
			}
		if (readName.charAt(7) != '.'){
			System.err.println("The eigth character is not a '.': "+ readName);
			throw(new InvalidID("The eigth character is not a '.': "+ readName));
			}
		}
			
	/**
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	final static File name2TraceFile(String rootDir, String readName) throws InvalidID{
		/**
		This method makes some assumptions about file naming schemes.
		It tries to find the two commonly stored compressed trace
		files and complains if it is not gzip'ped, as Java only
		has built in zipfile support
		*/
		File	readFile;
		
		readFile = new File(name2path(rootDir, readName, GZIP_EXTENSION));
		if (readFile.exists()){
			if (readFile.canRead()){
				return(readFile);
				}
			else	{
				System.out.println("Unavailable read: " + readName);
				throw(new InvalidID("Unavailable read: " + readName));
				}
			}
		else	{
			readFile = new File(name2path(rootDir, readName, COMPRESS_EXTENSION));
			if (readFile.exists()){
				System.err.println("Only unusable old format found: " + readName);
				throw(new InvalidID("Only unusable old format found: " + readName));
				}
			else	{
				System.err.println("Unknown read: " + readName);
				throw(new InvalidID("Unknown read: " + readName));
				}
			}
		}

	/**
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	final static File name2QualFile(String rootDir, String readName)
		throws InvalidID {
		/**
		This method makes some assumptions about file naming schemes.
		The first implementation of the trace server was in Saint Louis
		where the quality info is kept separate from the traces. This
		may be a useful assumption for most places though this particular
		routine may not be needed if qualities are stored in a DB.
		*/
		File	readFile;
		
		readFile = new File(name2path(rootDir, readName, QUAL_EXTENSION));
		if (readFile.exists()){
			if (readFile.canRead()){
				return(readFile);
				}
			}
		System.err.println("Unavailable read: " + readName);
		throw(new InvalidID("Unavailable read: " + readName));
		}
}
