package embl.ebi.trace;
/**
  This class does all the implementation dependent running around
  the file systems to find the required EST trace, it then reads the trace
  into a byte array to hand back to the client. Most of these methods
  will need to be overridden at other sites using different filing
  conventions, or DBMS to store traces and qualities.

  The IDL has been extended to suport server side decompression
  of gzip files. And so requires JDK 1.1 :-)
*/

import java.io.*;
import java.util.*;
import java.text.DateFormat;

public class TraceStoreImpl extends _TraceStoreImplBase {
	String	rootDir = "";
	String	serverName = "";
	String	description = "";
	boolean	everythingWorking = true;
	int	methodAccessCount = 0;
	String	statusString = "";
	public	boolean getQualitiesFromTrace = false;
	public	boolean getSequenceFromTrace = true;

	public TraceStoreImpl() {
		}

	public TraceStoreImpl(String storeName, String baseDir, String description) {
		initialiseImplementation(storeName, baseDir, description);
		}

	public void initialiseImplementation(String storeName, String baseDir, String description) {
		this.rootDir = baseDir;
		this.serverName = storeName;
		this.description = description;
		DateFormat localDate = DateFormat.getInstance();
		localDate.setTimeZone(TimeZone.getDefault());
		statusString = "Started at " + localDate.format(new Date());
		}

	// Implement the general server attributes for status
	public boolean up(){
		return(everythingWorking);
		}

	public int methodCallCount(){
		return(methodAccessCount);
		}

	public String statusMessage(){
		return(statusString);
		}

	public String description(){
		return(description);
		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	*/
	public boolean exists(String accession) throws
		org.omg.CORBA.SystemException {
		// implement operation...
		methodAccessCount++;
		try	{
			FindEST.name2TraceFile(rootDir, accession);
			}
		catch (InvalidID ex){
			// If here, then it doesn't exist so return false
			return(false);
			}
		// If not caught then exists
		return(true);

		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	public java.lang.String getLibraryName(String accession) throws
		InvalidID,
		org.omg.CORBA.SystemException {
		throw new InvalidID(accession + " library name cannot be retrieved by this server implementation."); 
		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	@exception InvalidID thrown if the database or readname cannot be read. 

	To get to the sequence, it is necessary to parse the entire trace.
	*/
	public String getBases(java.lang.String accession) throws InvalidID, org.omg.CORBA.SystemException {
		methodAccessCount++;
		Chromatogram chromData = null;
		if (getSequenceFromTrace){
			try	{
				byte[] fileBytes = this.getFile(accession);
				ByteArrayInputStream bufferIn = new ByteArrayInputStream(fileBytes);
				chromData = Chromatogram.createChromatogramSubType(bufferIn);
				bufferIn.close();
				}
			catch (IOException ioExc){
				throw new InvalidID(accession + " basecalls cannot be retrieved by this server implementation.");
				}
			}
		else	{
			throw new InvalidID(accession + " separated basecalls cannot be retrieved by this server implementation."); 
			}

		return(chromData.getSequence());
		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	public byte[] getQualities(String accession) throws
		InvalidID,
		org.omg.CORBA.SystemException {
		/*
		It would be nice to allow user alteration in quality values, but assume a
		single, separate file for now. The File is expected to only contain
		ASCII, space-separated decimal integers in the range 0-127 (i.e. bytes)
		*/
		methodAccessCount++;
		if (getQualitiesFromTrace){
			throw new InvalidID(accession + " Qualities cannot be retrieved by this server implementation.");
			}
		else	{
			Vector tempQuals = new Vector();
			BufferedReader readFile;
			try	{
				readFile = new BufferedReader(new FileReader(FindEST.name2QualFile(rootDir, accession)));
				String line;
				StringTokenizer tokens;
				while((line = readFile.readLine())!= null){
					tokens = new StringTokenizer(line);
					while (tokens.hasMoreTokens()){
						try	{
							tempQuals.addElement(Byte.valueOf(tokens.nextToken()));
							}
						catch (NumberFormatException NumEx){
							System.out.println("Read qualities failed for: " + accession +
								" with line=\n" + line);
							throw new InvalidID("Read qualities failed for: " + accession);
							}
						}
					}	 
				}
			catch (IOException ioEx){
				System.out.println("Read qualities failed for: " + accession);
				throw new InvalidID("Read qualities failed for: " + accession);
				}
	
			// Now convert into an array of bytes
			int	qualCount = tempQuals.size();
			byte[]	quals = new byte[qualCount];
			for (int i = 0; i < qualCount; i++){
				quals[i] = ((Byte)tempQuals.elementAt(i)).byteValue();
				}
			return(quals);
			}
		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	public byte[] getGZIPFile(String accession) throws
		InvalidID,
		org.omg.CORBA.SystemException {
		// Read the file into an array of bytes and return

		methodAccessCount++;
		File readFile = FindEST.name2TraceFile(rootDir, accession);
		byte[] contents;
		FileInputStream fileIn;
	
		contents = new byte[(int)readFile.length()];
		try	{
			fileIn = new FileInputStream(readFile);
			fileIn.read(contents);	// Reads it all in !
			}
		catch (IOException ex){
			throw new InvalidID("Read failed for: " + accession);
			}

		return(contents);
		}

	/**
	@exception org.omg.CORBA.SystemException Usually only thrown if there is a communication failure
	@exception InvalidID thrown if the database or readname cannot be read. 
	*/
	public byte[] getFile(String accession) throws
		InvalidID,
		org.omg.CORBA.SystemException {
		// Read and gunzip the file into an array of bytes and return

		File readFile;
		byte[] contents;
	
		methodAccessCount++;
		try	{
			readFile = FindEST.name2TraceFile(rootDir, accession);
			contents = TraceUtils.gzipStream2bytes(new FileInputStream(readFile));
			}
		catch (IOException ex){
			System.err.println(ex + "gunzip read failed for: " + accession);
			throw new InvalidID(ex +"Read failed for: " + accession);
			}
			
		return(contents);
		}
}

