/*
 * DataFormat.java
 *
 * Created on January 30, 2008, 11:54 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.core.plugintypes;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.MappedByteBuffer;
import org.biolegato.core.plugins.PluginLoader;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.plugins.PluginWrapper;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

/**
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public abstract class DataFormat extends FileFilter {
	/**
	 * Creates a new instance of DataFormat
	 */
	public DataFormat () {
	}

	/**
	 * This function loads all of the available file formats into a JFileChooser.
	 *
         * @param filechooser the JFileChooser to load the formats into.
         * @param defaultFormat the default file format for the file chooser.
	 */
	public static void addFormats (JFileChooser filechooser, String defaultFormat) {
		DataFormat selected = null;	    // the defautl file format.
		DataFormat currentFormat = null;    // the current file format to test.

		// itterate through all file format subclasses
		for (PluginWrapper plugin : PluginLoader.getPlugins(DataFormat.class)) {
			// attempt to create a new instance of each plugin
			currentFormat = (DataFormat) plugin.create();
			if (currentFormat != null) {
				// add the file filter to the JFileChooser
				filechooser.addChoosableFileFilter(currentFormat);
				
				// if the format is the default format specified in the method call,
				// set it to be the default selected format for the JFileChooser.
				if (currentFormat.getName().equals(defaultFormat)) {
					selected = currentFormat;
				}
			}
		}
		
		// if the default file format was found, set the JFileChooser to have it selected.
		if (selected != null) {
			filechooser.setFileFilter(selected);
		}
	}

	/**
	 * Reads a file's contained sequences using the file filter's translation.
	 *
	 * @param file the file to writer to.
	 * @return the contents of the file
         * @throws IOException passes any exceptions from reading the file
	 */
	public Sequence[] readFile (File file) throws IOException {
            Sequence[] result = new Sequence[0];
            MappedByteBuffer stream = null;
            
            if (file != null && file.exists() && file.isFile() && file.length() > 0) {
                /*
		    possible faster method to read big files.
		    -----------------------------------------
		    stream = new FileInputStream(file.getPath()).getChannel().map(FileChannel.MapMode.READ_ONLY, 0, file.length());
		    result = translateFrom(new LineNumberReader(stream));
		 */
                result = translateFrom(new BufferedReader(new InputStreamReader(new BufferedInputStream(new FileInputStream(file)))));
            }
            return result;
	}

	/**
	 * Writes a list of sequences to a file.
	 *
	 * @param file the file to writer to.
	 * @param data the contents to writer.
	 * @throws IOException throws any exceptions that occur while writing to the file
	 */
	public void writeFile (File file, Sequence[] data) throws IOException {
		FileWriter writer = new FileWriter(file);   // the file writer object to write the file with.
                
		// translate and write the file.
                for (Sequence line : data) {
                    writer.write(translateTo(line));
                }
		
		// flush and close the file writer buffer.
		writer.flush();
		writer.close();
	}

	/**
	 * Obtains the name of the file format for File Chooser purposes
	 *
	 * @return the name of the file format
	 */
	public String getName () {
		return "";
	}

	/**
	 * Translates a string from the BioLegato internal format to the given file format.
	 *
	 * @param data the string to convert
	 * @return the resulting string
	 */
	public abstract String translateTo (Sequence data);

	/**
	 * Translates a string from the given file format into the BioLegato internal format.
	 *
	 * @param data the string to convert.
	 * @return the resulting string.
	 * @throws IOException any exeptions that occur while reading the stream are passed.
	 */
	public abstract Sequence[] translateFrom (java.io.BufferedReader data) throws IOException;

	/**
	 * Used to auto-detect Bio Legato formats
	 *
	 * @param test the string to test
	 * @return whether or not the format is correct
	 */
	public abstract boolean isFormat (java.io.Reader test);

	/**
	 * Finds the filetype that corresponds to the given hashname (null if not successful).
	 *
	 * @param hashname the hashname to search for.
	 * @return The result of the search (returns null on failure).
	 */
	public static DataFormat getFormat (String hashname) {
		DataFormat found = null;	    // the data format found (null if not found).
		DataFormat currentFormat = null;    // the current data format being tested

		// itterate through the list of plugins
		for (PluginWrapper plugin : PluginLoader.getPlugins(DataFormat.class)) {
			// create the file format plugin to test
			if (hashname != null && plugin != null) {
				currentFormat = (DataFormat) plugin.create();
				
				// break if successful match
				if (currentFormat != null && hashname.equalsIgnoreCase(currentFormat.getName())) {
					found = currentFormat;
					break;
				}
			}
		}
		return found;
	}

	/**
	 * Used to auto-detect and parse files into BioLegato sequence objects.
	 *
	 * @param file the file to read.
	 * @return The resulting sequences.
	 */
	public static Sequence[] auto (File file) {
            Sequence[] result = new Sequence[0];    // the resulting sequence set
            try {
		// autodetect the file.
                result = auto(new FileInputStream(file));
            } catch (IOException ioe) {
		// print an error message on failure
                org.biolegato.core.main.BLMain.error("Bad file: " + file.getAbsolutePath(), "DataFormat.auto");
            }
            return result;
        }
        
	/**
	 * Used to auto-detect and parse an input stream's data into BioLegato sequence objects.
	 *
	 * @param stream the stream to read.
	 * @return The resulting sequences.
	 */
	public static Sequence[] auto (InputStream stream) {
            return auto(new BufferedReader(new InputStreamReader(new BufferedInputStream(stream))));
        }
        
	/**
	 * Used to auto-detect and parse data into BioLegato sequence objects.
	 *
	 * @param data the string to convert
	 * @return The resulting sequences.
	 */
	private static Sequence[] auto (java.io.BufferedReader data) {
		Sequence[] result = new Sequence[0];	// the result of the file read.
		DataFormat currentFormat = null;	// the current format to test for a match.

		// itterate through all file formats
		for (PluginWrapper plugin : PluginLoader.getPlugins(DataFormat.class)) {
			if (data != null && plugin != null) {
                            try {
				// create an instance of the current format to use.
                                currentFormat = (DataFormat) plugin.create();
				
				// if the current format is a match, then translate using it.
                                if (currentFormat != null && currentFormat.isFormat(data)) {
                                    result = currentFormat.translateFrom(data);
                                    break;
                                }
                            } catch (IOException e) {
                            }
			}
		}
		
		// return the resulting sequences.
		return result;
	}
}
