/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.ezmenu;

import java.awt.Container;
import java.awt.Dimension;
import java.util.Hashtable;
import org.biolegato.ezmenu.variables.GDETempFile;
import org.biolegato.ezmenu.variables.GDEVariable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JScrollPane;
import org.biolegato.core.main.BLMain;

/**
 * The window used for setting options to run a menu command.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class RunWindow extends JDialog implements Runnable, ActionListener {
    /**
     * The files attached to the window.
     */
    private LinkedList<GDETempFile> files = new LinkedList<GDETempFile>();
    /**
     * The current running instance of BioLegato.
     */
    private BLMain program = null;
    /**
     * This variable stores the basic command string.
     * This basic command string will be used to determine the command to run,
     * and will be left unmodified after execution.
     */
    private String command = "";
    /**
     * The container to add variables to.
     */
    protected Container variablePane = new Box(BoxLayout.PAGE_AXIS);
    /**
     * The container to add buttons to.
     */
    protected Container buttonPane = new Box(BoxLayout.LINE_AXIS);
    /**
     * The list of variabes currently used by the window.  This is used for
     * creating the runtime command string.
     */
    protected Hashtable<String,GDEVariable> variables = new Hashtable<String,GDEVariable>();
    /**
     * Used in serialization.
     */
    private static final long serialVersionUID = 7526472295622777007L;

    /**
     * Creates a new instance of a run window.
     *
     * @param parent the BioLegato program instance to obtain information from.
     * @param name the name of the run window.
     */
    public RunWindow (BLMain parent, String name) {
        // call the superclass constructor and make the window modal
        super(parent, "BioLegato: " + name, true);

        // create the pane and make it scrollable
        setContentPane(new JScrollPane(new Box(BoxLayout.PAGE_AXIS) {
            /* used for serialization */
            private static final long serialVersionUID = 7526472295622777003L;


            {
                add(variablePane);
                add(buttonPane);
            }
        }));

        // clean up and display the window
        setMinimumSize(new Dimension(50,50));
        setSize(new Dimension(100,100));
        setLocationRelativeTo(parent);
        setVisible(false);

        // set the instance variable for the program window
        this.program = parent;

        // set the appropriate class variables
        addButton(new ActionThread("Run", this));
    }

	/**
	 * Adds a button to the window.
	 *
	 * @param run the AbstractAction used to create the button
	 */
	public void addButton (Action run) {
		buttonPane.add(new JButton(run));
		pack();
	}

	/**
	 * This function is used to add variables to the window.
	 *
	 * @param variable the variable to add to the window
	 */
	public void addVariable (GDEVariable variable) {
		Container panel = null;
		if (variable != null) {
			panel = variable.display();
			if (panel != null) {
				variablePane.add(panel);
			}
			pack();
		}
        if (variable instanceof GDETempFile) {
            files.add((GDETempFile) variable);
        } else {
            variables.put(variable.getName().toLowerCase(), variable);
        }
	}
    
    /**
     * Changes the command string.
     *
     * @param command the command string to change to.
     */
    public void setCommand (String command) {
        this.command = command;
    }
    
    /**
     * Used for running the command.
     * <p>
     *	This function gathers all of the parameter settings from the widgets
     *	in the window, then generates and executes the corresponding command
     *	string.
     * </p>
     */
    public void run () {
        String run = command;

        // hide the current window
        setVisible(false);
        
        run = replaceArguments(run);

        // execute the program and collect program output
        program.shellCommand(run, "");

        cleanupExecution();
    }

    /**
     * Class used for making abstract actions which launch threads.
     *
     * @author Graham Alvare
     * @author Brian Fristensky
     */
    public static class ActionThread extends AbstractAction {

        /**
         * The action to perform.
         */
        private Runnable run = null;
        /**
         * This constant is used for serialization
         */
        private static final long serialVersionUID = 7526472295622777001L;

        /**
         * Creates a new instance of ActionThread.
         *
         * @param name the name of the action
         * @param run the command for the thread to run
         */
        public ActionThread (String name, Runnable run) {
            super(name);
            this.run = run;
        }

        /**
         * Runs the command.
         *
         * @param e ignored by this function
         */
        public void actionPerformed (ActionEvent e) {
            new Thread(run).start();
        }

    }

    /**
     * Makes the window visible so all of the variables may be set.
     *
     * @param event this is currently ignored
     */
    public void actionPerformed (ActionEvent event) {
        setVisible(true);
    }

    /**
     * Replaces the variables in the command string with their corresponding values.
     *
     * @param run the command to do the replacements on.
     * @return the altered command string.
     */
    private String replaceArguments (String run) {
        int start = -1;
        int end = -1;
        String name = "";
        String value = "";

        // create the command string (NOTE: variable.toString().replaceAll(Pattern.quote("$"), "\\\\\\$") is used to prevent regex grouping (e.g. $0, etc.))
        while ((start = run.indexOf('$', start)) >= 0) {
            end = start + 1;
            while (end < run.length() && isDigit(run.charAt(end))) {
                end++;
            }
            name = run.substring(start + 1, end).toLowerCase();
            value = "";
            if (variables.containsKey(name)) {
                value = variables.get(name).getValue().toString();
                run = (start == 0 ? "" : run.substring(0, start)) + value
                        + (end >= run.length() ? "" : run.substring(end));
                start --;
            } else {
                start ++;
            }
        }
        
        for (GDETempFile file : files) {
	    // reset the start and positions
	    start = -1;
	    end = -1;
	    
	    // make sure name is valid
            if (file.getName() != null &&  ! file.getName().equals("")) {
                name = file.getName();
                value = file.getValue().toString();
		
		// itterate through and find each instance of the name to replace
                while ((start = run.indexOf(name, end + 1)) >= 0) {
                    end = start + name.length();
		    
		    // make sure that the name is not part of a bigger name
                    if ((start == 0 || notDigit(run.charAt(start - 1)))
                            && (end >= run.length() || notDigit(run.charAt(end)))) {
                        run = (start == 0 ? "" : run.substring(0, start)) + value
                                + (end >= run.length() ? "" : run.substring(end));
                        end = start - 1;
                    }
                }
            }
        }
        return run;
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is not a digit, otherwise false.
     */
    private boolean notDigit(char test) {
        return ((test < '0' || test > '9')
            && (test < 'a' || test > 'z')
            && (test < 'A' || test > 'Z'));
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is a digit, otherwise false.
     */
    private boolean isDigit(char test) {
        return ((test >= '0' && test <= '9')
            || (test >= 'a' && test <= 'z')
            || (test >= 'A' && test <= 'Z'));
    }

    /**
     * Cleans up any pending information after execution of the command.
     */
    private void cleanupExecution() {
        // releases all of the input files
        for (GDETempFile file : files) {
            file.close();
        }
    }
}
