package org.biolegato.gdesupport.canvas;

/*
 * BLGDECanvas.java
 *
 * Created on December 3, 2007, 1:56 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import org.biolegato.core.properties.PropertiesListener;
import org.biolegato.gdesupport.canvas.colour.GDECharColourMap;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import org.biolegato.core.main.BLMain;
import org.biolegato.core.plugintypes.DataCanvas;

/**
 * The GDE-style sequence-based canvas class
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDECanvas extends DataCanvas implements CursorListener, FocusListener, ModeListener, PropertiesListener {

    /**
     * Whether to use the name list or the canvas for selections
     */
    private boolean useNameList = false;
    /**
     * Used for selecting genes
     */
    private GDEList nameList = new GDEList(program);
    /**
     * The text area for data manipulation
     */
    private GDETextArea dataCollector = new GDETextArea(program);
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622776154L;
    /**
     * The box to store all split canvases
     */
    private Box canvasPane = new Box(BoxLayout.LINE_AXIS);
    /**
     * Reference pointer to self
     */
    private final GDECanvas canvasSelf = this;
    /**
     * Stores the row and column position status
     */
    private final JLabel status = new JLabel("Row: 1 Col: 1");
    /**
     * Stores the insertion mode status
     */
    private final JLabel insertStatus = new JLabel("     ");
    /**
     * Action for splitting the canvas
     */
    private final JMenuItem splitAction = new JMenuItem(new AbstractAction("Split") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
            canvasPane.add(new JScrollPane(new GDETextArea(program), JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS));
            dataCollector.addPopupMenuItem(joinAction);
        }

    });
    /**
     * A window used for selection sequences based on a name search.
     */
    public final SelectByNameWindow selectionWindow = new SelectByNameWindow(program, nameList);
    /**
     * The menu item "Select sequence by name"
     */
    public final JMenuItem selectByNameAction = new JMenuItem(new AbstractAction("Select sequence by name") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                selectionWindow.setVisible(true);
            }

        });
    /**
     * Action for joining split canvases
     */
    private final JMenuItem joinAction = new JMenuItem(new AbstractAction("Join") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
            System.out.println(((JMenuItem) e.getSource()).getParent().getClass());
            //canvasPane.remove(e.getSource());
            //dataCollector.removePopupMenuItem(joinAction);
        }

    });
    /**
     * The menu item "Cut"
     */
    private final JMenuItem cutAction = new JMenuItem(new AbstractAction("Cut") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.cut();
        }

    });
    /**
     * The menu item "Copy"
     */
    private final JMenuItem copyAction = new JMenuItem(new AbstractAction("Copy") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.copy();
        }

    });
    /**
     * The menu item "Paste"
     */
    private final JMenuItem pasteAction = new JMenuItem(new AbstractAction("Paste") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.paste();
        }

    });
    /**
     * The menu item "Change Case"
     */
    private final JMenuItem changeCaseAction = new JMenuItem(new AbstractAction("Paste") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.changeCase();
        }

    });
    /**
     * The menu item "Select All"
     */
    private final JMenuItem selectAllAction = new JMenuItem(new AbstractAction("Select All") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.selectAll();
        }

    });
    /**
     * Self-reference for inner classes.
     */
    public final GDECanvas gdeCanvasSelf = this;

    /**
     * Creates a new instance of BLGDECanvas
     *
     * @param program the BioLegato instance to associate the canvas with.
     */
    public GDECanvas (final BLMain program) {
        super(program, BoxLayout.PAGE_AXIS);

        // setup listeners
	dataCollector.addFocusListener(this);
        dataCollector.addCursorListener(this);
        dataCollector.addModeListener(this);
	nameList.addFocusListener(this);

        program.addPropertiesListener("font.size", this);
        program.addPropertiesListener("font.bold", this);
        
        // add items to textarea popup menu
        //dataCollector.addPopupMenuItem(splitAction);

        // set up textarea colours
        dataCollector.setColourMap(new GDECharColourMap());
        dataCollector.setBackground(new Color(255, 255, 240));

        // create the scroll panes
        JScrollPane dataPane = new JScrollPane(dataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        JScrollPane listPane = new JScrollPane(nameList, JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);

        dataPane.setPreferredSize(new Dimension(200, 150));
        listPane.setPreferredSize(new Dimension(100, 150));

        // make the vertical scroll bars the same for both scroll panes
        dataPane.setVerticalScrollBar(listPane.getVerticalScrollBar());

        // sets the fixed cell height for the list's items to the row height of the jtextarea
        updateFont();

        // add the dataPane to the canvasPane
        canvasPane.add(dataPane);

        // create the main split pane
        JSplitPane mainPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, listPane, canvasPane);
        mainPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
        mainPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);

        // create the status bar
        Box statusBar = new Box(BoxLayout.LINE_AXIS);
        statusBar.setAlignmentX(Box.LEFT_ALIGNMENT);
        statusBar.add(status);
        statusBar.add(insertStatus);

        // add the panes to the canvas
        add(mainPane, BorderLayout.PAGE_START);
        add(new JSeparator(SwingConstants.HORIZONTAL));
        add(statusBar, BorderLayout.PAGE_END);
    }

    /**
     * Refreshes the canvas on tab changes.
     * @param	event   used to determine whether or not to refresh
     */
    @Override
    public void stateChanged (ChangeEvent event) {
        Object source = event.getSource();
        if (source instanceof JTabbedPane && ((JTabbedPane) source).getSelectedComponent() == this) {
            program.addMenuHeading(1, "Edit").insert(cutAction, 0);
            program.addMenuHeading(1, "Edit").insert(copyAction, 1);
            program.addMenuHeading(1, "Edit").insert(pasteAction, 2);
            program.addMenuHeading(1, "Edit").insert(selectAllAction, 3);
            program.addMenuHeading(1, "Edit").insert(nameList.getInfoAction, 4);
            program.addMenuHeading(1, "Edit").insert(selectByNameAction, 5);
        } else {
            program.addMenuHeading(1, "Edit").remove(cutAction);
            program.addMenuHeading(1, "Edit").remove(copyAction);
            program.addMenuHeading(1, "Edit").remove(pasteAction);
            program.addMenuHeading(1, "Edit").remove(selectAllAction);
            program.addMenuHeading(1, "Edit").remove(nameList.getInfoAction);
            program.addMenuHeading(1, "Edit").remove(selectByNameAction);
        }
    }

    /**
     * Returns the current/selected data in the canvas.
     *
     * @return the current data for usage by commands
     */
    @Override
    public Sequence[] getData () {
        LinkedList<Sequence> lines = new LinkedList<Sequence>();
        Sequence[] result = new Sequence[0];

        // check whether to use the row(s) selected or the selection in the JTextArea
        if (useNameList) {
            // get the row(s) selected
            for (int lineNumber : nameList.getSelectedIndices()) {
                if (program.getSeqDoc().getSequence(lineNumber) != null) {
                    lines.add(program.getSeqDoc().getSequence(lineNumber));
                }
            }
            result = lines.toArray(new Sequence[0]);
        } else {
            // get the text selected
            result = dataCollector.getData();
        }
        return (result != null ? result : new Sequence[0]);
    }

    /**
     * Returns the name to display in the canvas tab for
     * @return  "GDE"
     */
    public String getTabName () {
        return "GDE";
    }

    /**
     * This function is used to determine whether to return the data in the
     * JTextArea or JList when getData is called.
     *
     * @param event the focus event to process
     */
    public void focusGained (FocusEvent event) {
        if (event.getComponent() instanceof GDEList) {
            useNameList = true;
        } else if (event.getComponent() instanceof BLTextArea) {
            useNameList = false;
        }
    }

    /**
     * This function is currently unused.
     *
     * @param event the focus event to process
     */
    public void focusLost (FocusEvent event) {
    }

    /**
     * Updates the font of the canvas to "newFont".
     */
    private void updateFont () {
        int fontsize = 12;
        String fontsizestr = program.getProperty("font.size");
        if (fontsizestr != null) {
            try {
                fontsize = Integer.parseInt(fontsizestr);
            } catch (NumberFormatException nfe) {
                nfe.printStackTrace();
            }
        }
        Font currentFont = new Font("Lucida Sans Typewriter", Font.PLAIN, fontsize);
        
        if ("true".equalsIgnoreCase(program.getProperty("font.bold"))) {
            dataCollector.setFont(currentFont.deriveFont(Font.BOLD));
        } else {
            dataCollector.setFont(currentFont);
        }
        nameList.setFont(currentFont);
    }

    /**
     * Copies content from the current Editable object to the clipboads.
     */
    public void copy () {
        if (useNameList) {
            nameList.copy();
        } else {
            dataCollector.copy();
        }
    }

    /**
     * Cuts content from the current Editable object to the clipboads.
     */
    public void cut () {
        if (useNameList) {
            nameList.cut();
        } else {
            dataCollector.cut();
        }
    }

    /**
     * Pastes the current clipboard into the current Editable object.
     */
    public void paste () {
        if (useNameList) {
            nameList.paste();
        } else {
            dataCollector.paste();
        }
    }

    /**
     * Selects all the sequences within the canvas.
     */
    public void selectAll () {
	nameList.selectAll();
	dataCollector.selectAll();
	nameList.requestFocus();
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    private void changeCase () {
        if (useNameList) {
            nameList.changeCase();
        } else {
            dataCollector.changeCase();
        }
    }

    /**
     * Receives cursor updates for use in the status bar.
     *
     * @param source the source of the cursor change.
     * @param column the new column of the cursor.
     * @param row the new row of the cursor.
     */
    public void cursorChange (BLTextArea source, int column, int row) {
        status.setText("Row: " + (row + 1) + " Col: " + (column + 1));
    }

    /**
     * Receives insertion mode change updates for use in the status bar.
     *
     * @param mode the new insertion mode status.
     */
    public void insertionMode (boolean mode) {
        if (mode) {
            insertStatus.setText("[INS]");
        } else {
            insertStatus.setText("     ");
        }
    }

    /**
     * Used to intercept font size changes (since the font size is stored in the properties)
     *
     * @param key the key of the property changed.
     * @param value the new value of the property.
     */
    public void propertiesUpdate(String key, String value) {
        if (key.toLowerCase().startsWith("font.")) {
            updateFont();
        }
    }
}
