/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas;

import java.awt.event.ItemEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JTextField;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.colourmask.ImportFileAction;
import org.biolegato.gdesupport.canvas.colour.PositionColourMap;

/**
 * A window for editing sequence properties.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDESequenceWindow extends JDialog implements ActionListener,
                                                          ItemListener {

    /**
     * The sequence associated with the window
     */
    private Sequence seq;
    /**
     * The text area used to obtain the sequence name from
     */
    private JTextField name = new JTextField();
    /**
     * The checkbox used to modify the protection of the sequence's alignment gaps
     */
    private JCheckBox protectAlignmentCB = new JCheckBox("Alignment gaps");
    /**
     * The checkbox used to modify the protection of the sequence's ambiguous characters
     */
    private JCheckBox protectAmbiguousCB = new JCheckBox("Ambiguous characters");
    /**
     * The checkbox used to modify the protection of the sequence's unambiguous characters
     */
    private JCheckBox protectUnambiguousCB = new JCheckBox(
            "Unambiguous characters");
    /**
     * The combobox used for modifying the sequence's type
     */
    private JComboBox type = new JComboBox(new Object[] {Sequence.Type.DNA,
                                                         Sequence.Type.RNA,
                                                         Sequence.Type.PROTEIN,
                                                         Sequence.Type.TEXT});
    /**
     * The combobox used for modifying the sequence's direction
     */
    private JComboBox direction = new JComboBox(new Object[] {
                Sequence.Direction.FROM5TO3, Sequence.Direction.FROM3TO5});
    /**
     * The combobox used for modifying the sequence's topology
     */
    private JComboBox topology = new JComboBox(new Object[] {
                Sequence.Topology.LINEAR, Sequence.Topology.CIRCULAR});
    /**
     * The combobox used for modifying the sequence's strandedness
     */
    private JComboBox strandedness = new JComboBox(
            new Object[] {Sequence.Strandedness.SINGLE,
                          Sequence.Strandedness.MIXED,
                          Sequence.Strandedness.DOUBLE});
    /**
     * The colour mask selection component
     */
    private ImportFileAction mask;
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622777013L;

    /**
     * Creates a window for editing the properties of a given sequence.
     *
     * @param index the index of the sequence within the BioDoc.
     * @param window the parent window.
     * @param seq the sequence to edit.
     */
    public GDESequenceWindow (final int index, final BLMain window,
                              final Sequence seq) {
        super(window, "Sequence properties");

        // transfer parameters to local class variables
        this.seq = seq;

        // configure the windows's main box
        Box mainBox = new Box(BoxLayout.PAGE_AXIS);

        // configure the sequqnece's name box
        Box nameBox = new Box(BoxLayout.LINE_AXIS);
        nameBox.add(new JLabel("Name:"));
        nameBox.add(name);
        name.setText(seq.getField("name").toString());
        mainBox.add(nameBox);

        // configure the sequqnece's type box
        Box typeBox = new Box(BoxLayout.LINE_AXIS);
        typeBox.add(new JLabel("Type:"));
        typeBox.add(type);
        type.setSelectedItem(seq.getField("type"));
        type.addItemListener(this);
        mainBox.add(typeBox);

        // configure the sequqnece's direction box
        Box directionBox = new Box(BoxLayout.LINE_AXIS);
        directionBox.add(new JLabel("Direction:"));
        directionBox.add(direction);
        direction.setSelectedItem(seq.getField("direction"));
        mainBox.add(directionBox);

        // configure the sequqnece's topology box
        Box topologyBox = new Box(BoxLayout.LINE_AXIS);
        topologyBox.add(new JLabel("Topology:"));
        topologyBox.add(topology);
        topology.setSelectedItem(seq.getField("topology"));
        mainBox.add(topologyBox);

        // configure the sequqnece's strandedness box
        Box strandednessBox = new Box(BoxLayout.LINE_AXIS);
        strandednessBox.add(new JLabel("Strandedness:"));
        strandednessBox.add(strandedness);
        strandedness.setSelectedItem(seq.getField("strandedness"));
        mainBox.add(strandednessBox);

        // configure the sequence protection box
        Box protectionBox = new Box(BoxLayout.PAGE_AXIS);
        protectionBox.add(protectAlignmentCB);
        protectAlignmentCB.setSelected(
                seq.getField("protect_align") != null && seq.getField(
                "protect_align") instanceof Boolean && ((Boolean) seq.getField(
                "protect_align")).booleanValue());
        protectionBox.add(protectAmbiguousCB);
        protectAmbiguousCB.setSelected(
                seq.getField("protect_ambig") != null && seq.getField(
                "protect_ambig") instanceof Boolean && ((Boolean) seq.getField(
                "protect_ambig")).booleanValue());
        protectionBox.add(protectUnambiguousCB);
        protectUnambiguousCB.setSelected(seq.getField("protect_unambig") != null &&
                                         seq.getField("protect_unambig") instanceof Boolean &&
                                         ((Boolean) seq.getField(
                "protect_unambig")).booleanValue());
        protectionBox.setBorder(BorderFactory.createTitledBorder(
                "Set character protections"));
        mainBox.add(protectionBox);

        // handle the colour mask selection widgets
        Box maskBox = new Box(BoxLayout.LINE_AXIS);
        PositionColourMap currentMask = null;
        if (seq.getField("mask") != null &&
            seq.getField("mask") instanceof PositionColourMap) {
            currentMask = (PositionColourMap) seq.getField("mask");
        }
        mask = new ImportFileAction(this, currentMask);
        maskBox.add(new JLabel("Colour mask:"));
        maskBox.add(mask);
        mainBox.add(maskBox);

        // add the update button
        JButton update = new JButton("Update");
        update.setActionCommand("update");
        update.addActionListener(this);
        mainBox.add(update);

        // display the window
        itemStateChanged(new ItemEvent(type, type.getSelectedIndex(),
                                       type.getSelectedItem(),
                                       ItemEvent.SELECTED));

        // create the window
        add(mainBox);
        pack();
        setLocationRelativeTo(window);
        setVisible(true);
    }

    /**
     * Adds an action handler to update the sequence data based on the window's values.
     *
     * @param e this is used to confirm that the update button was pressed.
     */
    public void actionPerformed (ActionEvent e) {
        //container.datachange(container, container.getLineStartOffset(index), 0);
        if ("update".equals(e.getActionCommand())) {
            seq.setField("name", name.getText());
            seq.setField("type", type.getSelectedItem());
            seq.setField("direction", direction.getSelectedItem());
            seq.setField("topology", topology.getSelectedItem());
            seq.setField("strandedness", strandedness.getSelectedItem());
            seq.setField("mask", mask.getSelectedItem());
            seq.setField("protect_align", new Boolean(
                    protectAlignmentCB.isSelected()));
            seq.setField("protect_ambig", new Boolean(
                    protectAmbiguousCB.isSelected()));
            seq.setField("protect_unambig", new Boolean(
                    protectUnambiguousCB.isSelected()));
            setVisible(false);
        }
    }

    /**
     * Handles type changes made (this includes greying out those parameters which are
     * not applicable to the current sequence type.
     *
     * @param e is currently ignored by the function.
     */
    public void itemStateChanged (ItemEvent e) {
        if (Sequence.Type.DNA.equals(type.getSelectedItem()) ||
		Sequence.Type.RNA.equals(type.getSelectedItem())) {
            direction.setEnabled(true);
            topology.setEnabled(true);
            strandedness.setEnabled(true);
            mask.setEnabled(true);
        } else if (Sequence.Type.PROTEIN.equals(type.getSelectedItem())) {
            direction.setEnabled(false);
            topology.setEnabled(false);
            strandedness.setEnabled(false);
            mask.setEnabled(true);
        } else if (Sequence.Type.TEXT.equals(type.getSelectedItem())) {
            direction.setEnabled(false);
            topology.setEnabled(false);
            strandedness.setEnabled(false);
            mask.setEnabled(false);
        }
    }

}
