/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas;

import org.biolegato.core.data.sequence.Sequence;
import java.util.regex.Pattern;
import org.biolegato.core.main.BLMain;

/**
 * Custom text area for allowing box selection
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDETextArea extends BLTextArea {

    /**
     * Used for serialization
     */
    private static final long serialVersionUID = 7526472295622777017L;

    /**
     * Constructs a new instance of GDETextArea.
     * 
     * @param instance the instance of BioLegato to attach the document wrapper to.
     */
    public GDETextArea (BLMain instance) {
        super(instance);
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        String data = null;
        Sequence[] seqList = null;

        // copy - paste section
        if ( ! selection.isEmpty()) {
            seqList = selection.getSelectionSequences(instance.getSeqDoc());
            for (Sequence seq : seqList) {
                data = seq.getField("sequence").toString();

                if (data.length() >= 0) {
                    if (Character.isUpperCase(data.charAt(0))) {
                        seq.setField("sequence", data.toLowerCase());
                    } else {
                        seq.setField("sequence", data.toUpperCase());
                    }
                }
            }
            deleteSelection();
            insert(seqList);
        }
    }

    /**
     * Inserts a character into the textarea's underlying SeqDoc
     *
     * @param character the character to insert.
     * @return whether or not the insertion was successful.
     */
    @Override
    protected boolean insert (char character) {
        boolean result = true;
        String text = character + "";
        Sequence current = instance.getSeqDoc().getSequence(row);

        result = (result &&  ! getProtectAlignment(current, text) &&
                   ! getProtectAmbiguous(current, text) &&
                   ! getProtectUnambiguous(current, text));

        if (result) {
            result = super.insert(character);
        }
        return result;
    }

    /**
     * Removes text from a document.
     *
     * @param offset the offset to start the deletion from.
     * @param length the number of characters to delete.
     * @return whether the deletion was a success.
     */
    @Override
    public boolean delete (int offset, int length) {
        boolean result = true;
        final int start = offset;
        final int end = start + length;
        final int startLine = Math.max(0, instance.getSeqDoc().getLineNumber(
                start));
        final int startColumn =
                  Math.max(0, Math.min(instance.getSeqDoc().getLineEndOffset(
                startLine),
                                       start -
                                       instance.getSeqDoc().getLineStartOffset(
                startLine)));
        final int endLine =
                  Math.max(0, Math.min(instance.getSeqDoc().getLineNumber(end),
                                       instance.getSeqDoc().getLineCount()));
        final int endColumn =
                  Math.max(0, Math.min(instance.getSeqDoc().getLineEndOffset(
                startLine),
                                       end -
                                       instance.getSeqDoc().getLineStartOffset(
                endLine)));
        String text = null;
        Sequence current = null;
        for (int count = startLine; count <= endLine; count ++) {
            current = (Sequence) instance.getSeqDoc().getSequence(count).clone();
            text = current.getField("sequence").toString();

            if (startLine == endLine && startColumn > 0 && endColumn > 0 &&
                endColumn < text.length()) {
                text = text.substring(startColumn, endColumn);
            } else if (count == startLine && startColumn > 0) {
                text = text.substring(startColumn);
            } else if (count == endLine && endColumn < text.length()) {
                text = text.substring(0, endColumn);
            }
            result = (result &&  ! getProtectAlignment(current, text) &&
                       ! getProtectAmbiguous(current, text) &&
                       ! getProtectUnambiguous(current, text));
        }
        if (result) {
            result = super.delete(offset, length);
        }
        return result;
    }

    /**
     * Checks a string against a sequence's alignment gap protection settings.
     *
     * This is done by obtaining the sequence's alignment gap protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * alignment gap character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAlignment (Sequence seq, String text) {
        boolean protect = false;

        if (Boolean.TRUE.equals(seq.getField("protect_align"))) {
            if (Sequence.Type.DNA.equals(seq.getField("type")) ||
                Sequence.Type.RNA.equals(seq.getField("type"))) {
                protect = Pattern.matches(
                        "[^ACTGUactguRYWSMKHBVIDNrywsmkhbvidn]", text);
            } else if (Sequence.Type.PROTEIN.equals(seq.getField("type"))) {
                protect = Pattern.matches("[\\s-]", text);
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's ambiguous character protection settings.
     *
     * This is done by obtaining the sequence's ambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * ambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAmbiguous (Sequence seq, String text) {
        boolean protect = false;

        if (Boolean.TRUE.equals(seq.getField("protect_ambig"))) {
            if (Sequence.Type.DNA.equals(seq.getField("type")) ||
                Sequence.Type.RNA.equals(seq.getField("type"))) {
                protect = Pattern.matches("[RYWSMKHBVIDNrywsmkhbvidn]", text);
            } else if (Sequence.Type.PROTEIN.equals(seq.getField("type"))) {
                protect = Pattern.matches("[BZ\\*Xbzx]", text);
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's unambiguous character protection settings.
     *
     * This is done by obtaining the sequence's unambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * unambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectUnambiguous (Sequence seq, String text) {
        boolean protect = false;

        if (Boolean.TRUE.equals(seq.getField("protect_unambig"))) {
            if (Sequence.Type.DNA.equals(seq.getField("type")) ||
                Sequence.Type.RNA.equals(seq.getField("type"))) {
                protect = Pattern.matches("[ACTGUactgu]", text);
            } else if (Sequence.Type.PROTEIN.equals(seq.getField("type"))) {
                protect = Pattern.matches("[^BZ\\*Xbzx\\s-]", text);
            }
        }
        return protect;
    }

}
