/*
 * SeqDoc.java
 *
 * Created on September 30, 2008, 11:08 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.core.data.seqdoc;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.Arrays;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;
import java.util.LinkedList;
import java.util.Stack;
import org.biolegato.core.main.BLMain;

/**
 * The internal document format for BioLegato.
 * <p>
 * This document is structured as a linked list of sequences.  Each character has an offset based on its
 * position within the list and it's position within its containing sequence.  Sequences start at 0
 * (first character in the first sequence in the list, and end with the last character in the last sequence
 * within the list.
 * </p>
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class SeqDoc implements SequenceListener, Transferable {

    /**
     * This linked list used to store all of the lines in the document.
     * <p>
     *  Each line is stored as a linked list of sequence wrappers.
     *  Each sequence wrapper is characterized by a sequence and a start offset.
     *  The start offset is used to determine the offset of the sequence within
     *  the document.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final LinkedList<Sequence> lines = new LinkedList<Sequence>();
    /**
     * This list is used to store all of the document's sequence listeners.
     * <p>
     *  Each time any modification is done to a sequence within the document, or
     *  any sequences are added/removed from the document, all of the document's
     *  listeners are notified of the event.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final LinkedList<SeqDocListener> listeners = new LinkedList<SeqDocListener>();
    /**
     * A data flavour representing the SeqDoc data type.
     */
    public static final DataFlavor seqDocFlavour = new DataFlavor(SeqDoc.class, "BioLegato sequence document");

    /**
     * Creates a new instance of SeqDoc
     */
    public SeqDoc () {
    }

    /**
     * Creates a new instance of SeqDoc using the existing sequence list
     *
     * @param sequenceList the list of sequences to initialize the SeqDoc with.
     */
    public SeqDoc (Sequence[] sequenceList) {
	for (Sequence seq : sequenceList) {
	    addSequence(getLineCount(), seq);
	}
    }
    
////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////

    /**
     * Inserts a string into the document on a given line.
     *
     * @param x the offset in the document to insert the text.
     * @param y the line number to insert the string.
     * @param string the text to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean insert (final int x, final int y, final String string) {
	return lines.get(y).insertField("sequence", x, string);
    }
    
    /**
     * Inserts sequences at a specific offset in the document.
     * <p>
     *  The first sequence is always merged with the sequence located at offset, unless the offset
     *  points to the end of the sequence.  The same applies to the last sequence. <br />
     * <br />
     *  <b>Example:</b>
     * <br />
     *  insertion X offset: 4<br />
     *  insertion Y offset: 0<br />
     * <br />
     *  Document: ggggtttt<br />
     * <br />
     *  Sequence[0]: aaaa<br />
     *  ....<br />
     *  Sequence[last]: cccc<br />
     * <br />
     *  Result: ggggaaaa<br />
     *          ....<br />
     *          cccctttt
     * </p>
     *
     * @param x the X co-ordinate offset in the document to insert the sequences.
     * @param y the Y co-ordinate offset in the document to insert the sequences.
     * @param sequences the array of sequences to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean insert (final int x, final int y, final Sequence[] sequences) {
        Sequence split = null;                                  // if the offset lands in the middle of the line, then this variable will store all of the sequence data after offset
        String current;                                         // the content of the current line
        String splitStr = "";                                   // the current line's sequence field
	boolean result = false;					// the result of the operation (true if successful)

        // make sure that the variables are all valid
        if (sequences != null && sequences.length > 0 && y >= 0 && (y <= getLineCount() || getLineCount() <= 0)) {
            
	    // itterate through all of the sequences
	    for (int count = 0; count < sequences.length; count++) {
		if (count + y < getLineCount()) {
		    result = insert(x, y + count, sequences[count].get("sequence").toString()) | result;
		} else {
		    result = addSequence(getLineCount(), sequences[count]) | result;
		}
	    }
	    
        } else if (sequences == null || sequences.length <= 0) {
            // report if the sequences to be inserted are invalid
            BLMain.error("Invalid sequence parameter", "SeqDoc.insert(int, Sequence[])");
        } else if (y < 0 || y > getLineCount()) {
            // report if the offset is invalid
            BLMain.error("Invalid Y-offset", "SeqDoc.insert(int, Sequence[])");
        }
        return result;
    }
    
    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.
     * </p>
     *
     * <b>NOTE: ALL DELETIONS ARE PERFORMED AS FOLLOWS:</b>
     * <br />
     *  Sequence[0]: ggggaaaa<br />
     *          ....<br />
     *  Sequence[3]: cccctttt
     * <br />
     *	<i>delete(4, 0, 4, 3);</i>
     * <br />
     *  Result: gggg
     *          ....
     *          cccc<br />
     * 
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @return true if the deletion was successful, otherwise false.
     */
    public boolean delete (final int x, final int y, final int w) {
	String data = getLineText(y);
	
	if (x >= 0 && x < data.length()) {
	    if (x + w >= data.length()) {
		data = data.substring(0, x);
	    } else {
		data = data.substring(0, x) + data.substring(x + w);
	    }
	    lines.get(y).put("sequence", data);
	}
	return true;
    }

    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.
     * </p>
     *
     * <b>NOTE: ALL DELETIONS ARE PERFORMED AS FOLLOWS:</b>
     * <br />
     *  Sequence[0]: ggggaaaa<br />
     *          ....<br />
     *  Sequence[3]: cccctttt
     * <br />
     *	<i>delete(4, 0, 4, 3);</i>
     * <br />
     *  Result: gggg
     *          ....
     *          cccc<br />
     * 
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return true if the deletion was successful, otherwise false.
     */
    public boolean delete (final int x, final int y, final int w, final int h) {
        String data = "";                                       // the data for the sequence after the removal
	boolean result = false;		                        // the result of the operation (true if successful)

	// make sure all of the parameters are valid
	if (!lines.isEmpty() && getLineCount() > 0 
		&& y >= 0 && y + h < getLineCount() && x >= 0) {
	    // substring the sequence (delete the data)
	    for (int count = 0; count <= h; count++) {
		result |= delete(x, y + count, w);
	    }
	}
        return result;
    }
    
    /**
     * Adds a sequence to the data container.  This function calls all listeners.
     *
     * @param y is the line number to insert the sequence.
     * @param seq is the sequence to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean addSequence (int y, Sequence seq) {
	lines.add(y, seq);
	seq.addListener(this);
	for (SeqDocListener listener : listeners) {
	    listener.sequenceAdded(this, y, seq);
	}
	return true;
    }
    
    /**
     * Removes a sequences from the data container.
     *
     * @param lineNumber the line number of the sequence to remove.
     * @return true if the deletion was successful, otherwise false.
     */
    public boolean removeSequence (final int lineNumber) {
        int length;			    // the length of the sequence removed.
        String text;			    // the data of the sequence removed.
        Sequence removed;		    // the sequence wrapper object removed.
	boolean result = false;		    // the result of the function
	
	// check the bounds of the line number to delete
	if (getLineCount() > lineNumber && lineNumber >= 0) {
	    // remove the sequence
	    removed = lines.remove(lineNumber);

	    // get information about the sequence removed
	    text = removed.get("sequence").toString();
	    length = text.length();
	    removed.removeListener(this);

	    // send sequence removed event
	    for (SeqDocListener listener : listeners) {
		listener.sequenceRemoved(this, lineNumber, removed);
	    }
	    result = true;
	}
	return result;
    }

//////////////////////////////////
//******************************//
//* INDIRECT DATA MODIFICATION *//
//******************************//
//////////////////////////////////
    
    /**
     * Removes an array of sequences from the data container.
     *
     * @param lineNumbers the line numbers to remove.
     */
    public void removeSequences (final int[] lineNumbers) {

        // sort the line numbers
        Arrays.sort(lineNumbers);
        
        // ensure that there are lines to delete
        if (!lines.isEmpty()) {
            // itterate backwards through each sequence line number and delete it (using the removeSequence method)
            for (int count = lineNumbers.length - 1; count >= 0; count--) {
		removeSequence(lineNumbers[count]);
            }
        }
    }

//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////

    /**
     * Retrieves a sequence object from the SeqDoc specified by its line number.
     *
     * @param y the line number to retreive the sequence.
     * @return the sequence.
     */
    public Sequence getLine (final int y) {
        return (y >= 0 && y < getLineCount() ? lines.get(y) : null);
    }
    
    /**
     * Retrieves text from the SeqDoc specified by its line number.
     *
     * @param lineNumber the line number to retreive the text.
     * @return the text.
     */
    private String getLineText(int y) {
	Sequence line = getLine(y);
        return (line != null ? line.get("sequence").toString() : "");
    }
    
    /**
     * Retrieves the length of a line in the document.
     *
     * @param y the line number to find the length of.
     * @return the length of the line (in characters).
     */
    public int getLineLength (final int y) {
        return getLineText(y).length();
    }

    /**
     * Returns the number of lines in the document.
     *
     * @return the number of lines in the document.
     */
    public int getLineCount () {
        return lines.size();
    }
    
    /**
     * Returns the length of the longest line in the data container
     *
     * @return the length of the longest line
     */
    public int getLongestLine() {
        int lineLength = 0;
        
        for (int count = 0; count < getLineCount(); count++) {
            lineLength = Math.max(lineLength, getLineLength(count));
        }
        return lineLength;
    }

    /**
     * Retrieves the line number for a sequence object in the SeqDoc.
     *
     * @param seq the sequence to search for.
     * @return the line number of the sequence.
     */
    public int indexOf (final Sequence seq) {
        return lines.indexOf(seq);
    }
    
/////////////////
//*************//
//* LISTENERS *//
//*************//
/////////////////
    /**
     * Adds a listener object to the data container.
     *
     * @param listener the listener to add.
     */
    public void addListener (final SeqDocListener listener) {
        listeners.add(listener);
    }

    /**
 * Called when a field in a sequence is modified.
     *
     * @param sequence the sequence modified.
     * @param key the key of the modified field in the sequence.
     */
    public void sequenceChanged(final Sequence sequence, final String key) {
        for (SeqDocListener listener : listeners) {
            listener.sequenceChanged(this, lines.indexOf(sequence), sequence, key);
        }
    }

////////////////////
//****************//
//* TRANSFERABLE *//
//****************//
////////////////////
    /**
     * Dictates what formats the SeqDoc can be converted to.
     *
     * @return an array of supported formats
     */
    public DataFlavor[] getTransferDataFlavors() {
	return new DataFlavor[] {seqDocFlavour, DataFlavor.stringFlavor};
    }

    /**
     * Dictates whether a given format is supported for conversion
     *
     * @param flavour the flavour to test for compatability
     * @return true if the format is supported
     */
    public boolean isDataFlavorSupported(DataFlavor flavour) {
	boolean result = false;
	DataFlavor[] test = getTransferDataFlavors();
	
	for (int count = 0; count < test.length && !result ; count++) {
	    result = flavour.equals(test[count]);
	}
	return result;
    }

    /**
     * Translates the SeqDoc to the given format
     *
     * @param flavour the data flavour to use for translation
     * @return the translated object
     * @throws UnsupportedFlavorException if the SeqDoc cannot be translated into the requested data flavour
     * @throws IOException if there is a problem with I/O during the translation
     */
    public Object getTransferData(DataFlavor flavour) throws UnsupportedFlavorException, IOException {
	Object result = this;

	if (!isDataFlavorSupported(flavour)) {
	    throw new UnsupportedFlavorException(flavour);
	}
	if (DataFlavor.stringFlavor.equals(flavour)) {
	    result = toString();
	}
	return result;
    }
    
///////////////
//***********//
//* GENERAL *//
//***********//
///////////////

    /**
     * Converts the SeqDoc into a string representation
     *
     * @return the string representation of the sequence document
     */
    public String toString () {
	String result = "";
	for (Sequence seq : lines) {
	    if (!result.equals("")) {
		result += "\n";
	    }
	    result += seq.get("sequence");
	}
	return result;
    }
    
    /**
     * Converts the SeqDoc into a sequence array representation
     *
     * @return the sequence array representation of the sequence document
     */
    public Sequence[] toArray() {
	return lines.toArray(new Sequence[0]);
    }
    
    /**
     * Finalizes the object - removes all references to itself
     */
    protected void finalize () {
	for (Sequence seq : lines) {
	    seq.removeListener(this);
	}
	lines.clear();
    }
}
