/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.core.properties;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileInputStream;
import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.Properties;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import org.biolegato.core.main.BLMain;
import org.biolegato.core.plugins.PluginLoader;
import org.biolegato.core.plugins.PluginWrapper;
import org.biolegato.core.plugintypes.PropertiesExtension;

/**
 * Stores the properties for BioLegato.
 * The current properties registered are:
 *
 * <table>
 *	<tr><th>Property</th>               <th>Used internally</th>    <th>Description</th></tr>
 *      <tr><td>debug</td>                  <td>Yes</td>                <td>determines whether or not to display debug information</td></tr>
 *	<tr><td>plugins</td>                <td>Yes</td>                <td>sets the plugin directory</td></tr>
 *	<tr><td>temp</td>                   <td>Yes</td>                <td>sets the temporary files directory</td></tr>
 *	<tr><td>font.size</td>              <td>Yes</td>                <td>determines the default font size for objects in BioLegato</td></tr>
 *	<tr><td>font.type</td>              <td>Yes</td>                <td>determines the default font type for objects in BioLegato</td></tr>
 *	<tr><td>shell.name</td>             <td>Yes</td>                <td>determines the shell to execute</td></tr>
 *	<tr><td>shell.parameter</td>        <td>Yes</td>                <td>determines the parameter for directing the shell to execute a command</td></tr>
 *	<tr><td>user.properties</td>        <td>Yes</td>                <td>determines whether or not to read the user's propery files (located in the user's home directory</td></tr>
 *	<tr><td>user.properties</td>        <td>Yes</td>                <td>determines whether or not to read the user's plugin files (located in the user's home directory</td></tr>
 *	<tr><td>default.canvas</td>         <td>Yes</td>                <td>determines the default canvas to show on startup</td></tr>
 *	<tr><td>default.fileformat</td>     <td>Yes</td>                <td>determines the default file format for open/save dialogs</td></tr>
 *	<tr><td>GDE.menu</td>               <td>No</td>                 <td>determines whether or not to read the GDE menu files (backwards compatibility mode)</td></tr>
 *	<tr><td>GDE.help.viewer</td>        <td>No</td>                 <td>determines which file to feed the help file to</td></tr>
 *	<tr><td>GDE.help.path</td>          <td>No</td>                 <td>the location to search for the help files</td></tr>
 *	<tr><td>undo</td>		    <td>No</td>                 <td>whether or not to enable undo support</td></tr>
 *	<tr><td>undo.size</td>		    <td>No</td>                 <td>whether or not to enable undo support</td></tr>
 * </table>
 * 
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class BLProperties extends Properties {

    /**
     * The list of all properties change listeners.
     */
    private Hashtable<String, LinkedList<PropertiesListener>> propertiesListeners =
                                                              new Hashtable<String, LinkedList<PropertiesListener>>();
    /**
     * The list containing all of the loaded properties extensions for BioLegato.
     */
    private static LinkedList<PropertiesExtension> propertyExtensionList =
                                                   new LinkedList<PropertiesExtension>();
    /**
     * Used for Serialization
     */
    private static final long serialVersionUID = 7526472295622776168L;

    /**
     * Create a new BLProperties instance
     */
    public BLProperties () {
        // set the default properties
        super(new Properties() {

            /**
             * Used for Serialization
             */
            private static final long serialVersionUID = 7526472295622777034L;


            {
                /* determines whether or not to display debug information */
                setProperty("debug", "false");
                /* sets the plugin directory */
                setProperty("plugins",
                            BLMain.PROGRAM_DIR +
                            File.separator + "plugins");
                /* sets the temporary file directory */
                setProperty("temp",
                            BLMain.CURRENT_DIR);
                /* determines the default font size */
                setProperty("font.size", "12");
                /* determines the default font type */
                setProperty("font.type", "Monospace");
                /* determines the shell to execute */
                setProperty("shell.name", (System.getProperty("os.name").startsWith("Windows 9") || System.getProperty("os.name").equals("Windows ME"))
                                          ? "command.com"
                                          : (System.getProperty("os.name").startsWith("Windows")
                                             ? "cmd.exe" : "/bin/sh"));
                /* determines the parameter for directing the shell to execute a command */
                setProperty("shell.parameter", (System.getProperty("os.name").startsWith("Windows")
                                                ? "/C" : "-c"));
                /* determines whether or not to read the user's propery files (located in the user's home directory */
                setProperty("user.properties", "true");
                /* determines whether or not to read the user's plugin files (located in the user's home directory */
                setProperty("user.plugins", "true");
                /* determines whether or not to enable undo/redo support */
                setProperty("undo", "false");
                /* determines the maximum size for the undo stack */
                setProperty("undo.size", "5");
                /* determines the default canvas to show on startup */
                setProperty("default.canvas", "Raw");
                /* determines the default fileformat for all open/save dialogs */
                setProperty("default.fileformat", "genbank");
                /* determines the name of new sequences */
                setProperty("default.sequencename", "New sequence");
            }

        });

        PropertiesExtension propertyExtension;  // variable for loading properties extensions.

        // reset the properties to the defaults
        reset();

        // read properties files
        loadPropertiesFile(BLMain.PROGRAM_DIR +
                           File.separator + ".blproperties");
        if ("true".equalsIgnoreCase(getProperty("user.properties")) &&
            System.getProperty("user.home") != null) {
            loadPropertiesFile(System.getProperty("user.home") + File.separator +
                               ".blproperties");
        }
    }

    /**
     * Changes a property within the BLProperties object
     *
     * @param key the key of the property to alter.
     * @param value the new value of the property.
     * @return the old value of the property changed (null if no previous value).
     */
    @Override
    public Object setProperty (String key, String value) {
        // make sure the key is always lower case
        key = key.toLowerCase();

        Object result = super.setProperty(key, value);  // the old value of the property changed
        LinkedList<PropertiesListener> listenerList;    // the listener list for sending listener events.

        // obtain the list of listeners to send an event update to
        listenerList = propertiesListeners.get(key.toLowerCase());

        // send the listener event update
        if (listenerList != null &&  ! listenerList.isEmpty()) {
            for (PropertiesListener listener : listenerList) {
                listener.propertiesUpdate(key, value);
            }
        }

        // return the function
        return result;
    }

    /**
     * Obtains a property from the BLProperties object.
     *
     * @param key the key of the property.
     * @return the current value of the property (returns "" if not set).
     */
    @Override
    public String getProperty (String key) {
        // make sure the key is always lower case
        key = key.toLowerCase();

        String result = super.getProperty(key);
        if (result == null) {
            result = "";
        }
        return result;
    }

    /**
     * Adds a listener for property changes
     * NOTE: <b>this method allows filtering based on which property changed</b>
     *
     * @param key the property key to listen for.
     * @param listener the listener to register.
     */
    public void addPropertiesListener (String key, PropertiesListener listener) {
        // make sure the key is always lower case
        key = key.toLowerCase();

        if ( ! propertiesListeners.containsKey(key)) {
            propertiesListeners.put(key, new LinkedList<PropertiesListener>());
        }
        propertiesListeners.get(key).add(listener);
    }

    /**
     * Sets all values to program defaults
     */
    private void reset () {
        super.clear();
        
        //******************//
        //* GDE PROPERTIES *//
        //******************//
        /* determines whether or not to read the GDE menu files (backwards compatibility mode) */
        setProperty("GDE.menu", "true");
        /* determines which file to feed the help file to */
        setProperty("GDE.help.viewer", "$BIRCH/script/gde_help_viewer.csh");
        /* the location to search for the help files */
        setProperty("GDE.help.path", (System.getenv("GDE_HELP_DIR") != null ? System.getenv("GDE_HELP_DIR")
                                      : ""));
	/*
	 * DEFAULT GDE COLOURS
	 */
	// Red
	setProperty("GDE.colour.1.red", "255");
	setProperty("GDE.colour.1.green", "0");
	setProperty("GDE.colour.1.blue", "0");
	
	// Orange
	setProperty("GDE.colour.2.red", "240");
	setProperty("GDE.colour.2.green", "180");
	setProperty("GDE.colour.2.blue", "20");
	
	// Blue
	setProperty("GDE.colour.3.red", "0");
	setProperty("GDE.colour.3.green", "0");
	setProperty("GDE.colour.3.blue", "255");
	
	// Black
	setProperty("GDE.colour.4.red", "0");
	setProperty("GDE.colour.4.green", "0");
	setProperty("GDE.colour.4.blue", "0");
	
	// Green
	setProperty("GDE.colour.5.red", "0");
	setProperty("GDE.colour.5.green", "200");
	setProperty("GDE.colour.5.blue", "0");
	
	// Pink
	setProperty("GDE.colour.6.red", "255");
	setProperty("GDE.colour.6.green", "0");
	setProperty("GDE.colour.6.blue", "160");
	
	// Tirquoise
	setProperty("GDE.colour.7.red", "0");
	setProperty("GDE.colour.7.green", "160");
	setProperty("GDE.colour.7.blue", "200");
	
	// Cyan
	setProperty("GDE.colour.8.red", "0");
	setProperty("GDE.colour.8.green", "127");
	setProperty("GDE.colour.8.blue", "255");
	
	// Olive 2
	setProperty("GDE.colour.9.red", "105");
	setProperty("GDE.colour.9.green", "139");
	setProperty("GDE.colour.9.blue", "30");
	
	// Purple
	setProperty("GDE.colour.10.red", "128");
	setProperty("GDE.colour.10.green", "0");
	setProperty("GDE.colour.10.blue", "255");
	
	// Blue-grey
	setProperty("GDE.colour.11.red", "125");
	setProperty("GDE.colour.11.green", "158");
	setProperty("GDE.colour.11.blue", "192");
	
	// Dark yellow
	setProperty("GDE.colour.12.red", "205");
	setProperty("GDE.colour.12.green", "205");
	setProperty("GDE.colour.12.blue", "0");
	
	// Deep pink
	setProperty("GDE.colour.13.red", "139");
	setProperty("GDE.colour.13.green", "10");
	setProperty("GDE.colour.13.blue", "80");
	
	// Burnt sienna
	setProperty("GDE.colour.14.red", "233");
	setProperty("GDE.colour.14.green", "116");
	setProperty("GDE.colour.14.blue", "81");
	
	// Medium tirquoise
	setProperty("GDE.colour.15.red", "0");
	setProperty("GDE.colour.15.green", "180");
	setProperty("GDE.colour.15.blue", "127");
	
	// Charteuse (med)
	setProperty("GDE.colour.16.red", "127");
	setProperty("GDE.colour.16.green", "230");
	setProperty("GDE.colour.16.blue", "0");
    }

    /**
     * Reads a properties file into BioLegato
     */
    private void loadPropertiesFile (String fileName) {
        if (new File(fileName).exists()) {
            try {
                load(new FileInputStream(fileName));
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Displays the properties window for this instance of BioLegato
     *
     * @param window the main program window to display the properties.
     */
    public void showPropertiesWindow (JFrame window) {
        final JDialog propertiesWindow = new JDialog(window);
        final Box outerPane = new Box(BoxLayout.PAGE_AXIS);
        final Box mainPane = new Box(BoxLayout.PAGE_AXIS);
        final JButton changeButton = new JButton(new AbstractAction("Close") {
            /* Used for serialization purposes */

            private static final long serialVersionUID = 7526472295622777011L;

            public void actionPerformed (ActionEvent e) {
                // close the properties window
		propertiesWindow.dispose();
            }

        });

        // add the outer
        outerPane.add(mainPane);
        outerPane.add(changeButton);

        PropertiesNumber fontChangeBox = new PropertiesNumber(this, "Font size",
                                                              "font.size", 10,
                                                              100);

        mainPane.add(fontChangeBox);
        for (PropertiesExtension ext : propertyExtensionList) {
            ext.propertiesComponent(mainPane);
        }

        propertiesWindow.add(outerPane);
        propertiesWindow.setLocationRelativeTo(window);
        propertiesWindow.pack();
        propertiesWindow.setVisible(true);
	propertiesWindow.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
    }

    /**
     * Adds a properties extension to the properties registry.
     *
     * @param extension the properties extension to add.
     */
    public void addExtension (PropertiesExtension extension) {
        if (extension != null) {
            propertyExtensionList.add(extension);
        }
    }

}
