/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.textarea;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.util.Hashtable;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import org.biolegato.core.data.sequence.Sequence;
import java.util.regex.Pattern;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.colourmap.GDECharColourMap;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;

/**
 * Custom text area for allowing box selection
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDETextArea extends BLTextArea {

    /**
     * Used for serialization
     */
    private static final long serialVersionUID = 7526472295622777017L;

    /**
     * Constructs a new instance of GDETextArea.
     */
    public GDETextArea () {
        // set up textarea colours
        setColourMap(new GDECharColourMap());
        setBackground(new Color(255, 255, 240));
	greyedOutMap = new GDECharColourMap(Color.BLACK, Color.LIGHT_GRAY, Color.DARK_GRAY, Color.WHITE);
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        String data = null;
        Sequence[] seqList = null;

        // copy - paste section
        if ( ! isEmptySelection()) {
            seqList = getData();
            for (Sequence seq : seqList) {
                data = seq.get("sequence").toString();

                if (data.length() >= 0) {
                    if (Character.isUpperCase(data.charAt(0))) {
                        seq.put("sequence", data.toLowerCase());
                    } else {
                        seq.put("sequence", data.toUpperCase());
                    }
                }
            }
            deleteSelection();
            insert(column, row, seqList);
        }
    }
    
    /**
     * Cuts content from the current Editable object.
     */
    @Override
    public void cut () {
        if ( ! isEmptySelection()) {
            copy();
            pdeleteSelection();
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    @Override
    public void paste () {
        pdeleteSelection();
        insert(column, row, BLMain.getClipboard());
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     * NOTE: ng stands for non-grouped
     *
     * @return whether or not the deletion was performed.
     */
    public boolean pdeleteSelection () {
	final int x = getMinSX();
	final int y = getMinSY();
	final int w = getMaxSX() - x;
	final int h = getMaxSY() - y;
	boolean result = false;
	
        if ( ! isEmptySelection() && pdelete(x, y, w, h)) {
	    moveSelection(0, 0);
	    stretchSelection(0, 0);
	    changePosition(x, y);
	    result = true;
        }
        return result;
    }
    
    /**
     * This function is used to test if a line is selected.
     *
     * @param y the Y co-ordinate (or line number) to test.
     * @return true if the line is within the selection area, and should be printed as such.
     */
    public boolean isSelectedLine(int y) {
	boolean result = super.isSelectedLine(y);
	Sequence current = null;
	LinkedList<Sequence> group = null;
	
	if (!result) {
	    current = BLMain.getSeqDoc().getLine(y);
	    if (current.get("group") instanceof Integer && (group = Sequence.getgroup((Integer)current.get("group"))) != null) {
		// itterate through the group and perform the mass insertion
		for (Sequence seq : group) {
		    if (super.isSelectedLine(BLMain.getSeqDoc().indexOf(seq))) {
			result = true;
			break;
		    }
		}
	    }
	}
	return result;
    }

    /**
     * Moves the start of a selection to a new co-ordinate
     *
     * @param newx the new X co-ordinate the selection has been moved to.
     * @param newy the new Y co-ordinate the selection has been moved to.
     */
    public void moveSelection(int newx, int newy) {
        final int x = 0;					// the X co-ordinate to start repainting at
        final int y = 0;					// the width of the area to repaint
        final int w = (int)getSize().getWidth();		// the Y co-ordinate to start repainting at
        final int h = (int)getSize().getHeight();		// the height of the area to repaint
	
	super.moveSelection(newx, newy);
	
	// repaint the affected canvas area
        repaint(x, y, w, h);				// repaint the modified area of the textarea	
    }
    
    /**
     * Stretches the selection by changing the co-ordinate of the selection's end
     *
     * @param newx the new X co-ordinate the selection has been stretched to.
     * @param newy the new Y co-ordinate the selection has been stretched to.
     */
    public void stretchSelection(int newx, int newy) {
        final int x = 0;					// the X co-ordinate to start repainting at
        final int y = 0;					// the width of the area to repaint
        final int w = (int)getSize().getWidth();		// the Y co-ordinate to start repainting at
        final int h = (int)getSize().getHeight();		// the height of the area to repaint

	super.stretchSelection(newx, newy);
	
	// repaint the affected canvas area
        repaint(x, y, w, h);				// repaint the modified area of the textarea	
    }
    
    /**
     * Checks a string against all of a sequence's protection settings.
     *
     * This is done by obtaining the sequence's protection settings,
     * the type of the sequence, and whether the text contains a character in any
     * protected character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean isProtectionsOn(Sequence current, String text) {
	return getProtectAlignment(current, text) ||
	   getProtectAmbiguous(current, text) ||
	   getProtectUnambiguous(current, text);
    }
    
    /**
     * Checks a string against a sequence's alignment gap protection settings.
     *
     * This is done by obtaining the sequence's alignment gap protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * alignment gap character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAlignment (Sequence seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_align"))) {
            if (Sequence.Type.DNA.equals(seq.get("type")) ||
		    Sequence.Type.RNA.equals(seq.get("type"))) {
		
		test = text.toLowerCase().toCharArray();
		
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || (test[count] != 'a' && test[count] != 'c' && test[count] != 't'
			    && test[count] != 'g' && test[count] != 'u' && test[count] != 'r'
			    && test[count] != 'y' && test[count] != 'w' && test[count] != 's'
			    && test[count] != 'm' && test[count] != 'k' && test[count] != 'h'
			    && test[count] != 'b' && test[count] != 'v' && test[count] != 'i'
			    && test[count] != 'd' && test[count] != 'n');
		}
            } else if (Sequence.Type.PROTEIN.equals(seq.get("type"))) {
		
		test = text.toLowerCase().toCharArray();
		
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == ' ' || test[count] == '\n'
			    || test[count] == '\t' || test[count] == '\r' || test[count] == '-';
		}
                protect = Pattern.matches("[\\s-]", text);
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's ambiguous character protection settings.
     *
     * This is done by obtaining the sequence's ambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * ambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAmbiguous (Sequence seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_ambig"))) {
            if (Sequence.Type.DNA.equals(seq.get("type")) ||
                Sequence.Type.RNA.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'r' || test[count] == 'y' || test[count] == 'w'
			    || test[count] == 's' || test[count] == 'm' || test[count] == 'k'
			    || test[count] == 'h' || test[count] == 'b' || test[count] == 'v'
			    || test[count] == 'i' || test[count] == 'd' || test[count] == 'n';
		}
            } else if (Sequence.Type.PROTEIN.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'b' || test[count] == '*'
			    || test[count] == 'z' || test[count] == 'x';
		}
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's unambiguous character protection settings.
     *
     * This is done by obtaining the sequence's unambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * unambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectUnambiguous (Sequence seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_unambig"))) {
            if (Sequence.Type.DNA.equals(seq.get("type")) ||
                Sequence.Type.RNA.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'a' || test[count] == 'c'
			    || test[count] == 't' || test[count] == 'g' || test[count] == 'u';
		}
            } else if (Sequence.Type.PROTEIN.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || (test[count] != 'b' && test[count] != '*'
			    && test[count] != 'z' && test[count] != 'x' && test[count] != ' '
			    && test[count] != '\n' && test[count] != '\t' && test[count] != '\r'
			    && test[count] != '-');
		}
            }
        }
        return protect;
    }
    
    /**
     * Inserts a character into the textarea's underlying SeqDoc
     * (NOTE: this should ONLY be called by sub-methods)
     *
     * @param x the X co-ordinate (column number) to insert the character at.
     * @param y the Y co-ordinate (row number) to insert the character at.
     * @param character the character to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert (final int x, final int y, String text) {
	int gln;					    // the current line number in the sequence group to add the characters to
        boolean result = true;
        Sequence current = BLMain.getSeqDoc().getLine(y);
	LinkedList<Sequence> group;

	if (current.get("group") instanceof Integer && (group = Sequence.getgroup((Integer)current.get("group"))) != null) {
	    // itterate through the group and perform the mass insertion
	    for (Sequence seq : group) {
		gln = BLMain.getSeqDoc().indexOf(seq);
		if (gln >= 0 && gln < BLMain.getSeqDoc().getLineCount()) {
		    if ( ! isProtectionsOn(seq, text)) {
			result &= super.insert(x, gln, text);
		    }
		} else {
		    BLMain.error("Invalid row number: " + gln, "boolean GDETextArea.insert (char)");
		}
	    }
	} else if ( ! isProtectionsOn(current, text)) {
	    result = super.insert(x, y, text);
	}
        return result;
    }

    /**
     * Removes text from a document.
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return whether the deletion was a success.
     */
    @Override
    public boolean delete (final int x, final int y, final int w, final int h) {
        boolean result = false;
        Sequence current = null;
	int gln;					    // the current line number in the sequence group to add the characters to
	LinkedList<Sequence> group;

	for (int count = y; count <= y + h; count ++) {
            current = (Sequence) BLMain.getSeqDoc().getLine(count).clone();
	    
	    if (current.get("group") instanceof Integer && (group = Sequence.getgroup((Integer)current.get("group"))) != null) {
		// itterate through the group and perform the mass insertion
		for (Sequence seq : group) {
		    gln = BLMain.getSeqDoc().indexOf(seq);
		    if ((gln < y || gln > y + h) && gln < BLMain.getSeqDoc().getLineCount()) {
			result &= pdelete(x, gln, w, 0);
		    } else if (gln >= BLMain.getSeqDoc().getLineCount()) {
			BLMain.error("Invalid row number: " + y, "boolean GDETextArea.insert (char)");
		    }
		}
	    }
	    result = pdelete(x, count, w, 0);
	}
        return result;
    }
    
    /**
     * Removes text from one line of a document
     * (doesn't do any group processing - permission processing only).
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @return whether the deletion was a success.
     */
    public boolean pdelete (final int x, final int y, final int w, final int h) {
	Sequence current = null;
	String text = null;
	boolean result = false;

	for (int count = y; count <= y + h; count++) {
	    current = (Sequence) BLMain.getSeqDoc().getLine(count).clone();
	    text = current.get("sequence").toString();
	    
	    if (x > 0 && x < text.length()) {
		text = text.substring(x);
	    } else if (x > text.length()) {
		text = "";
	    }
	    if (w > 0 && w < text.length()) {
		text = text.substring(0, w);
	    } else if (w < 0) {
		text = "";
	    }
	    
	    if ( ! isProtectionsOn(current, text)) {
		result = super.delete(x, count, w, 0);
	    }
	}
	
	return result;
    }
}
