/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.textarea;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.util.Hashtable;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import org.biolegato.core.data.sequence.Sequence;
import java.util.regex.Pattern;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.*;
import org.biolegato.gdesupport.canvas.colourmap.GDECharColourMap;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import org.biolegato.gdesupport.canvas.undo.UndoChangePosition;
import org.biolegato.gdesupport.canvas.undo.UndoDelete;
import org.biolegato.gdesupport.canvas.undo.UndoInsert;
import org.biolegato.gdesupport.canvas.undo.UndoMoveSelection;
import org.biolegato.gdesupport.canvas.undo.UndoMulti;
import org.biolegato.gdesupport.canvas.undo.UndoStretchSelection;
import org.biolegato.gdesupport.canvas.undo.Undoable;

/**
 * Custom text area for allowing box selection
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class UndoableGDETextArea extends GDETextArea {

    /**
     * Stores a symbolic link to the parent canvas.
     */
    private GDECanvas canvas = null;
    /**
     * Self-reference for inner classes.
     */
    public final UndoableGDETextArea ugdeTextAreaSelf = this;
    /**
     * Used for serialization
     */
    private static final long serialVersionUID = 7526472295622777017L;

    /**
     * Constructs a new instance of GDETextArea.
     */
    public UndoableGDETextArea (GDECanvas canvas) {
        // initialize variables
	this.canvas = canvas;

	// add undo/redo to the menu
	addPopupMenuItem(new JMenuItem(canvas.undoAction));
	addPopupMenuItem(new JMenuItem(canvas.redoAction));
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        String data = null;
        Sequence[] seqList = null;
	UndoMulti result = new UndoMulti();

        // copy - paste section
        if ( ! isEmptySelection()) {
            seqList = getData();
            for (Sequence seq : seqList) {
                data = seq.get("sequence").toString();

                if (data.length() >= 0) {
                    if (Character.isUpperCase(data.charAt(0))) {
                        seq.put("sequence", data.toLowerCase());
                    } else {
                        seq.put("sequence", data.toUpperCase());
                    }
                }
            }
            result.addUndo(udeleteSelection());
            result.addUndo(uinsert(column, row, seqList));
	    canvas.addUndo(result);
        }
    }
    
    /**
     * Cuts content from the current Editable object.
     */
    @Override
    public void cut () {
        if ( ! isEmptySelection()) {
            copy();
            canvas.addUndo(updeleteSelection());
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    @Override
    public void paste () {
	UndoMulti paste = new UndoMulti();
	
        paste.addUndo(updeleteSelection());
        paste.addUndo(uinsert(column, row, BLMain.getClipboard()));
	canvas.addUndo(paste);
    }

    /**
     * Processes the typing of keys within the text area
     *
     * @param event the KeyEvent for the key typed
     */
    public void keyTyped (KeyEvent event) {
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    boolean canInsert = true;
	    Undoable returnedUndo = null;
	    UndoMulti resultUndo = new UndoMulti();

	    try {
		switch (event.getKeyChar()) {
		    case KeyEvent.VK_BACK_SPACE:
		    case KeyEvent.VK_DELETE:
			break;
		    case KeyEvent.VK_ENTER:
			// now only changes the row (same effect as downkey, except clears selection)
			if (row + 1 < BLMain.getSeqDoc().getLineCount()) {
			    changePosition(Math.min(BLMain.getSeqDoc().getLineLength(row + 1), column), row + 1);
			    resetSelection();
			}
			break;
		    default:
			if ( ! isEmptySelection()) {
			    resultUndo.addUndo(udeleteSelection());
			} else if (insertMode && BLMain.getSeqDoc().getLineCount() > 0
				&& BLMain.getSeqDoc().getLineLength(0) > 0) {
			    returnedUndo = udelete(column, row, 1, 0);
			}
			if (isEmptySelection() || !insertMode || returnedUndo != null) {
			    resultUndo.addUndo(returnedUndo);
			    resultUndo.addUndo(uinsert(column, row, "" + event.getKeyChar()));
			    resultUndo.addUndo(uchangePosition(column + 1, row));
			}
		    break;
		}
	    } catch (Throwable e) {
		e.printStackTrace();
	    }
	    if ( ! resultUndo.isEmpty()) {
		canvas.addUndo(resultUndo);
	    }
	} else {
	    super.keyTyped(event);
	}
    }

    /**
     * Processes key presses within the text area
     *
     * @param event the KeyEvent for the key pressed
     */
    public void keyPressed (KeyEvent event) {
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    int backspacecol = 0;
	    boolean moved = false;  // used for updating selections
	    UndoMulti result = new UndoMulti();
	    Undoable tempResult = null;

	    try {
		switch (event.getKeyChar()) {
		    case KeyEvent.VK_BACK_SPACE:
			if ( ! isEmptySelection()) {
			    result.addUndo(udeleteSelection());
			} else if (row > 0 || column > 0) {
			    if (column <= 0 && row > 0) {
				backspacecol = BLMain.getSeqDoc().getLineLength(row - 1);
			    }
			    if (column - 1 >= 0) {
				tempResult = udelete(column - 1, row, 1, 0);
				if (tempResult != null) {
				    result.addUndo(tempResult);
				    if (column <= 0 && row > 0) {
					result.addUndo(uchangePosition(backspacecol, row - 1));
				    } else {
					result.addUndo(uchangePosition(column - 1, row));
				    }
				}
			    }
			}
			break;
		    case KeyEvent.VK_DELETE:
			if ( ! isEmptySelection()) {
			    result.addUndo(udeleteSelection());
			} else if (BLMain.getSeqDoc().getLineCount() > 0
				&& BLMain.getSeqDoc().getLineLength(0) > 0) {
			    result.addUndo(udelete(column, row, 1, 0));
			}
			break;
		    default:
			super.keyPressed(event);
			break;
		}
	    } catch (Throwable e) {
		e.printStackTrace();
	    }

	    if ( ! result.isEmpty()) {
		canvas.addUndo(result);
	    }
	} else {
	    super.keyTyped(event);
	}
    }
    
    /**
     * Inserts a character into the textarea's underlying SeqDoc
     * (NOTE: this should ONLY be called by sub-methods)
     *
     * @param x the X co-ordinate (column number) to insert the character at.
     * @param y the Y co-ordinate (row number) to insert the character at.
     * @param character the character to insert.
     * @return whether or not the insertion was successful.
     */
    public Undoable uinsert (final int x, final int y, String text) {
	int gln;					    // the current line number in the sequence group to add the characters to
        Undoable result = null;
        Sequence current = BLMain.getSeqDoc().getLine(y);
	LinkedList<Sequence> group;

	if (current.get("group") instanceof Integer && (group = Sequence.getgroup((Integer)current.get("group"))) != null) {
	    // change successful insertion conditions
	    result = new UndoMulti();
	    
	    // itterate through the group and perform the mass insertion
	    for (Sequence seq : group) {
		gln = BLMain.getSeqDoc().indexOf(seq);
		if (gln < BLMain.getSeqDoc().getLineCount()) {
		    if ( ! isProtectionsOn(seq, text) && insert(x, y, text)) {
			((UndoMulti)result).addUndo(new UndoInsert(this, x, y, text.length()));
		    }
		} else {
		    BLMain.error("Invalid row number: " + y, "boolean GDETextArea.insert (char)");
		}
	    }
	} else if ( ! isProtectionsOn(current, text) && super.insert(x, y, text)) {
	    result = new UndoInsert(this, x, y, text.length());
	}
        return result;
    }

    /**
     * Inserts an array of sequences into the textarea's underlying SeqDoc
     *
     * @param x the X co-ordinate (column number) to insert the sequences at.
     * @param y the Y co-ordinate (row number) to insert the sequences at.
     * @param sequences the array of sequences to insert.
     * @return whether or not the insertion was successful.
     */
    public Undoable uinsert (final int x, final int y, Sequence[] sequences) {
        UndoMulti result = null;
	
        if (insert(x, y, sequences)) {
	    for (int count = 0; count < sequences.length; count++) {
		result.addUndo(new UndoInsert(this, x, y + count, sequences[count].get("sequnce").toString().length()));
	    }
        }
        return result;
    }
    
    /**
     * Removes text from a document.
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return whether the deletion was a success.
     */
    public Undoable udelete (final int x, final int y, final int w, final int h) {
        UndoMulti result = new UndoMulti();
        Sequence current = null;
	int gln;					    // the current line number in the sequence group to add the characters to
	LinkedList<Sequence> group;

	for (int count = y; count <= y + h; count ++) {
            current = (Sequence) BLMain.getSeqDoc().getLine(count).clone();
	    
	    if (current.get("group") instanceof Integer && (group = Sequence.getgroup((Integer)current.get("group"))) != null) {
		// change successful insertion conditions
		result = new UndoMulti();

		// itterate through the group and perform the mass insertion
		for (Sequence seq : group) {
		    gln = BLMain.getSeqDoc().indexOf(seq);
		    if ((gln < y || gln > y + h) && gln < BLMain.getSeqDoc().getLineCount()) {
			result.addUndo(updelete(x, gln, w, 0));
		    } else {
			BLMain.error("Invalid row number: " + y, "boolean GDETextArea.insert (char)");
		    }
		}
	    } else {
		result.addUndo(updelete(x, y, w, 0));
	    }
	}
        return result;
    }
    
    /**
     * Removes text from one line of a document
     * (doesn't do any group processing - permission processing only).
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @return an undoable object corresponding to the deletion
     */
    public Undoable updelete (final int x, final int y, final int w, final int h) {
	Sequence current = null;
	UndoMulti result = new UndoMulti();

	for (int count = 0; count <= h; count++) {
	    current = BLMain.getSeqDoc().getLine(count).subseq(x, x + w);
	    if (pdelete(x, y + count, w, 0)) {
		result.addUndo(new UndoDelete(this, x, y + count, current.get("sequence").toString()));
	    }
	}
	return result;
    }
    
    /**
     * The undoable version of the "deleteSelection" function
     *
     * @return creates an undoable object to undo selection deleteions.
     */
    public Undoable updeleteSelection () {
	final int x = getMinSX();
	final int y = getMinSY();
	final int w = getMaxSX() - x;
	final int h = getMaxSY() - y;
	Undoable resultDelete = null;
	UndoMulti resultFunction = new UndoMulti();
	
        if ( ! isEmptySelection()) {
	    resultDelete = updelete(x, y, w, h);
	    if (resultDelete != null) {
		resultFunction.addUndo(resultDelete);
		resultFunction.addUndo(uchangePosition(x, y));
		resultFunction.addUndo(umoveSelection(0, 0));
		resultFunction.addUndo(ustretchSelection(0, 0));
	    }
        }
        return resultFunction;
    }
    
    /**
     * The undoable version of the "deleteSelection" function
     *
     * @return creates an undoable object to undo selection deleteions.
     */
    public Undoable udeleteSelection () {
	final int x = getMinSX();
	final int y = getMinSY();
	final int w = getMaxSX() - x;
	final int h = getMaxSY() - y;
	Undoable resultDelete = null;
	UndoMulti resultFunction = new UndoMulti();
	
        if ( ! isEmptySelection()) {
	    resultDelete = udelete(x, y, w, h);
	    if (resultDelete != null) {
		resultFunction.addUndo(resultDelete);
		resultFunction.addUndo(uchangePosition(x, y));
		resultFunction.addUndo(umoveSelection(0, 0));
		resultFunction.addUndo(ustretchSelection(0, 0));
	    }
        }
        return resultFunction;
    }
    /**
     * The undoable version of the "moveSelection" function
     *
     * @param newx the new X co-ordinate the selection has been stretched to.
     * @param newy the new Y co-ordinate the selection has been stretched to.
     * @return creates an undoable object to undo selection movement.
     */
    public Undoable umoveSelection(int newx, int newy) {
	final int oldx = getSX1();
	final int oldy = getSY1();
	
	moveSelection(newx, newy);
	
	return new UndoMoveSelection(this, oldx, oldy);
    }
    /**
     * The undoable version of the "stretchSelection" function
     *
     * @param newx the new X co-ordinate the selection has been stretched to.
     * @param newy the new Y co-ordinate the selection has been stretched to.
     * @return creates an undoable object to undo selection stretches.
     */
    public Undoable ustretchSelection(int newx, int newy) {
	final int oldx = getSX2();
	final int oldy = getSY2();
	
	stretchSelection(newx, newy);
	
	return new UndoStretchSelection(this, oldx, oldy);
    }
    /**
     * The undoable version of the "changePosition" function
     *
     * @param newColumn the column co-ordinate of the new position.
     * @param newRow the row co-ordinate of the new position.
     * @return creates an undoable object to undo position changes.
     */
    public Undoable uchangePosition (int newColumn, int newRow) {
        final int oldRow = row;
        final int oldColumn = column;
        
	super.changePosition(newColumn, newRow);
	
	return new UndoChangePosition(this, oldColumn, oldRow);
    }
}
