//=====================================================================
// File:    CutoffFunction.java
// Class:   CutoffFunction
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * An abstract class that is used to represent different functions for
 * cutting off noise from data peaks. The parameters need to compute the
 * cutoff for a position should be passed in as options. Examples
 * would be things like simple linear functions, or a decaying exponential.
 * in the first case, the value would be the same for every position, but
 * in the case of a decaying exponential, the cutoff would decrease as
 * the position increased. In many cases, the parameters required will
 * not truly be optional, but the "options" can be marked as required. This
 * is done so that the program can tell the user what the function requires.
 *
 * <b>NOTE:</b> The first option <b>must</b> set the height of the function.
 * This will correspond to the cutoff value at the position defined by
 * <code>Cutoff</code>. This is required so that the other parts of the
 * program can interface with the different functions in a standard manner.
 *
 * @author James J. Benham
 * @version 0.1.0
 * @date June 15, 1998
 */

public abstract class CutoffFunction extends Operation implements Cloneable
{
  /**
   * Returns the cutoff value for the specified position
   *
   * @return the cutoff at the position
   */
  public abstract double getCutoff(double position);

  /**
   * Gives an object where every field has been copied from this object
   * to the new object.
   *
   * @return a copy of this object.
   */
  public Object clone()
  {
    try
      {
     return super.clone();
      }
    catch(CloneNotSupportedException e)
      {
     System.err.println("Could not clone Cutoff Function. " + 
                  e.getMessage());
      }
    return null;
  }

  /**
   * Writes all of the information this class needs to store in order
   * to be recreated. This will be used for things like storing the
   * class in a file. Therefore, the write should output enough information
   * so that <code>read</code> can recreate the essential properties of this
   * class.
   *
   * @param out  the destination to write the data to.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void write(DataOutputStream out) throws IOException;

  /**
   * Reads in the properties of this class from the specified input stream.
   * The stream data should have been created by <code>write</code>. This
   * will retrieve this classes state from the input stream. All of the
   * current data in this class will be replaced by the data from the 
   * stream.
   *
   * @param in  the input stream with the data for the class.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void read(DataInputStream in) 
       throws IOException;
}
