//=====================================================================
// File:    ImportFilter.java
// Class:   ImportFilter
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.File;
import java.io.IOException;

/** 
 * This is an abstract class that should be extended to create
 * filters that read data from files. The data files should be of two
 * basic types, gel or lane. A lane file would contain only data for a
 * single lane. A gel file would contain the entire gel
 * In that case, the lanes would have to be tracked and
 * extracted, and then reassembled into a Gel. The filter should size
 * the data using a size standard if neccessary. Support for this may
 * be added elsewhere in the program, in which case, the filter could
 * simply call the approriate methods. Obviously a filter cannot
 * implement both readLane and readGel, so for the one that is not
 * supported, <code>null</code> should be returned. The class
 * <code>ImportMgr</code> can be used to track the different
 * filters. By adding the filters to it, one should be able to have a
 * program that read several different file types.
 *
 * @see Operation
 * @see Lane
 * @see Gel
 * @see SizeFunction
 * @see ImportMgr
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public abstract class ImportFilter extends Operation
{
  // Constants
  /** constant to determine type of file filter is used on */
  public final int LANE = 0;
  /** constant to determine type of file filter is used on */
  public final int GEL = 1;

  protected int filetype;        // the type, see constants above

  /**
   * Returns the type of input file supported by this filter, options are
   * <code>ImportFilter.LANE</code> or <code>ImportFilter.GEL</code>. This
   * information should be used by the caller to determine which method to 
   * call <code>readLane</code> or <code>readGel</code>.
   *
   * @return either constant GEL or LANE, as described above.
   */
  public abstract int getFileType();

  /**
   * This is the method that is called to preform the actual reading of the
   * file. The data in the file represents data from a gel. If 
   * the filter has any options to be selected, they should be set using the 
   * <code>get/setOptions</code> methods. This method should check to see 
   * that the options are appropriate before continuing. Note that a filter 
   * will only work on one type of data file, gel or lane. Therefore, one 
   * of the methods <code>readLane</code> or <code>readGel</code>, will not 
   * apply to the filter. For the one that does not apply, <code>null</code> 
   * should be returned.
   *
   * @param inputFile  The file that contains the lane data.
   *
   * @return a Lane object with all of the appropriate information. It will
   *     return <code>null</code> if the filter reads gels instead of lanes.
   *
   * @exception IOException If an error is encountered in the file,
   * then this exception will be thrown 
   */
  public abstract Lane [] readLane(File inputFile) throws IOException;

  /**
   * This is the method that is called to preform the actual reading of the
   * file. The data in the file represents data from a gel. If 
   * the filter has any options to be selected, tose should be set using the 
   * <code>get/setOptions</code> methods. This method should check to see 
   * that the options are appropriate before continuing. Note that a filter 
   * will only work on one type of data file, gel or lane. Therefore, one 
   * of the methods <code>readLane</code> or <code>readGel</code>, will not 
   * apply to the filter. For the one that does not apply, <code>null</code> 
   * should be returned.
   *
   * @param inputFile  The file that contains the gel data.
   *
   * @return a Lane object with all of the appropriate information. It will
   *     return <code>null</code> if the filter reads lanes instead of gels.
   *
   * @exception IOException If an error is encountered in the file,
   * then this exception will be thrown 
   */
  public abstract Gel readGel(File inputFile) throws IOException;
}
