//=====================================================================
// File:    SCFFilter.java
// Class:   SCFFilter
// Package: AFLPcore
//
// Author:  Philip DeCamp (using much code written by James J. Benham)
// Date:    May 31, 2001
// Contact: decamp@portalofevil.com or james_benham@hmc.edu
// 
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.File;
import java.io.RandomAccessFile;
import java.io.IOException;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.FileWriter;
import java.io.FileNotFoundException;
import java.util.NoSuchElementException;

/**
 * This class reads data from an SCF file, which are fairly common.
 * Reading in the data is actually fairly complicated in that genographer
 * is only very useful with molecular weight standards.  SCF files always have
 * four channels for data (these are usually used for A, C, G, T).  However,
 * genographer only looks at one channel at a time for data, plus one other
 * channel for the molecular weight standards.  Unlike ABI files, SCF files
 * do not contain the peak information, so this filter must use "TracePeak
 * Finder" in order to size the data.  Basically, this filter does the
 * following:
 * 1)  Extract necessary file information from the header.
 * 2)  Extract data from data channel.
 * 3)  Extract data from the molecular weight standards channel
 * 4)  Detect peaks in the standards channel.
 * 5)  Read in the sizing method from the standards.cfg file
 * 6)  Combine the information from the peaks with the sizing method file
 *		to calibrate the sizing function.
 * 7)  Parse through the comments for Name, Gel Name, and Lane Number
 * 8)  Return Lane object, which contains the trace data and sizing function.
 *
 * You've probably noticed, if you've ever actually run this beast, that this
 * filter has a number of options that need to be set.
 * 1)  Data channel
 * 2)  Standards channel
 * 3)  Sizing Function
 * 4)  Standards Used
 * 5)  Min Peak Height
 * 
 * These can be manipulated using getOptions() and setOptions(Option[]),
 * which are located within this file.  This class uses the FeatureListc
 * lass to retrieve the known functions. Once the options have been set, 
 * the readLane method can be called to read the actual file.
 *
 * The File Format
 * I got all my info from the "Staden Package" site, so you may want to
 * take a look if it's still up.  The address at the time of this writing
 * is http://www.mrc-lmb.cam.ac.uk/pubseq/manual/formats_unix_toc.html
 * 
 * There are several sections of a SCF file.  The first is the header,
 * which should be 128 bytes long.  It looks a little something like this:
 * 
 * unsigned 4-byte integer: magic number - should equal 779314022 (".scf")
 * unsigned 4-byte integer: number of samples
 * unsigned 4-byte integer: offset from the start of the file to the samples
 * unsigned 4-byte integer: number of bases - we don't care
 * unsigned 4-byte integer: bases left clip - we don't care
 * unsigned 4-byte integer: bases right clip - we don't care
 * unsigned 4-byte integer: bases offset - we don't care
 * unsigned 4-byte integer: comments size in bytes
 * unsigned 4-byte integer: offset from the start of the file to the comments
 * 4 1-byte characters: version - should by '3' '.' '0' '0'
 * unsigned 4-byte integer: sample size - 1=8bit samples, 2=16bit samples
 * unsigned 4-byte integer: code_set - we don't care
 * unsigned 4-byte integer: private size - we don't care
 * unsigned 4-byte integer: private offset - we don't care
 * unsigned 4-byte integer * 18: absolutely nothing
 * 
 * Now, onto the data area, which starts at "sampleOffset".  In SCF,
 * all four data's will be present, even if some of them are empty.  The 
 * structure of the entire file looks like this:
 * Header (128 bytes)
 * Data A (numberOfSamples * sampleSize)
 * Data C (numberOfSamples * sampleSize)
 * Data G (numberOfSamples * sampleSize)
 * Data T (numberOfSamples * sampleSize)
 * Offsets for bases (Number of bases * 4)(data in integers)
 * Accuracy estimate for A bases (Number of bases)(data in unsigned bytes)
 * Accuracy estimate for C bases (Number of bases)(data in unsigned bytes)
 * Accuracy estimate for G bases (Number of bases)(data in unsigned bytes)
 * Accuracy estimate for T bases (Number of bases)(data in unsigned bytes)
 * Reserved (Number of bases * 3)
 * Comments (commentsSize)(data in characters)
 * Private data (privateSize)
 * 
 * So, the data channel you want should be located at byte position:
 * dataOffset + sampleSize * numberOfSamples * numberOfChannel
 * 
 * However, with versions earlier than 3.00, the file structure is a little
 * different.
 * Header (128 Bytes)
 * Sample Structures (4 * samplesize * numberOfSamples)
 * Base Structures (12 * numberOfBases)
 * Comments (commentSize)
 * Private data (privateSize)
 * 
 * A Sample Structure looks like this 
 * Data A (samplesize)
 * Data C (samplesize)
 * Data G (samplesize)
 * Data T (samplesize)
 * 
 * That is, the difference in the data channels between SCF v1/2 and SCF v3 is the data in 
 * an SCF 1 or 2 file alternates between each base, whereas in SCF v3 all the samples for
 * A are stored, then C, then G, then T.
 * 
 * The data for 3.00 is also run under a pseudo-compression algorithm.  I say "pseudo" 
 * because the algorithm does not make the data any smaller, it just makes it 
 * easier to compress using other programs like tar, or zip.  Each value is only the 
 * difference of the difference between each sample.  If you look down in the 
 * "readLane" method, you can see the code that "uncompresses" this, which is quite 
 * simple.
 * 
 * int [] trace = new int[numberOfSamples];
 * int delta = 0, temp = 0;
 * for(int i = 0; i < numberOfSamples; i++)
 * {
 *     if(sampleSize == 1)
 *        temp = (int)inputFile.readByte();
 *     else
 *        temp = (int)inputFile.readShort();
 * 
 *     trace[i] = delta + temp;
 *     delta = trace[i];
 * }
 * delta = 0;
 * for(int i = 0; i < numberOfSamples; i++)
 * {
 *     trace[i] += delta;
 *     delta = trace[i];
 * }
 * 
 * If you get a negative value, it probably means that the channel you're looking
 * at is emtpy.  Take note that I haven't checked the example code above, and that 
 * it differs from the code that I actually used.  One of them should work.
 * 
 * Comments are stored in the format "VALU=STRING".  They are separated by '/n's.
 * For example, Comments = "LANE=1\nGELN=abc+ct+t102\nALeF=NULL" and so on.
 */

public class SCFFilter extends ImportFilter
{
  // Variables from parent class
  //private protected int filetype;        // the type, see constants above
  //private protected String name;         // the name of this filter
  //private protected String descript;     // a brief description
  //private protected File helpFile;       // represents the file that contains
                                           // the help info for this filter.
  // Used to indentify the different entries of interest in the ABI file and
  // store the index into an array that contains the info.

  /** color channel - note: these probably only correspond to the channels
   * of an SCF file in an arbitrary way.  Maybe not.*/
  public static final int YELLOW = 2;  /** color channel */
  public static final int RED = 3;     /** color channel */
  public static final int BLUE = 0;    /** color channel */
  public static final int GREEN = 1;
  public static final int ALL = 4;

  //Declaration and initialization.  The initialization values are arbitrary, 
  //based only on the settings that I usually use.
  
  private int colorChannel=2, stdColorChannel = 0;
  private String standardName;
  private SizeFunction sizeFn;
  private double minPeakWidth = 6, minPeakHeight = 50;
  
  // Just stores the name of the color channels, which are used in
  //several places in the filter.
  private String[] colors;
  private String[] subcolors;
  
  /**
   * CONSTRUCTOR for new SCFFilter.
   */
  public SCFFilter()
  {
    // Initialize the variables for this filter
    filetype = LANE;
    name = "SCF";
    descript = "Reads lane files from SCF, not gel files.";
    helpFile = "scf.html";
    
	// Options must be set.
    options = null; 

    standardName = "not set";
    sizeFn = null;
	
	colors = new String[5];
	colors[RED] = "Red (Channel 1)";
	colors[BLUE] = "Blue (Channel 3)";
	colors[GREEN] = "Green (Channel 2)";
	colors[YELLOW] = "Yellow (Channel 4)";
	colors[ALL] = "All Channels";
	
	subcolors = new String[4];
	subcolors[RED] = "Red (Channel 1)";
	subcolors[BLUE] = "Blue (Channel 3)";
	subcolors[GREEN] = "Green (Channel 2)";
	subcolors[YELLOW] = "Yellow (Channel 4)";	
 }

  /**
   * Access the name of the filter. 
   *
   * @return name of the import filter
   */
  public String getName()
  {
	  	  return name;
  }

  /**
   * Returns the type of input file supported by this filter In this case
   * <code>ImportFilter.LANE</code>, since the filter reads in lane data.
   *
   * @return constant LANE.
   */
  public int getFileType()
  {
	 return filetype;
  }

  /**
   * Retrieves a short, approximately one sentence, description of the filter.
   *
   * @return the description
   */
  public String getDescription()
  {
	return descript;
  }

  /**
   * The help file describes which files the filter reads and the options
   * that this filter accepts.
   *
   * @return File that contains the help information, either html or
   *   plaintext.
   */
  public String getHelpFile()
  {
	return helpFile;
  }

  /**
   * Returns the options for this filter.  
   * 1)Data Channel - selects one of four channels in an SCF file
   *	from which to retrieve the sample data.
   * 2)Standards Channel - selects one of four channels in the SCF
   *	file from which to retrieve the sample data for the 
   *	molecular weight standards.
   * 3)SizingFunction - selects which algorithm will be used to determine
   *	the size of the data points based on the locations of the
   *	molecular weight samples. 
   * 4)Standards - selects which set of standards are being used.  This
   *	is very important because this SCF filter must get the weights of
   *	the molecular weight standards from this standards set.  Standard
   *	sets should be entered manually into the standards.cfg file.
   * 5)Label - this just says "Peak detection parameters".  Peaks are detected
   *	in the standards trace data, and then correlated with the chosen
   *	Standards Set to create a set of benchmarks to compare the data with.
   * 6)MinPeakHeight - this determines the minimum height a point must be
   *	to be considered a peak.
   * 7)MinPeakWidth - this determines the minimum width a peak must be.
   * 
   * @return an array containing the options described above.
   *
   * @see Option
   * @see FeatureList
   * @see SizeFunction
   * @see SizeStandard
   */
  
  public Option[] getOptions()
  {
	Option[] returnOpts = new Option[8];
	
	//Declaring all the default values.
	String defColor = null;
	String defStdColor = null;
	String defSzMethod = null;
	String defSzStandard = null;
	int defPeakHeight = 50;
	int defPeakWidth = 6;
	
	/**This little group opens up "SCF.def", which should contain a set
	 * of default options that the user can save to.  This is just so
	 * that the user doesn't have to reenter the same values a billion
	 * times.  Since this is an unnecessary feature, if anything goes
	 * wrong, it just aborts the attempt and sets all the values back
	 * to nothing.
	 */
	try{
		FileReader F = new FileReader("SCF.def");
		BufferedReader defaultsFile = new BufferedReader(F);
		defColor = defaultsFile.readLine().trim();
		defStdColor = defaultsFile.readLine().trim();
		defSzMethod = defaultsFile.readLine().trim();
		defSzStandard = defaultsFile.readLine().trim();
		Double temp = new Double(defaultsFile.readLine().trim());
		defPeakHeight = (int)temp.doubleValue();
		temp = new Double(defaultsFile.readLine().trim());
		defPeakWidth = (int)temp.doubleValue();
		defaultsFile.close();
	}
	catch(Exception e)
	{
		defColor = null;
		defStdColor = null;
		defSzMethod = null;
		defSzStandard = null;
		defPeakHeight = 50;
		defPeakWidth = 6;
	}
	
	//Select Data Channel option
	Option param = new Option("Data Channel", Option.CHOICE, true, colors, defColor);
	returnOpts[0] = param;
	
	//Select Standards Data Channel option
	param = new Option("Standards Chan", Option.CHOICE, true, subcolors, defStdColor);
	returnOpts[1] = param;

    // The size function option, possiblities retrieved from the
    // feature list.
    param = new Option("Size Method", Option.CHOICE, true,
                 FeatureList.getSizeMgr().getNames(), defSzMethod);
	
    returnOpts[2] = param;

    // the size standards defined
    try
      {
     param = new Option("Size Standard", Option.CHOICE, true, 
                  FeatureList.getStandardMgr().getNames(), defSzStandard);
      }
    catch(IOException e)
      {
     throw new MissingParameterError("Error accessing standards file. " +
                         e.getMessage());
      }

    returnOpts[3] = param;
	
	//Just a label.  Does absolutely nothing except look nice.
	param = new Option("Peak Detection Parameters", Option.LABEL, true);
	returnOpts[4] = param;
	
	param = new Option("Min Peak Height", Option.NUMBER, true, defPeakHeight);
    returnOpts[5] = param;
	
	param = new Option("Min Peak Width", Option.NUMBER, true, defPeakWidth);
	returnOpts[6] = param;
	
	String [] saveOptions = new String [2];
	saveOptions[1] = "Save these settings";
	saveOptions[0] = "Use stored defaults";
	
	param = new Option("For next time... ", Option.CHOICE, true, saveOptions, "Use stored defaults");
	returnOpts[7] = param;
			
	return returnOpts;
  }

  /**
   * Sets the parameters for the filter to the specified values.  That is,
   * after the user clicks "OK" in the OptionsDialog window, this method
   * is called to convert all the users selections into variables that can
   * easily be read by the SCFFilter.  This portion also saves the selected
   * values to "SCF.def" to use as default values, should the user select 
   * that option.
   */
  public void setOptions(Option[] opts)
  {
	// Check the length.
    if(opts.length < 8)
      throw new IllegalArgumentException("Invalid options for SCF " +
                          "Filter. 8 options expected, but " +
                          opts.length + " were provided.");

    // extract the option
    String value = opts[0].getStringValue();

    // store the options
    options = opts;

    // check to make sure we have a string
    if (value == null)
      throw new MissingParameterError("Color not provided as parameter to " +
                          "SCF Filter.");
    if(value.equalsIgnoreCase("Red (Channel 1)"))
      colorChannel = RED;
    else if(value.equalsIgnoreCase("Blue (Channel 3)"))
      colorChannel = BLUE;
    else if(value.equalsIgnoreCase("Green (Channel 2)"))
      colorChannel = GREEN;
    else if(value.equalsIgnoreCase("Yellow (Channel 4)"))
      colorChannel = YELLOW;
	else if(value.equalsIgnoreCase("All Channels"))
	  colorChannel = ALL;
    else
      {
     // didn't match a color, so something is wrong.

     // set the options back to null since the ones we got were no good.
     options = null;

     // and complain
     throw new IllegalArgumentException("Invalid color specified for SCF" +
                            " Filter.");
      }

	// extract the option
    value = opts[1].getStringValue();

    // store the options
    options = opts;

    // check to make sure we have a string
    if (value == null)
      throw new MissingParameterError("Color not provided as parameter to " +
                          "SCF Filter.");
    if(value.equalsIgnoreCase("Red (Channel 1)"))
      stdColorChannel = RED;
    else if(value.equalsIgnoreCase("Blue (Channel 3)"))
      stdColorChannel = BLUE;
    else if(value.equalsIgnoreCase("Green (Channel 2)"))
      stdColorChannel = GREEN;
    else if(value.equalsIgnoreCase("Yellow (Channel 4)"))
      stdColorChannel = YELLOW;
    else
      {
     // didn't match a color, so something is wrong.
     // set the options back to null since the ones we got were no good.
     options = null;

     // and complain
     throw new IllegalArgumentException("Invalid color specified for SCF" +
                            " Filter.");
      }
	
	// Next should be the size function
    String sizeFnName = opts[2].getStringValue();
    try
      {
     sizeFn = (SizeFunction) FeatureList.getSizeMgr().get(sizeFnName);
      }
    catch(NoSuchElementException e)
      {
     options = null;
     throw new IllegalArgumentException("Invalid sizing function specified"
                            + " for SCF Filter. ");
      }
	
	//This one gets checked, later.
	standardName = opts[3].getStringValue();
	
	//There's not a whole lot that can go wrong with these.  If they're
	//not valid, though, the program throws a little exception fit.
	minPeakHeight = opts[5].getNumValue();
	minPeakWidth = opts[6].getNumValue() / 2;
	
	if(minPeakHeight <= 0 || minPeakWidth <= 0)
		throw new IllegalArgumentException("Peak Detection Parameters must be" + 
				" greater than zero!");
	
	/**This routine gets executed if the user wants his settings to be saved.
	 * Note that if there are any exceptions, it just throws a
	 * "PrettyInsignificantError", which is called that because the program
	 * just says the settings won't be saved, then goes about its business.
	 */
	if(opts[7].getStringValue().equals("Save these settings")){
	
		try{
			FileWriter F = new FileWriter("SCF.def");
			
			for(int i = 0; i < 4; i++){
				F.write(opts[i].getStringValue() + "\r\n");
			}
			
			Integer writeMe = new Integer((int)opts[5].getNumValue());
			F.write(writeMe.toString() + "\r\n");
			writeMe = new Integer((int)opts[6].getNumValue());
			F.write(writeMe.toString() + "\r\n");
			F.close();
		}
		catch(IOException e){
			throw new PrettyInsignificantError("Option Settings could not be saved." + 
						"  Sorry.");
		}
			
	}
		 
  }

  /**
   * This is the method that is called to preform the actual reading of the
   * file. The data in the file represents data from a single lane. The 
   * options/parameters required for the filter should be set using
   * <code>setOptions</code>, and if they are not, an exception will be 
   * thrown.
   *
   * @param inputFile  The file that contains the lane data.
   *
   * @return a Lane object with all of the appropriate information. 
   *
   * @exception MissingParameterError occurs if the options are not
   *     set. Since this includes the required color, the filter cannot
   *     read in the lane.
   * @exception IOException If an error is encountered in the file,
   *     then this exception will be thrown 
   */
  public Lane [] readLane(File inputFile) throws IOException
  {
	  
	double [] stdTrace;
	String [] comments;

    SizeStandard sizeStd;
    SizeFunction sizeFn;

    // Make sure we have options set, including the color channel
    if(options == null)
      throw new MissingParameterError("The options have not been set." +
                          "  The filter cannot work.");

    // Open the file. Set the mode to read only.
    RandomAccessFile in = new RandomAccessFile(inputFile, "r");

	// Check the file type. An SCF should start with ".scf", which
    // becomes 779314022 in decimal ((int)'.' << 24 + (int)'s' << 16.
    // + (int)'c' << 8 + (int)'f')
	int magicNum = in.readInt();
	if(magicNum != 779314022)
		throw new IOException("This does not appear to be a SCF file." + 
					" See help for more info.");
	
	//Now, it reads in the rest of the header and puts the data into
	//appropriately named variables.
	int traceSize = in.readInt();
	int dataOffset = in.readInt();
	in.seek(28);
	int commentSize = in.readInt();
	int commentOffset = in.readInt();
	String version = "";
	version += (char)in.readByte();
	version += (char)in.readByte();
	version += (char)in.readByte();
	version += (char)in.readByte();	
	Float ver = new Float(version);
	int ssize = in.readInt();
	
	//See parseComments(RandomAccessFile, int, int) method, at the bottom of
	//this file, for details.  Basically, it just reads all the useful comments
	//into a String array and returns it.
	comments = parseComments(in, commentSize, commentOffset);
	double delta = 0;
	short realChannel = 0;
	Lane [] laneArray;
	boolean allChannels;
	boolean flag;
	
	
	if(colorChannel == ALL){
		allChannels = true;
		laneArray = new Lane[4];
		for(int i = 0; i < 4; i++)
			laneArray[i] = null;
		colorChannel = -1;
	}
	else{
		allChannels = false;
		laneArray = new Lane[1];
	}
	
	//Procedure for obtaining standards channel data
	switch(stdColorChannel){
	case 0:
		realChannel = 2;
		break;
	case 1:
		realChannel = 1;
		break;
		case 2:
		realChannel = 3;
		break;
	case 3:
		realChannel = 0;
	}
		
	stdTrace = new double[traceSize];
	try{
		in.seek(realChannel * ssize * traceSize + dataOffset);
		
		for(int i = 0; i < traceSize; i++){
					
			if(ver.floatValue() < 3f)
				in.seek(dataOffset + realChannel * i * ssize);
			
			if(ssize == 1)
				stdTrace[i] = (double)((int)in.readByte());
			else
				stdTrace[i] = (double)((int)in.readShort());
		}
	}
	catch(IOException e){
		throw new IOException("Error reading file.  File may be corrupt.");
	}
	 
	if(ver.floatValue() == 3f){
		delta = 0;
		for(int i = 0; i < traceSize; i++){
			stdTrace[i] += delta;
		
			if(ssize == 1)
				stdTrace[i] %= 256;
			else
				stdTrace[i] %= 66536;
		
			delta = stdTrace[i];
		}
	
		delta = 0;
		double max = 0;
		for(int i = 0; i < traceSize; i++){
			stdTrace[i] += delta;
		
			if(ssize == 1)
				stdTrace[i] %= 256;
			else
				stdTrace[i] %= 66536;
	 
			delta = stdTrace[i];
			if(delta > max)
				max = delta;
		}
	}
	
	for(int i = 0; i < traceSize; i++)
		if(stdTrace[i] < 0)
			throw new IOException("Bad Data in Color Channel " + colors[stdColorChannel]);
		
	//Checking the Size Standard.  
	try{
		sizeStd = (SizeStandard) FeatureList.getStandardMgr().get(standardName);
	} catch(NoSuchElementException e){
		throw new IOException("Unknown size standard! '" + standardName + "'");
	}
		
	/**
	* Now that all the data has been collected, it must be analyzed.  Namely, the peaks
	* must be detected within the standards sample data.  All the data is placed in a 
	* DataList.  The peaks are found by running 
	* TracePeakFinder(double [], int, double, double) until peaks are found for every
	* molecular weight standard.
	*/
	DataList stdPoints = new DataList();
	int temp = 0;
	Peak pk;
	for(int i=0; i < sizeStd.getSize(); i++)
	   {
		temp = (int)TracePeakFinder.find(stdTrace, temp, minPeakHeight, minPeakWidth);
		
		if(temp != -1)
			stdPoints.addData(new Peak(sizeStd.getPeakLocation(i), stdTrace[temp], temp));
		else
			break;
	}
			
	for(int z = 0; z < 4; z++){
		
		double [] trace = new double[traceSize];
		//If we are inputing all the channels, we must find a non-empty channel
		if(allChannels){
			for(;;){
				colorChannel++;
				if(colorChannel > 3)
					break;
				if(colorChannel != stdColorChannel){
						
					switch(colorChannel){
					case 0:
						realChannel = 2;
						break;
					case 1:
						realChannel = 1;
						break;
					case 2:
						realChannel = 3;
						break;
					case 3:
						realChannel = 0;
					}
			
					//Finds the correct data channel, hopefully, and then reads
					//all the data out.  If something goes wrong, complains.
					try{
						in.seek(realChannel * traceSize * ssize + dataOffset);
								
						for(int i = 0; i < traceSize; i++){
						
							//If this is an earlier SCF file, before 3.00, the data
							//is stored differently.  This line is just the correction.
							if(ver.floatValue() < 3f)
								in.seek(dataOffset + realChannel * i * ssize);
												
							if(ssize == 1)
								trace[i] = (double)((int)in.readByte());
							else
								trace[i] = (double)((int)in.readShort());
						}
						//The data has been read in, but it needs to be decompressed if it's v3.00.  
						//There's more information about this at the top of this file.
						if(ver.floatValue() == 3f)
							unZip(trace, traceSize, ssize);
						
						flag = true;
						for(int i = 0; i < traceSize; i++){
							if(trace[i] < 0){
								flag = false;
								break;
							}
						}
						if(flag)
							break;
					}
					catch(IOException e){
						throw new IOException("Error reading file.  File may be corrupt.");
					}		
				}	
			}
		}
		
		else{
			//Unfortunately, the colors are assigned numbers throughout genographer
			//that don't quite match up with the channel numbers in SCF files, so a
			//quick conversion needs to take place before the reading can occur.
			switch(colorChannel){
			case 0:
				realChannel = 2;
				break;
			case 1:
				realChannel = 1;
				break;
			case 2:
				realChannel = 3;
				break;
			case 3:
				realChannel = 0;
			}
			
			//Finds the correct data channel, hopefully, and then reads
			//all the data out.  If something goes wrong, complains.
			try{
				in.seek(realChannel * traceSize * ssize + dataOffset);
								
				for(int i = 0; i < traceSize; i++){
						
					//If this is an earlier SCF file, before 3.00, the data
					//is stored differently.  This line is just the correction.
					if(ver.floatValue() < 3f)
						in.seek(dataOffset + realChannel * i * ssize);
										
					if(ssize == 1)
						trace[i] = (double)((int)in.readByte());
					else
						trace[i] = (double)((int)in.readShort());
					
				}
				
				//The data has been read in, but it needs to be decompressed if it's v3.00.  
				//There's more information about this at the top of this file
				if(ver.floatValue() == 3f)
						unZip(trace, traceSize, ssize);
					
				for(int i = 0; i < traceSize; i++)
					if(trace[i] < 0)
						throw new IOException("Bad Data in Color Channel " + 
								colors[colorChannel]);
			}
			catch(IOException e){
				throw new IOException("Error reading file.  File may be corrupt.");
			}		
		}			
		
		if(colorChannel > 3)
			break;
						
		//Now that all the data is collected and analyzed, an object of type Lane can
		//be created.  
		Integer laneNumber = new Integer(comments[1]);
		laneArray[z] = new Lane(trace);
			
		laneArray[z].setGelName(comments[2]);
		laneArray[z].setLaneNumber(laneNumber.intValue());
		laneArray[z].setName(comments[0]);
		laneArray[z].setColor(colorChannel);
		
		//================= set the size function ==============
	    String sizeName = options[2].getStringValue();
	    sizeFn = (SizeFunction) FeatureList.getSizeMgr().get(sizeName);
	    sizeFn = (SizeFunction) sizeFn.clone();    
	    sizeFn.init(stdPoints);
	    sizeFn.setMaxScan(laneArray[z].getNumPoints() - 1);
	    laneArray[z].setSizeFunction(sizeFn);
	    //==================clean up=============================
	    		
		if(!allChannels)
			break;
	}
	
	in.close();
	
	if(allChannels){
		for(int i = 0; i < 4; i++){
			if(laneArray[i] != null){
				allChannels = false;
				break;
			}
		}
		if(allChannels)
			throw new IOException("No data channel could be found");
	}
		
	return laneArray;
  }

  /**
   * This filter does not read gels.
   *
   * @return Always <code>null</code>
   */
  public Gel readGel(File inputFile) throws IOException
  {
    return null;
  }
  
  /**
   * Reads through the comments section of the file and returns the comments as an
   * array of strings.
   * 
   * Parameters:
   * @ifile: File that is being filtered
   * 
   * @commentSize: the size, in bytes, of the comments area
   * 
   * @commentOffset: the distance, in bytes, from the beginning of the file
   *	to the comments area
   */

  private String [] parseComments(RandomAccessFile ifile, int commentSize, int commentOffset)throws IOException
  {
  
	String [] realComments = new String[3];
	for(int i = 0; i < 2; i++)
			realComments[i] = "";
	
	if(commentSize == 0)
		return realComments;
	
	char temp;
	int count = 1;
	ifile.seek(commentOffset);
	String [] comments = null;
	
	try{
		//Comments are separated by a '\n', so these characters are
		//counted to determine the size of the comments array.
		for(int i = 0; i < commentSize; i++){
			if((char)ifile.readByte() == '\n')
				count++;
		}
	 
		comments = new String[count * 2];

		ifile.seek(commentOffset);
		count = 0;

		for(int i = 0; i < 2; i++)
			realComments[i] = "";
	
		for(int i = 0; i < comments.length; i++)
			comments[i] = "";
	  
		for(int i = 0; i < commentSize; i++){
			temp = (char)ifile.readByte();
			if(temp == '=' || temp == '\n')
				count++;
			else if(temp != 0)
				comments[count] += temp;
		}
	}
	catch(Exception e){
		throw new IOException("Error reading file Comments.  File could be corrupt.");
	}
	
	//There are only a few things that we are looking for, which are returned in
	//the separate array "realComments".
	for(int i = 0; i < comments.length; i+=2){
		if(comments[i].equalsIgnoreCase("GN"))
			realComments[2] = comments[i + 1];
		if(comments[i].equalsIgnoreCase("GELN"))
			realComments[2] = comments[i + 1];
		if(comments[i].equalsIgnoreCase("LANE"))
			realComments[1] = comments[i + 1];
		if(comments[i].equalsIgnoreCase("LN"))
			realComments[1] = comments[i + 1];
		if(comments[i].equalsIgnoreCase("NAME"))
			realComments[0] = comments[i + 1];	
	}
	
	return realComments;

  }
  
  private void unZip(double [] trace, int traceSize, int ssize){
	double delta = 0;
	for(int i = 0; i < traceSize; i++){
		trace[i] += delta;
	
		if(ssize == 1)
			trace[i] %= 256;
		else
			trace[i] %= 66536;
		delta = trace[i];
	}
	
	delta = 0;
	for(int i = 0; i < traceSize; i++){
		trace[i] += delta;
			
		if(ssize == 1)
			trace[i] %= 256;
		else
			trace[i] %= 66536;
			delta =	trace[i];
	}  
	 
  }

}
