//=====================================================================
// File:    ErrorDialog.java
// Class:   ErrorDialog
// Package: AFLPgui
//
// Author:  James J. Benham
// Date:    August 11, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPgui;

import java.awt.Button;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Label;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * This dialog can display errors, using the <code>showErrorM/code> method.
 * The message will be retrieved from the error and displayed to the user.
 * The user can print out a stack trace if one is desired. It will also
 * show a default message for a <code>NullPointerException</code> and
 * for an <code>OutOfMemoryError</code>.
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 11, 1998
 */

class ErrorDialog extends Dialog implements ActionListener
{
  private Label line1;
  private Label line2;
  private Button okButton;
  private Button detailsButton;

  private Throwable error;

  /**
   * Create a new ErrorDialog with the specified frame.
   *
   * @param parent  the owner of this dialog.
   */
  public ErrorDialog(Frame parent)
  {
    super(parent, "Error", true);

    setSize(400, 132);
    setLayout(null);
    setResizable(false);

    okButton = new Button("Ok");
    detailsButton = new Button("Stack Trace Dump.");

    okButton.addActionListener(this);
    detailsButton.addActionListener(this);

    add(okButton);
    add(detailsButton);

    okButton.setBounds(160, 100, 80, 22);
    detailsButton.setBounds(10, 30, 120, 22);

    line1 = new Label("");
    line2 = new Label("");

    add(line1);
    add(line2);

    line1.setBounds(10, 55, 380, 20);
    line2.setBounds(10, 75, 380, 20);
  }

  /**
   * Displays the given error message in the dialog box. The error
   * should have it's message set to something descriptive.
   * 
   * @param e  the error message
   */
  public void showError(Throwable e)
  {
    error = e;
    String message = e.getMessage();
    if(message == null)
      {
	line1.setText("Unknown Error. No message givin.");
	line2.setText("");
      }
    else if(message.length() > 70)
      {
	line1.setText(message.substring(0, 50));
	line2.setText(message.substring(50));
      }
    else
      {
	line1.setText(message);
	line2.setText("");
      }

    if(e instanceof OutOfMemoryError)
      {
	line1.setText("Out of Memory! Try zooming to a smaller percentage");
	line2.setText("or using fewer lanes. See help for how to increase " +
		      " the memory.");
      }

    if(e instanceof NullPointerException)
      {
	line2.setText("Null Pointer Error. See help for instructions.");
      }

    show();
  }

  /**
   * Handle the buttons. If the ok button is clicked, then the dialog
   * is dismissed. If the details button is clicked, a trace of the error
   * is dumped to the standard output stream.
   */
  public void actionPerformed(ActionEvent e)
  {
    if(e.getSource() == okButton)
      {
	setVisible(false);
	dispose();
      }
    else if(e.getSource() == detailsButton)
      {
	error.printStackTrace();
      }
  }
}

