#!/bin/csh

#Version  1/29/2007
# Run seqboot as a command
#Synopsis: seqboot.csh infile datatype rseed method replicates\
#              percent blocksize outweights outformat outfile

# Notes:
#   1. outformat must agree with the format of infile. If you want
#   seqboot to produce an interleaved file (outformat=i) infile
#   must be interleaved. If you want a sequential format file,
#   (outformat = s), then infile must be in sequential format.
#   2. Discrete state data (eg. rest sites) MUST be in 
#   sequential format.
#   3. If outweights = yes, weights are written to $OUTFILE
#      otherwise, sequences are written.

#Convert arguments to variables
set INFILE      = $1
set DATATYPE    = $2
set RSEED       = $3
set METHOD      = $4
set REPLICATES  = $5
set PERCENT     = $6
set BLOCKSIZE   = $7
set OUTWEIGHTS  = $8
set OUTFORMAT   = $9
set OUTFILE     = $10

# RM_CMD - command to be used for removing files and directories
if (-e /usr/bin/rm) then
   set RM_CMD = /usr/bin/rm
else
   if (-e /bin/rm) then
      set RM_CMD = /bin/rm
   else
      set RM_CMD = rm
   endif
endif

# Make a temporary directory to run the program in 
set TEMPDIR = SEQBOOT.$$
mkdir $TEMPDIR
cp $INFILE $TEMPDIR/infile
cd $TEMPDIR

#----------------- generate keyboard input to send to program -----

# Percent of characters to sample
if ( ( $PERCENT < 5 ) || ( $PERCENT > 100 ) ) set PERCENT = 100
echo '%' >> PARAMFILE
echo $PERCENT >> PARAMFILE


# Block size for resampling
if ( ( $BLOCKSIZE < 1 ) || ( $BLOCKSIZE > 100 ) ) set BLOCKSIZE = 1
if ( $BLOCKSIZE > 1 ) then
   echo 'B' >> PARAMFILE
   echo $BLOCKSIZE >> PARAMFILE
endif

# Choose datatype type
switch ($DATATYPE)
  case "s":  # Molecular sequences
    breaksw
  case "m":  # Discrete morphology or molecular markers
     echo d >> PARAMFILE
    breaksw
  case "r":  # Restriction sites
     echo d >> PARAMFILE
     echo d >> PARAMFILE
     echo e >> PARAMFILE # num. of enzymes is on  1st line of infile
    breaksw
  case "R": #  Restriction sites, no enzyme number in infile
     echo d >> PARAMFILE
     echo d >> PARAMFILE
    breaksw
  case "g": # Allele frequencies
     echo d >> PARAMFILE
     echo d >> PARAMFILE
     echo d >> PARAMFILE
    breaksw
endsw


# Choose resampling method
# Only bootstrapping and jacknifing can generate weights.
# Others must generate complete datasets
switch ($METHOD)
  case "b":                # bootstrap
    if ($OUTWEIGHTS == yes) echo s >> PARAMFILE
    breaksw
  case "d":                # delete-half jacknife
    echo j >> PARAMFILE
    if ($OUTWEIGHTS == yes) echo s >> PARAMFILE     
    breaksw
  case "ps":                # permute species for each character
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     set OUTWEIGHTS = no
    breaksw
  case "po":                # permute character order
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     set OUTWEIGHTS = no
    breaksw
  case "pw":                # permute within species
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     set OUTWEIGHTS = no
    breaksw
  case "rew":                # rewrite data
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE
     echo j >> PARAMFILE 
     set OUTWEIGHTS = no
    breaksw
  default :                
     set OUTWEIGHTS = no
    breaksw    
endsw


# Number of replicates
echo  r >> PARAMFILE
echo $REPLICATES >> PARAMFILE

# Output format: interleaved (default), sequential
if ($OUTFORMAT == 's') then
   echo  i >> PARAMFILE
endif

#accept current settings and do the analysis
echo y  >> PARAMFILE

# Random seed, odd, of the form 4n + 1
@ tempseed = ( ( ( $RSEED / 4 ) * 4 ) + 1 )
echo $tempseed >> PARAMFILE


#-------- Run seqboot, sending terminal output to /dev/null -----------
nice +10 seqboot < PARAMFILE > /dev/null

if ($OUTWEIGHTS == yes) then
   mv -f outweights ../$OUTFILE
else
   mv -f outfile ../$OUTFILE
endif

cd ..
$RM_CMD -rf $TEMPDIR






