/*
 * SeqDocTest.java
 * JUnit based test
 *
 * Created on March 15, 2010, 2:50 PM
 */

package org.biolegato.core.data.seqdoc;

import junit.framework.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.Arrays;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;
import java.util.LinkedList;
import java.util.Stack;
import org.biolegato.core.main.BLMain;

/**
 *
 * @author alvare
 */
public class SeqDocTest extends TestCase {
    final Sequence abc = new Sequence("ABC");
    final Sequence def = new Sequence("DEF");
    final Sequence ghi = new Sequence("GHI");
    final Sequence jkl = new Sequence("JKL");
    final Sequence mnopqr = new Sequence("MNOPQR");
    final Sequence stuvwxyz = new Sequence("STUVWXYZ");

    public SeqDocTest(String testName) {
	super(testName);
    }

    protected void setUp() throws Exception {
    }

    protected void tearDown() throws Exception {
    }

    public static Test suite() {
	TestSuite suite = new TestSuite(SeqDocTest.class);

	return suite;
    }

    public void testEmptyStringInsert() {
	final SeqDoc instance = new SeqDoc();

	// test insertions on empty Sequence document
	assertTrue(instance.insert(0, 0, ""));
	assertTrue(instance.insert(0, 0, (String) null));
	assertFalse(instance.insert(0, 0, "ABC"));
	assertFalse(instance.insert(1, 1, "ABC"));
	assertFalse(instance.insert(-1, -1, "ABC"));
    }

    public void testInsertSequence() {
	final SeqDoc instance = new SeqDoc();

	// adding a sequence to the start of the sequence document
	assertFalse(instance.insert(0, -1, new Sequence[] {abc}));
	assertEquals(0, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNull(instance.getLine(0));
	assertNull(instance.getLine(1));

	assertTrue(instance.insert(0, 0, new Sequence[] {def, ghi}));
	assertEquals(2, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertNull(instance.getLine(2));
	assertEquals(def, instance.getLine(0));
	assertEquals(ghi, instance.getLine(1));

	assertTrue(instance.insert(2, 0, new Sequence[] {jkl}));
	assertEquals(2, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertNull(instance.getLine(2));
	assertEquals("DEJKLF", instance.getLine(0).get("sequence"));
	assertEquals(ghi, instance.getLine(1));

	assertTrue(instance.insert(2, 1, new Sequence[] {mnopqr, abc}));
	assertEquals(3, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertNotNull(instance.getLine(2));
	assertNull(instance.getLine(3));
	assertEquals("DEJKLF", instance.getLine(0).get("sequence"));
	assertEquals("GHMNOPQRI", instance.getLine(1).get("sequence"));
	assertEquals("ABC", instance.getLine(2).get("sequence"));
    }

    public void testSingleLineNormalInsertString() {
	final SeqDoc instance = new SeqDoc(new Sequence[] {new Sequence("def")});

	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	
	assertTrue(instance.insert(0, 0, "123"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("123def", instance.getLine(0).get("sequence"));
	
	assertTrue(instance.insert(2, 0, "ABC"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));

	assertFalse(instance.insert(0, 2, "XYZ"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));

	assertFalse(instance.insert(100, 0, "XYZ"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));

	assertFalse(instance.insert(100, 0, "XYZ"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));
	
	assertFalse(instance.insert(-1, 0, "PQR"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));

	assertFalse(instance.insert(0, -1, "456"));
	assertNotNull(instance.getLine(0));
	assertEquals(1, instance.getLineCount());
	assertEquals("12ABC3def", instance.getLine(0).get("sequence"));
    }

    public void testDoubleLineNormalInsertString() {
	final SeqDoc instance = new SeqDoc(new Sequence[] {abc, def});
	
	assertTrue(instance.insert(0, 0, "123"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("123ABC", instance.getLine(0).get("sequence"));
	assertEquals("DEF", instance.getLine(1).get("sequence"));
	
	assertTrue(instance.insert(2, 0, "pqr"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("12pqr3ABC", instance.getLine(0).get("sequence"));
	assertEquals("DEF", instance.getLine(1).get("sequence"));

	assertTrue(instance.insert(0, 1, "XYZ"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("12pqr3ABC", instance.getLine(0).get("sequence"));
	assertEquals("XYZDEF", instance.getLine(1).get("sequence"));

	assertFalse(instance.insert(0, -1, "PQR"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("12pqr3ABC", instance.getLine(0).get("sequence"));
	assertEquals("XYZDEF", instance.getLine(1).get("sequence"));
	
	assertFalse(instance.insert(0, 3, "PQR"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("12pqr3ABC", instance.getLine(0).get("sequence"));
	assertEquals("XYZDEF", instance.getLine(1).get("sequence"));
	
	assertTrue(instance.insert(3, 1, "456"));
	assertEquals(2, instance.getLineCount());
	assertNotNull(instance.getLine(0));
	assertNotNull(instance.getLine(1));
	assertEquals("12pqr3ABC", instance.getLine(0).get("sequence"));
	assertEquals("XYZ456DEF", instance.getLine(1).get("sequence"));
    }

    /**
     * Test of delete method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testDeleteXYW() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{stuvwxyz});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{stuvwxyz, abc, def, ghi, mnopqr});

	assertEquals(0, instance0.getLineCount());
	assertNull(instance0.getLine(-1));
	assertNull(instance0.getLine(0));
	assertNull(instance0.getLine(1));
	
	assertTrue(instance0.delete(0, 0, 0));
	assertFalse(instance0.delete(-1, 0, 0));
	assertFalse(instance0.delete(0, -1, 0));
	assertFalse(instance0.delete(0, 0, -1));
	assertFalse(instance0.delete(-1, -1, -1));
	
	assertEquals(0, instance0.getLineCount());
	assertNull(instance0.getLine(-1));
	assertNull(instance0.getLine(0));
	assertNull(instance0.getLine(1));
	
	assertTrue(instance1.delete(2, 0, 0));
	assertTrue(instance1.delete(2, 0, 2));
	assertTrue(instance1.delete(3, 0, 1));
	
	assertEquals(1, instance1.getLineCount());
	assertNull(instance1.getLine(-1));
	assertNotNull(instance1.getLine(0));
	assertNull(instance1.getLine(1));
	assertEquals("STWYZ", instance1.getLine(0).get("sequence"));
    }

    /**
     * Test of delete method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testDeleteXYWH() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{stuvwxyz});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{stuvwxyz, abc, def, ghi, mnopqr});

	assertEquals(0, instance0.getLineCount());
	assertNull(instance0.getLine(-1));
	assertNull(instance0.getLine(0));
	assertNull(instance0.getLine(1));
	
	assertFalse(instance0.delete(0, 0, 0, 0));
	assertFalse(instance0.delete(-1, 0, 0, 0));
	assertFalse(instance0.delete(0, -1, 0, 0));
	assertFalse(instance0.delete(0, 0, -1, 0));
	assertFalse(instance0.delete(0, 0, 0, -1));
	assertFalse(instance0.delete(-1, -1, -1, -1));
	
	assertEquals(0, instance0.getLineCount());
	assertNull(instance0.getLine(-1));
	assertNull(instance0.getLine(0));
	assertNull(instance0.getLine(1));
	
	assertTrue(instance1.delete(2, 0, 0, 0));
	assertTrue(instance1.delete(2, 0, 2, 0));
	assertTrue(instance1.delete(3, 0, 1, 0));
	
	assertEquals(1, instance1.getLineCount());
	assertNull(instance1.getLine(-1));
	assertNotNull(instance1.getLine(0));
	assertNull(instance1.getLine(1));
	assertEquals("STWYZ", instance1.getLine(0).get("sequence"));
    }

    /**
     * Test of removeSequence method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testAddSequence() {
	final SeqDoc instance = new SeqDoc();

	// insert first sequence
	assertFalse(instance.addSequence(-1, abc));
	assertEquals(0, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNull(instance.getLine(0));
	assertNull(instance.getLine(1));
	
	assertTrue(instance.addSequence(0, abc));
	assertEquals(1, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(abc, instance.getLine(0));
	assertNull(instance.getLine(1));
	
	assertTrue(instance.addSequence(0, def));
	assertEquals(2, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertNull(instance.getLine(2));
	
	assertFalse(instance.addSequence(-1, ghi));
	assertEquals(2, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertNull(instance.getLine(2));
	
	assertTrue(instance.addSequence(2, ghi));
	assertEquals(3, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(ghi, instance.getLine(2));
	assertNull(instance.getLine(3));

	assertFalse(instance.addSequence(6, jkl));
	assertEquals(3, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(ghi, instance.getLine(2));
	assertNull(instance.getLine(3));
	
	assertTrue(instance.addSequence(2, jkl));
	assertEquals(4, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(jkl, instance.getLine(2));
	assertEquals(ghi, instance.getLine(3));
	assertNull(instance.getLine(4));
    }
	
    /**
     * Test of removeSequence method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testRemoveSequence() {
	final SeqDoc instance = new SeqDoc(new Sequence[]{def, abc, jkl, ghi});

	assertEquals(4, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(jkl, instance.getLine(2));
	assertEquals(ghi, instance.getLine(3));
	assertNull(instance.getLine(4));
	
	assertFalse(instance.removeSequence(-1));
	assertEquals(4, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(jkl, instance.getLine(2));
	assertEquals(ghi, instance.getLine(3));
	assertNull(instance.getLine(4));
	
	assertFalse(instance.removeSequence(100));
	assertEquals(4, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(abc, instance.getLine(1));
	assertEquals(jkl, instance.getLine(2));
	assertEquals(ghi, instance.getLine(3));
	assertNull(instance.getLine(4));
	
	assertTrue(instance.removeSequence(1));
	assertEquals(3, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(def, instance.getLine(0));
	assertEquals(jkl, instance.getLine(1));
	assertEquals(ghi, instance.getLine(2));
	assertNull(instance.getLine(3));
	
	assertTrue(instance.removeSequence(0));
	assertEquals(2, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(jkl, instance.getLine(0));
	assertEquals(ghi, instance.getLine(1));
	assertNull(instance.getLine(2));
	
	assertTrue(instance.removeSequence(1));
	assertEquals(1, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertEquals(jkl, instance.getLine(0));
	assertNull(instance.getLine(1));
	
	assertTrue(instance.removeSequence(0));
	assertEquals(0, instance.getLineCount());
	assertNull(instance.getLine(-1));
	assertNull(instance.getLine(0));
	assertNull(instance.getLine(1));
    }

    /**
     * Test of getLine method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetLine() {
	final SeqDoc instance0 = new SeqDoc();
	final SeqDoc instance1 = new SeqDoc(new Sequence[] {abc, def, ghi});

	assertNull(instance0.getLine(-1));
	assertNull(instance0.getLine(0));
	assertNull(instance0.getLine(1));

	assertNull(instance1.getLine(-1));
	assertEquals(abc, instance1.getLine(0));
	assertEquals(def, instance1.getLine(1));
	assertEquals(ghi, instance1.getLine(2));
	assertNull(instance1.getLine(3));
    }

    /**
     * Test of getLineLength method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetLineLength() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{stuvwxyz, abc, mnopqr, def, ghi, jkl});
	SeqDoc instance3 = new SeqDoc(new Sequence[]{abc, mnopqr, def, stuvwxyz, ghi, jkl});

	assertEquals(0, instance0.getLineLength(-1));
	assertEquals(0, instance0.getLineLength(0));
	assertEquals(0, instance0.getLineLength(1));
	
	assertEquals(0, instance1.getLineLength(-1));
	assertEquals(3, instance1.getLineLength(0));
	assertEquals(0, instance1.getLineLength(1));
	
	assertEquals(0, instance2.getLineLength(-1));
	assertEquals(8, instance2.getLineLength(0));
	assertEquals(3, instance2.getLineLength(1));
	assertEquals(6, instance2.getLineLength(2));
	assertEquals(3, instance2.getLineLength(3));
	assertEquals(3, instance2.getLineLength(4));
	assertEquals(3, instance2.getLineLength(5));
	assertEquals(0, instance2.getLineLength(6));

	assertEquals(0, instance3.getLineLength(-1));
	assertEquals(3, instance3.getLineLength(0));
	assertEquals(6, instance3.getLineLength(1));
	assertEquals(3, instance3.getLineLength(2));
	assertEquals(8, instance3.getLineLength(3));
	assertEquals(3, instance3.getLineLength(4));
	assertEquals(3, instance3.getLineLength(5));
	assertEquals(0, instance3.getLineLength(6));
    }

    /**
     * Test of getLineCount method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetLineCount() {
	System.out.println("getLineCount");

	final SeqDoc instance0 = new SeqDoc();
	final SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	final SeqDoc instance2 = new SeqDoc(new Sequence[]{abc, def});
	final SeqDoc instance3 = new SeqDoc(new Sequence[]{abc, def, ghi});

	assertEquals(0, instance0.getLineCount());
	assertEquals(1, instance1.getLineCount());
	assertEquals(2, instance2.getLineCount());
	assertEquals(3, instance3.getLineCount());
    }

    /**
     * Test of getLongestLine method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetLongestLine() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{abc, mnopqr, def, ghi, jkl});
	SeqDoc instance3 = new SeqDoc(new Sequence[]{abc, mnopqr, def, stuvwxyz, ghi, jkl});
	SeqDoc instance4 = new SeqDoc(new Sequence[]{mnopqr, def, ghi, jkl, stuvwxyz});
	SeqDoc instance5 = new SeqDoc(new Sequence[]{stuvwxyz, def, ghi, jkl, mnopqr});
	SeqDoc instance6 = new SeqDoc(new Sequence[]{stuvwxyz});

	assertEquals(0, instance0.getLongestLine());
	assertEquals(3, instance1.getLongestLine());
	assertEquals(6, instance2.getLongestLine());
	assertEquals(8, instance3.getLongestLine());
	assertEquals(8, instance4.getLongestLine());
	assertEquals(8, instance5.getLongestLine());
	assertEquals(8, instance6.getLongestLine());
    }

    /**
     * Test of indexOf method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testIndexOf() {
	SeqDoc instance0 = new SeqDoc(new Sequence[]{});
	SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{abc, mnopqr, def, ghi, jkl});
	SeqDoc instance3 = new SeqDoc(new Sequence[]{mnopqr, def, stuvwxyz, ghi, jkl});
	SeqDoc instance4 = new SeqDoc(new Sequence[]{mnopqr, ghi, jkl, abc, stuvwxyz, def});
	SeqDoc instance5 = new SeqDoc(new Sequence[]{abc, stuvwxyz, def, ghi, jkl, mnopqr});
	SeqDoc instance6 = new SeqDoc(new Sequence[]{stuvwxyz});

	assertEquals(-1, instance0.indexOf(abc));
	assertEquals(-1, instance0.indexOf(def));
	assertEquals(0, instance1.indexOf(abc));
	assertEquals(-1, instance1.indexOf(def));
	assertEquals(0, instance2.indexOf(abc));
	assertEquals(2, instance2.indexOf(def));
	assertEquals(-1, instance3.indexOf(abc));
	assertEquals(1, instance3.indexOf(def));
	assertEquals(3, instance4.indexOf(abc));
	assertEquals(5, instance4.indexOf(def));
	assertEquals(0, instance5.indexOf(abc));
	assertEquals(2, instance5.indexOf(def));
	assertEquals(-1, instance6.indexOf(abc));
	assertEquals(-1, instance6.indexOf(def));
    }

    /**
     * Test of getTransferDataFlavors method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testTransferDataFlavors() throws UnsupportedFlavorException, IOException {

	    SeqDoc instance0 = new SeqDoc();

	    assertTrue(instance0.isDataFlavorSupported(DataFlavor.stringFlavor));
	    assertTrue(instance0.isDataFlavorSupported(SeqDoc.seqDocFlavour));
	    
	    assertEquals(instance0, instance0.getTransferData(SeqDoc.seqDocFlavour));
	    assertEquals(instance0.toString(), instance0.getTransferData(DataFlavor.stringFlavor));
    }

    /**
     * Test of toArray method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testToArray() {
	final Sequence[] array = new Sequence[] {abc, def, ghi, jkl};
	final Sequence[] result = (new SeqDoc(array)).toArray();

	assertEquals(array.length, result.length);
	
	for (int count = 0; count < array.length; count++) {
	    assertEquals(array[count], result[count]);
	}
    }
}
