package org.biolegato.gdesupport.canvas;

/*
 * BLGDECanvas.java
 *
 * Created on December 3, 2007, 1:56 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.awt.Component;
import java.util.Stack;
import javax.swing.JPanel;
import javax.swing.SpringLayout;
import org.biolegato.core.properties.PropertiesListener;
import org.biolegato.gdesupport.canvas.colourmap.GDECharColourMap;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.gdesupport.canvas.list.GDEList;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import org.biolegato.core.main.BLMain;
import org.biolegato.core.plugintypes.DataCanvas;
import org.biolegato.gdesupport.canvas.textarea.BLTextArea;
import org.biolegato.gdesupport.canvas.textarea.GDETextArea;
import org.biolegato.gdesupport.canvas.textarea.UndoableGDETextArea;
import org.biolegato.gdesupport.canvas.undo.Undoable;

/**
 * The GDE-style sequence-based canvas class
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDECanvas extends DataCanvas implements CursorListener, FocusListener, ModeListener, PropertiesListener {

    /**
     * Whether to use the name list or the canvas for selections
     */
    private boolean useNameList = false;
    /**
     * Used for selecting genes
     */
    private GDEList nameList = new GDEList(this);
    /**
     * The text area for data manipulation
     */
    private GDETextArea dataCollector = null;
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622776154L;
    /**
     * The scroll pane containing the main canvas
     */
    private JScrollPane dataPane;
    /**
     * The scroll pane containing the split canvas
     */
    private JScrollPane altDataPane;
    /**
     * The box to store all split canvases
     */
    private Box canvasPane = new Box(BoxLayout.LINE_AXIS);
    /**
     * The split pane for split canvases
     */
    private JSplitPane splitPane;
    /**
     * Reference pointer to self
     */
    private final GDECanvas canvasSelf = this;
    /**
     * Stores the row and column position status
     */
    private final JLabel status = new JLabel("Row: 1 Col: 1");
    /**
     * Stores the insertion mode status
     */
    private final JLabel insertStatus = new JLabel("     ");
    /**
     * The stack of undoable events.
     * <p>
     *  Every time an event occurs within the BLTextArea that is undoable, an <b>undo object</b>
     *  is created and added to this stack.  This allows one to undo performed SeqDoc actions.
     *  To undo an action, pop the top object off of the stack and call its <b>undo method</b>.
     *  This will perform the undo and the <b>undo method</b> will return a <b>redo object</b>.
     *  The resulting <b>redo object</b> can then be added to the <b>redoStack</b> stack.
     * </p>
     */
    private Stack<Undoable> undoStack = new Stack<Undoable>();
    /**
     * The stack of redoable events
     * <p>
     *  Every time an event is undone, a redo object is created and added to this stack.
     *  A <b>redo object</b> is essentially an <b>undo object</b> created to "undo an undo".
     *  To perform a redo, pop the top object off the stact and call its <b>undo method</b>.
     *  This will perform the redo and the <b>undo method</b> will return an <b>undo object</b>.
     *  The resulting <b>undo object</b> can then be added to the <b>undoStack</b> stack.
     * </p>
     */
    private Stack<Undoable> redoStack = new Stack<Undoable>();
    /**
     * Action for splitting the canvas
     */
    private final AbstractAction splitAction = new AbstractAction("Split") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    GDETextArea altDataCollector = null;
	    
	    if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
		altDataCollector = new UndoableGDETextArea(canvasSelf);
	    } else {
		altDataCollector = new GDETextArea();
	    }
	    
	    canvasPane.remove(dataPane);
	    
	    altDataPane = new JScrollPane(altDataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
	    splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, dataPane, altDataPane);
	    splitPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
	    splitPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);
	    canvasPane.add(splitPane);
	    
            dataCollector.removePopupMenuItem(splitMenuItem);
            dataCollector.addPopupMenuItem(joinMenuItem);
	    
            altDataCollector.addPopupMenuItem(new JMenuItem(joinAction));
        }

    };
    /**
     * Action for joining split canvases
     */
    public final AbstractAction joinAction = new AbstractAction("Join") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    if (splitPane != null) {
		canvasPane.remove(splitPane);
		splitPane = null;
		altDataPane = null;
		canvasPane.add(dataPane);
	    }
	    
            dataCollector.removePopupMenuItem(joinMenuItem);
            dataCollector.addPopupMenuItem(splitMenuItem);
        }

    };
    /**
     * The action "Cut"
     */
    public final AbstractAction cutAction = new AbstractAction("Cut") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.cut();
        }

    };
    /**
     * The action "Copy"
     */
    public final AbstractAction copyAction = new AbstractAction("Copy") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.copy();
        }

    };
    /**
     * The action "Paste"
     */
    public final AbstractAction pasteAction = new AbstractAction("Paste") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.paste();
        }

    };
    /**
     * The action "Undo"
     */
    public final AbstractAction undoAction = new AbstractAction("Undo") {

        private static final long serialVersionUID = 7526472295622777038L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_U));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.undo();
        }

    };
    /**
     * The action "Redo"
     */
    public final AbstractAction redoAction = new AbstractAction("Redo") {

        private static final long serialVersionUID = 7526472295622777039L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_R));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.redo();
        }

    };
    /**
     * The action "Change Case"
     */
    public final AbstractAction changeCaseAction = new AbstractAction("Change case") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_E));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.changeCase();
        }

    };
    /**
     * The action "Select All"
     */
    public final AbstractAction selectAllAction = new AbstractAction("Select All") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.selectAll();
        }

    };
    /**
     * Action for grouping sequences in the canvas
     */
    public final AbstractAction groupAction = new AbstractAction("Group") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    Sequence.group(getData());
        }

    };
    /**
     * Action for ungrouping sequences in the canvas
     */
    public final AbstractAction ungroupAction = new AbstractAction("Ungroup") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    Sequence.ungroup(getData());
        }

    };
    /**
     * The menu item "Select sequence by name"
     */
    private final JMenuItem selectByNameMenuItem = new JMenuItem(nameList.selectByNameAction);
    /**
     * Action for splitting the canvas
     */
    private final JMenuItem splitMenuItem = new JMenuItem(splitAction);
    /**
     * The menu item for joining split canvases
     */
    private final JMenuItem joinMenuItem = new JMenuItem(joinAction);
    /**
     * The menu itemaction "Cut"
     */
    private final JMenuItem cutMenuItem = new JMenuItem(cutAction);
    /**
     * The menu item "Copy"
     */
    private final JMenuItem copyMenuItem = new JMenuItem(copyAction);
    /**
     * The menu item "Paste"
     */
    private final JMenuItem pasteMenuItem = new JMenuItem(pasteAction);
    /**
     * The menu item "Undo"
     */
    private final JMenuItem undoMenuItem = new JMenuItem(undoAction);
    /**
     * The menu item "Redo"
     */
    private final JMenuItem redoMenuItem = new JMenuItem(redoAction);
    /**
     * The menu item "Change Case"
     */
    private final JMenuItem changeCaseMenuItem = new JMenuItem(changeCaseAction);
    /**
     * The menu item "Select group"
     */
    private final JMenuItem selectGroupMenuItem = new JMenuItem(nameList.selectGroupAction);
    /**
     * The menu item "Select All"
     */
    private final JMenuItem selectAllMenuItem = new JMenuItem(selectAllAction);
    /**
     * The menu item for grouping sequences in the canvas
     */
    private final JMenuItem groupMenuItem = new JMenuItem(groupAction);
    /**
     * The menu item for ungrouping sequences in the canvas
     */
    private final JMenuItem ungroupMenuItem = new JMenuItem(ungroupAction);
    /**
     * The menu item for "Get Info..."
     */
    private final JMenuItem getInfoMenuItem = new JMenuItem(nameList.getInfoAction);
    
    /**
     * Self-reference for inner classes.
     */
    public final GDECanvas gdeCanvasSelf = this;

    /**
     * Creates a new instance of BLGDECanvas
     */
    public GDECanvas () {
        super(BoxLayout.PAGE_AXIS);

	// create new objects
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    dataCollector = new UndoableGDETextArea(this);
	} else {
	    dataCollector = new GDETextArea();
	}
	
        // setup listeners
	dataCollector.addFocusListener(this);
        dataCollector.addCursorListener(this);
        dataCollector.addModeListener(this);
	nameList.addFocusListener(this);

        BLMain.addPropertiesListener("font.size", this);
        BLMain.addPropertiesListener("font.bold", this);
        
        // add items to the textarea's popup menu
        dataCollector.addPopupMenuItem(splitMenuItem);

        // add items to the sequence list's popup menu
	nameList.addPopupMenuItem(new JMenuItem(groupAction));
	nameList.addPopupMenuItem(new JMenuItem(ungroupAction));
	
        // create the scroll panes
        dataPane = new JScrollPane(dataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        JScrollPane listPane = new JScrollPane(nameList, JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
	dataPane.setPreferredSize(new Dimension(200, 150));
        listPane.setPreferredSize(new Dimension(100, 150));

        // make the vertical scroll bars the same for both scroll panes
        dataPane.setVerticalScrollBar(listPane.getVerticalScrollBar());

        // sets the fixed cell height for the list's items to the row height of the jtextarea
        updateFont();

        // add the dataPane to the canvasPane
        canvasPane.add(dataPane);

        // create the main split pane
        JSplitPane mainPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, listPane, canvasPane);
        mainPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
        mainPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);

        // create the status bar
        Box statusBar = new Box(BoxLayout.LINE_AXIS);
        statusBar.setAlignmentX(Box.LEFT_ALIGNMENT);
        statusBar.add(status);
        statusBar.add(insertStatus);

        // add the panes to the canvas
        add(mainPane, BorderLayout.PAGE_START);
        add(new JSeparator(SwingConstants.HORIZONTAL));
        add(statusBar, BorderLayout.PAGE_END);
    }

    /**
     * Refreshes the canvas on tab changes.
     * @param	event   used to determine whether or not to refresh
     */
    @Override
    public void stateChanged (ChangeEvent event) {
        Object source = event.getSource();
        if (source instanceof JTabbedPane && ((JTabbedPane) source).getSelectedComponent() == this) {
            BLMain.addMenuHeading(1, "Edit").insert(cutMenuItem, 0);
            BLMain.addMenuHeading(1, "Edit").insert(copyMenuItem, 1);
            BLMain.addMenuHeading(1, "Edit").insert(pasteMenuItem, 2);
	    if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
		BLMain.addMenuHeading(1, "Edit").insert(undoMenuItem, 3);
		BLMain.addMenuHeading(1, "Edit").insert(redoMenuItem, 4);
	    }
	    BLMain.addMenuHeading(1, "Edit").insert(selectGroupMenuItem, 5);
            BLMain.addMenuHeading(1, "Edit").insert(selectAllMenuItem, 6);
            BLMain.addMenuHeading(1, "Edit").insert(groupMenuItem, 7);
            BLMain.addMenuHeading(1, "Edit").insert(ungroupMenuItem, 8);
            BLMain.addMenuHeading(1, "Edit").insert(getInfoMenuItem, 9);
            BLMain.addMenuHeading(1, "Edit").insert(changeCaseMenuItem, 10);
            BLMain.addMenuHeading(1, "Edit").insert(selectByNameMenuItem, 11);
        } else {
            BLMain.addMenuHeading(1, "Edit").remove(cutMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(copyMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(pasteMenuItem);
	    if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
		BLMain.addMenuHeading(1, "Edit").remove(undoMenuItem);
		BLMain.addMenuHeading(1, "Edit").remove(redoMenuItem);
	    }
            BLMain.addMenuHeading(1, "Edit").remove(selectGroupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(selectAllMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(groupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(ungroupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(getInfoMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(changeCaseMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(selectByNameMenuItem);
        }
    }

    /**
     * Returns the current/selected data in the canvas.
     *
     * @return the current data for usage by commands
     */
    @Override
    public Sequence[] getData () {
        LinkedList<Sequence> lines = new LinkedList<Sequence>();
        Sequence[] result = new Sequence[0];

        // check whether to use the row(s) selected or the selection in the JTextArea
        if (useNameList) {
            // get the row(s) selected
	    result = nameList.getData();
        } else {
            // get the text selected
            result = dataCollector.getData();
        }
        return (result != null ? result : new Sequence[0]);
    }

    /**
     * Returns the name to display in the canvas tab for
     * @return  "GDE"
     */
    public String getTabName () {
        return "GDE";
    }

    /**
     * This function is used to determine whether to return the data in the
     * JTextArea or JList when getData is called.
     *
     * @param event the focus event to process
     */
    public void focusGained (FocusEvent event) {
        if (event.getComponent() instanceof GDEList) {
            useNameList = true;
        } else if (event.getComponent() instanceof BLTextArea) {
            useNameList = false;
        }
    }

    /**
     * This function is currently unused.
     *
     * @param event the focus event to process
     */
    public void focusLost (FocusEvent event) {
    }

    /**
     * Updates the font of the canvas to "newFont".
     */
    private void updateFont () {
        int fontsize = 12;
        String fontsizestr = BLMain.getProperty("font.size");
        if (fontsizestr != null) {
            try {
                fontsize = Integer.parseInt(fontsizestr);
            } catch (NumberFormatException nfe) {
                nfe.printStackTrace();
            }
        }
        Font currentFont = new Font("Lucida Sans Typewriter", Font.PLAIN, fontsize);
        
        if ("true".equalsIgnoreCase(BLMain.getProperty("font.bold"))) {
            dataCollector.setFont(currentFont.deriveFont(Font.BOLD));
        } else {
            dataCollector.setFont(currentFont);
        }
        nameList.setFont(currentFont);
    }

    /**
     * Copies content from the current Editable object to the clipboads.
     */
    public void copy () {
        if (useNameList) {
            nameList.copy();
        } else {
            dataCollector.copy();
        }
    }

    /**
     * Cuts content from the current Editable object to the clipboads.
     */
    public void cut () {
        if (useNameList) {
            nameList.cut();
        } else {
            dataCollector.cut();
        }
    }

    /**
     * Pastes the current clipboard into the current Editable object.
     */
    public void paste () {
        if (useNameList) {
            nameList.paste();
        } else {
            dataCollector.paste();
        }
    }

    /**
     * Selects all the sequences within the canvas.
     */
    public void selectAll () {
	nameList.selectAll();
	dataCollector.selectAll();
	nameList.requestFocus();
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    private void changeCase () {
        if (useNameList) {
            nameList.changeCase();
        } else {
            dataCollector.changeCase();
        }
    }

    /**
     * Receives cursor updates for use in the status bar.
     *
     * @param source the source of the cursor change.
     * @param column the new column of the cursor.
     * @param row the new row of the cursor.
     */
    public void cursorChange (BLTextArea source, int column, int row) {
        status.setText("Row: " + (row + 1) + " Col: " + (column + 1));
    }

    /**
     * Receives insertion mode change updates for use in the status bar.
     *
     * @param mode the new insertion mode status.
     */
    public void insertionMode (boolean mode) {
        if (mode) {
            insertStatus.setText("[INS]");
        } else {
            insertStatus.setText("     ");
        }
    }

    /**
     * Used to intercept font size changes (since the font size is stored in the properties)
     *
     * @param key the key of the property changed.
     * @param value the new value of the property.
     */
    public void propertiesUpdate(String key, String value) {
        if (key.toLowerCase().startsWith("font.")) {
            updateFont();
        }
    }

/////////////////////////
//*********************//
//* UNDO/REDO METHODS *//
//*********************//
/////////////////////////
    /**
     * Undoes data modification.
     *
     * @return whether or not the undo was successful.
     */
    public boolean undo () {
        boolean result =  ! undoStack.empty();  // the result of the undo.
        Undoable undone = null;                 // the redo object (to redo the operation).

        if (result) {
            undone = undoStack.pop().undo();
            if (undone != null) {
                redoStack.push(undone);
            }
        }
        return result;
    }

    /**
     * Redoes data modification.
     *
     * @return whether or not the redo was successful.
     */
    public boolean redo () {
        boolean result =  ! redoStack.empty();  // the result of the undo.
        Undoable redone = null;                 // the undo object (to undo the redone operation).

        if (result) {
            redone = redoStack.pop().undo();
            if (redone != null) {
                undoStack.push(redone);
            }
        }
        return result;
    }
    
    /**
     * Adds an undoable object to the undo stack.
     *
     * @param undo the object to add to the stack.
     */
    public void addUndo (Undoable undo) {
	int undomax = 0;
	
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    redoStack.clear();
	    undoStack.push(undo);

	    if (BLMain.getProperty("undo.size") != null && BLMain.testNumber(BLMain.getProperty("undo.size").toCharArray())) {
		try {
		    undomax = Integer.parseInt(BLMain.getProperty("undo.size"));
		    while (undomax > 0 && undoStack.size() > undomax) {
			undoStack.remove(0);
		    }
		} catch (Throwable nfe) {
		}
	    }
	}
    }
}
