/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.list;

import java.awt.event.ItemEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;
import java.util.LinkedList;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JTextField;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.colourmap.GDECharColourMap;
import org.biolegato.gdesupport.canvas.colourmask.ImportFileAction;
import org.biolegato.gdesupport.canvas.colourmap.ColourMap;

/**
 * A window for editing sequence properties.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDESequenceWindow extends JDialog implements ActionListener,
                                                          ItemListener {
    
    /**
     * The sequence associated with the window
     */
    private Sequence[] sequences;
    /**
     * The text area used to obtain the sequence name from
     */
    private JTextField name = new JTextField();
    /**
     * The checkbox used to modify the protection of the sequence's alignment gaps
     */
    private JCheckBox protectAlignmentCB = new JCheckBox("Alignment gaps");
    /**
     * The checkbox used to modify the protection of the sequence's ambiguous characters
     */
    private JCheckBox protectAmbiguousCB = new JCheckBox("Ambiguous characters");
    /**
     * The checkbox used to modify the protection of the sequence's unambiguous characters
     */
    private JCheckBox protectUnambiguousCB = new JCheckBox(
            "Unambiguous characters");
    /**
     * The combobox used for modifying the sequence's type
     */
    private JComboBox type = new JComboBox(new Object[] {Sequence.Type.DNA,
                                                         Sequence.Type.RNA,
                                                         Sequence.Type.PROTEIN,
                                                         Sequence.Type.TEXT});
    /**
     * The combobox used for modifying the sequence's direction
     */
    private JComboBox direction = new JComboBox(new Object[] {
                Sequence.Direction.FROM5TO3, Sequence.Direction.FROM3TO5});
    /**
     * The combobox used for modifying the sequence's topology
     */
    private JComboBox topology = new JComboBox(new Object[] {
                Sequence.Topology.LINEAR, Sequence.Topology.CIRCULAR});
    /**
     * The combobox used for modifying the sequence's strandedness
     */
    private JComboBox strandedness = new JComboBox(
            new Object[] {Sequence.Strandedness.SINGLE,
                          Sequence.Strandedness.MIXED,
                          Sequence.Strandedness.DOUBLE});
    /**
     * The colour mask selection component
     */
    private ImportFileAction mask;
    /**
     * The text to display in a combobox when more than one item is selected among the list of sequences
     */
    private static final String MULTIPLE_SELECTED = "<multiple selected...>";
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622777013L;

    /**
     * Creates a window for editing the properties of a given array of sequences.
     *
     * @param window the parent window.
     * @param sequences the sequences to edit.
     */
    public GDESequenceWindow (final JFrame window,
                              final Sequence[] sequences) {
        super(window, "Sequence properties");

        // transfer parameters to local class variables
        this.sequences = sequences;

	// set window default values
        Object currentMask = null;
	
	if (sequences.length >= 1) {
	    if (sequences.length == 1) {
		name.setText(sequences[0].get("name").toString());
	    }
	    
	    type.setSelectedItem(sequences[0].get("type"));
	    direction.setSelectedItem(sequences[0].get("direction"));
	    topology.setSelectedItem(sequences[0].get("topology"));
	    strandedness.setSelectedItem(sequences[0].get("strandedness"));
	    
	    if (sequences[0].get("mask") != null &&
		sequences[0].get("mask") instanceof ColourMap) {
		currentMask = (ColourMap) sequences[0].get("mask");
	    } else if (sequences[0].get("mask") != null && !"".equals(sequences[0].get("mask"))) {
		BLMain.error("current mask = " + sequences[0].get("mask") + "\twrong class: " + sequences[0].get("mask").getClass());
	    }
	}
	for (int count = 1; (!protectAlignmentCB.isSelected()
		    || !protectAmbiguousCB.isSelected()
		    || !protectUnambiguousCB.isSelected()) && count < sequences.length; count++) {
	    if (sequences[count] != null) {
		if (!MULTIPLE_SELECTED.equals(type.getSelectedItem())) {
		    if (type.getSelectedItem().equals(sequences[count].get("type"))) {
			type.setSelectedItem(sequences[count].get("type"));
		    } else {
			type.insertItemAt(MULTIPLE_SELECTED, 0);
			type.setSelectedItem(MULTIPLE_SELECTED);
		    }
		}
		
		if (!MULTIPLE_SELECTED.equals(direction.getSelectedItem())) {
		    if (direction.getSelectedItem().equals(sequences[count].get("direction"))) {
			direction.setSelectedItem(sequences[count].get("direction"));
		    } else {
			direction.insertItemAt(MULTIPLE_SELECTED, 0);
			direction.setSelectedItem(MULTIPLE_SELECTED);
		    }
		}
		
		if (!MULTIPLE_SELECTED.equals(topology.getSelectedItem())) {
		    if (topology.getSelectedItem().equals(sequences[count].get("topology"))) {
			topology.setSelectedItem(sequences[count].get("topology"));
		    } else {
			topology.insertItemAt(MULTIPLE_SELECTED, 0);
			topology.setSelectedItem(MULTIPLE_SELECTED);
		    }
		}
		
		if (!MULTIPLE_SELECTED.equals(strandedness.getSelectedItem())) {
		    if (strandedness.getSelectedItem().equals(sequences[count].get("strandedness"))) {
			strandedness.setSelectedItem(sequences[count].get("strandedness"));
		    } else {
			strandedness.insertItemAt(MULTIPLE_SELECTED, 0);
			strandedness.setSelectedItem(MULTIPLE_SELECTED);
		    }
		}
	    
		if (!protectAlignmentCB.isSelected() && Boolean.TRUE.equals(sequences[count].get("protect_align"))) {
		    protectAlignmentCB.setSelected(true);
		}
		if (!protectAmbiguousCB.isSelected() && Boolean.TRUE.equals(sequences[count].get("protect_ambig"))) {
		    protectAmbiguousCB.setSelected(true);
		}
		if (!protectUnambiguousCB.isSelected() && Boolean.TRUE.equals(sequences[count].get("protect_unambig"))) {
		    protectUnambiguousCB.setSelected(true);
		}
		
		if (!MULTIPLE_SELECTED.equals(currentMask)) {
		    if (currentMask == null && (sequences[count].get("mask") == null
			    || !(sequences[count].get("mask") instanceof ColourMap))) {
			currentMask = null;
		    } else if (sequences[count].get("mask") != null && currentMask != null
			    && currentMask.equals(sequences[count].get("mask"))) {
			currentMask = (ColourMap) sequences[count].get("mask");
		    } else {
			currentMask = MULTIPLE_SELECTED;
		    }
		}
	    }
	}
	
        // configure the windows's main box
        Box mainBox = new Box(BoxLayout.PAGE_AXIS);

        // configure the sequqnece's name box
        Box nameBox = new Box(BoxLayout.LINE_AXIS);
        nameBox.add(new JLabel("Name:"));
        nameBox.add(name);
        mainBox.add(nameBox);

        // configure the sequqnece's type box
        Box typeBox = new Box(BoxLayout.LINE_AXIS);
        typeBox.add(new JLabel("Type:"));
        typeBox.add(type);
        type.addItemListener(this);
        mainBox.add(typeBox);

        // configure the sequqnece's direction box
        Box directionBox = new Box(BoxLayout.LINE_AXIS);
        directionBox.add(new JLabel("Direction:"));
        directionBox.add(direction);
        mainBox.add(directionBox);

        // configure the sequqnece's topology box
        Box topologyBox = new Box(BoxLayout.LINE_AXIS);
        topologyBox.add(new JLabel("Topology:"));
        topologyBox.add(topology);
        mainBox.add(topologyBox);

        // configure the sequqnece's strandedness box
        Box strandednessBox = new Box(BoxLayout.LINE_AXIS);
        strandednessBox.add(new JLabel("Strandedness:"));
        strandednessBox.add(strandedness);
        mainBox.add(strandednessBox);

        // configure the sequence protection box
        Box protectionBox = new Box(BoxLayout.PAGE_AXIS);
        protectionBox.add(protectAlignmentCB);
        protectionBox.add(protectAmbiguousCB);
        protectionBox.add(protectUnambiguousCB);
        protectionBox.setBorder(BorderFactory.createTitledBorder(
                "Set character protections"));
        mainBox.add(protectionBox);

        // handle the colour mask selection widgets
        Box maskBox = new Box(BoxLayout.LINE_AXIS);
        mask = new ImportFileAction(this, currentMask);
        maskBox.add(new JLabel("Colour mask:"));
        maskBox.add(mask);
        mainBox.add(maskBox);

        // add the update button
        JButton update = new JButton("Update");
        update.setActionCommand("update");
        update.addActionListener(this);
        mainBox.add(update);

        // display the window
        itemStateChanged(new ItemEvent(type, type.getSelectedIndex(),
                                       type.getSelectedItem(),
                                       ItemEvent.SELECTED));

        // create the window
        add(mainBox);
        pack();
        setLocationRelativeTo(window);
        setVisible(true);
	setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
    }

    /**
     * Adds an action handler to update the sequence data based on the window's values.
     *
     * @param e this is used to confirm that the update button was pressed.
     */
    public void actionPerformed (ActionEvent e) {
	LinkedList<Sequence> group;
		
        if ("update".equals(e.getActionCommand())) {
	    for (Sequence seq : sequences) {
		if (!"".equals(name.getText())) {
		    seq.put("name", name.getText());
		}
		if (!MULTIPLE_SELECTED.equals(type.getSelectedItem())) {
		    seq.put("type", type.getSelectedItem());
		}
		if (!MULTIPLE_SELECTED.equals(direction.getSelectedItem())) {
		    seq.put("direction", direction.getSelectedItem());
		}
		if (!MULTIPLE_SELECTED.equals(topology.getSelectedItem())) {
		    seq.put("topology", topology.getSelectedItem());
		}
		if (!MULTIPLE_SELECTED.equals(strandedness.getSelectedItem())) {
		    seq.put("strandedness", strandedness.getSelectedItem());
		}
		if (mask.getSelectedItem() instanceof ColourMap) {
		    seq.put("mask", mask.getSelectedItem());
		} else if (!MULTIPLE_SELECTED.equals(mask.getSelectedItem())) {
		    seq.put("mask", null);
		}
		
		seq.put("protect_align", new Boolean(
			protectAlignmentCB.isSelected()));
		seq.put("protect_ambig", new Boolean(
			protectAmbiguousCB.isSelected()));
		seq.put("protect_unambig", new Boolean(
			protectUnambiguousCB.isSelected()));
		if (seq.get("group") instanceof Integer) {
		    group = Sequence.getgroup((Integer)seq.get("group"));
		    if (group != null) {
			for (Sequence groupseq : group) {
			    groupseq.put("protect_align", new Boolean(
				    protectAlignmentCB.isSelected()));
			    groupseq.put("protect_ambig", new Boolean(
				    protectAmbiguousCB.isSelected()));
			    groupseq.put("protect_unambig", new Boolean(
				    protectUnambiguousCB.isSelected()));
			}
		    }
		}
		dispose();
	    }
	}
    }

    /**
     * Handles type changes made (this includes greying out those parameters which are
     * not applicable to the current sequence type.
     *
     * @param e is currently ignored by the function.
     */
    public void itemStateChanged (ItemEvent e) {
        if (Sequence.Type.DNA.equals(type.getSelectedItem()) ||
		Sequence.Type.RNA.equals(type.getSelectedItem())) {
            direction.setEnabled(true);
            topology.setEnabled(true);
            strandedness.setEnabled(true);
            mask.setEnabled(true);
        } else if (Sequence.Type.PROTEIN.equals(type.getSelectedItem())) {
            direction.setEnabled(false);
            topology.setEnabled(false);
            strandedness.setEnabled(false);
            mask.setEnabled(true);
        } else if (Sequence.Type.TEXT.equals(type.getSelectedItem())) {
            direction.setEnabled(false);
            topology.setEnabled(false);
            strandedness.setEnabled(false);
            mask.setEnabled(false);
        }
    }

}
