/*
 * BLMain.java
 *
 * Created on November 7, 2007, 1:04 PM
 *
 * This is the file which contains all of the main classes, and functions for
 * running BioLegato.
 *
 * Current serializable number:
 * private static final long serialVersionUID = 7526472295622777040L;
 *
 * Released numbers:
 */
package org.biolegato.core.main;

import java.awt.Font;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import javax.swing.Box;
import javax.swing.BoxLayout;
import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.core.plugins.PluginLoader;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.plugintypes.DataCanvas;
import org.biolegato.core.plugintypes.DataFormat;
import org.biolegato.core.plugintypes.MenuType;
import org.biolegato.core.plugins.PluginWrapper;
import java.io.BufferedReader;
import java.io.File;
import java.util.Hashtable;
import javax.swing.AbstractAction;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.text.BadLocationException;
import org.biolegato.core.plugintypes.PropertiesExtension;
import org.biolegato.core.properties.BLProperties;
import org.biolegato.core.properties.PropertiesListener;

/**
 * The main program and generic function class.
 * <p>
 *  This class is used to generate the main window, do all startup processing,
 *  and run the program.  This class also contains most of the utility functions.
 * </p>
 *
 * @version 0.6.2 12-Feb-2010
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public final class BLMain {

/////////////////////////
//*********************//
//* PROGRAM CONSTANTS *//
//*********************//
/////////////////////////
    /**
     * This constant stores the program's name
     */
    public static final String NAME = "BioLegato";
    /**
     * This constant is set to the path of BioLegato.jar
     * The value of this constant determined at runtime.
     */
    public static final String PROGRAM_DIR =
                               new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).isDirectory()
                               ? BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()
                               : new File(
            new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).getParent()).getPath();
    /**
     * This constant is set to the path of where biolegato was run
     * The value of this constant determined at runtime.
     */
    public static final String CURRENT_DIR = System.getProperty("user.dir");
    /**
     * This constant is used to keep track of version changes.
     */
    public static final String VERSION = "0.7.0";
    /**
     * The default font for BioLegato
     */
    public static final Font DEFAULT_FONT = new Font("Monospaced", Font.PLAIN,
                                                     12);
    /**
     * The amount of time between clicks to be considered a double click.
     */
    public static final int DOUBLE_CLICK_TIME = 300;
    /**
     * This constant is used for Serialization
     */
    public static final long serialVersionUID = 7526472295622776147L;

/////////////////
//*************//
//* VARIABLES *//
//*************//
/////////////////
    /**
     * Reference to the main program window
     */
    private static JFrame window = null;
    /**
     * Stores the properties for BioLegato.  See BLProperties for a list of properties intrinsic to BioLegato.
     */
    private static BLProperties properties = null;
    /**
     * The main data container for BioLegato.  This stores all data and sequences used by the program.
     */
    private static SeqDoc mainSeqDoc = new SeqDoc();
    /**
     * The tab pane storing all of the canvases.
     * This is stored so as to be able to get the current canvas for retrieving
     * the current data selection.
     */
    private static JTabbedPane canvasPane = null;
    /**
     * Stores the main menu for the program
     */
    private static JMenuBar menu = null;
    /**
     * Stores all of the menu headings in BioLegato's main window.
     * This hashtable is used to add menu items to BioLegato's menu headings.
     * The key of the hashtable corresponds to the label of the menu heading.
     * The value of the hashtable is the menu heading's object.
     */
    private static Hashtable<String, JMenu> menuHeadings =
                                     new Hashtable<String, JMenu>();

/////////////////////////
//*********************//
//* STARTUP FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Starts up BioLegato.
     *
     * @param args the command line arguments for BioLegato.
     */
    public static void main (String[] args) {
//////////////////////////
//**********************//
//* FUNCTION VARIABLES *//
//**********************//
//////////////////////////
        DataCanvas currentCanvas = null;    // used for loading canvases (for testing proper object creation)
        MenuType currentMenuLoader = null;  // used for menu plugins

/////////////////////////
//*********************//
//* PROCESS ARGUMENTS *//
//*********************//
/////////////////////////
	
        // load the properties, plugins and file formats
	properties = new BLProperties();
        PluginLoader.loadPlugins(envreplace(getProperty("plugins")));
        processArgs(args);

        // load properties extensions
        for (PluginWrapper plugin : PluginLoader.getPlugins(
                PropertiesExtension.class)) {
            properties.addExtension((PropertiesExtension) plugin.create(new Class[] {
                        BLProperties.class}, new Object[] {properties}));
        }

////////////////////////
//********************//
//* CONFIGURE WINDOW *//
//********************//
////////////////////////
	canvasPane = new JTabbedPane();
	window = new JFrame(NAME);
	
	// add tabbed pane to window
	menu = new JMenuBar();
	Box displayBox = new Box(BoxLayout.PAGE_AXIS);
	window.setJMenuBar(menu);
	//JToolBar toolBar = new JToolBar();
	//toolBar.add(menu);
	//displayBox.add(toolBar);
	//displayBox.add(new JScrollPane(menu));
	//displayBox.add(canvasPane);
	//window.add(displayBox);
	window.add(canvasPane);
	
////////////
//********//
//* MENU *//
//********//
////////////
        //////////////////////////////////////
        //**********************************//
        //* ADD THE DEFAULT TOP MENU ITEMS *//
        //**********************************//
        //////////////////////////////////////

        /*************************
         * Add the "Open" button *
         *************************/
        addMenuItem("File", new JMenuItem(new AbstractAction("Open...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/



            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_O));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed (java.awt.event.ActionEvent evt) {           /* Event handler - open the file */
                JFileChooser openDialog = new JFileChooser();

                openDialog.setCurrentDirectory(new File(System.getProperty(
                        "user.dir")));
                openDialog.setAcceptAllFileFilterUsed(true);
                org.biolegato.core.plugintypes.DataFormat.addFormats(openDialog,
                                                                     getProperty("default.fileformat").toString());
                // if a file is selected, open it
                if (openDialog.showOpenDialog(window) ==
                    JFileChooser.APPROVE_OPTION &&
                    openDialog.getSelectedFile() != null &&
                    openDialog.getSelectedFile().exists() &&
                    openDialog.getSelectedFile().isFile()) {
                    try {
                        if (openDialog.getFileFilter() != null && ! openDialog.getFileFilter().equals(
                                openDialog.getAcceptAllFileFilter())) {
                            addData(
                                    ((org.biolegato.core.plugintypes.DataFormat) openDialog.getFileFilter()).readFile(openDialog.getSelectedFile()));
                        } else {
                            addData(
                                    org.biolegato.core.plugintypes.DataFormat.auto(openDialog.getSelectedFile()));
                        }
                    } catch (Throwable e) {
                        e.printStackTrace();
                    }
                }
            }

        }));

        /*******************************
         * Add the "Save As..." button *
         *******************************/
        addMenuItem("File", new JMenuItem(new AbstractAction("Save As...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/



            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - save the file */
                JFileChooser saveDialog = new JFileChooser();

                saveDialog.setCurrentDirectory(new File(System.getProperty(
                        "user.dir")));
                saveDialog.setAcceptAllFileFilterUsed(false);
                org.biolegato.core.plugintypes.DataFormat.addFormats(saveDialog,
                                                                     getProperty("default.fileformat").toString());
                // if a file is selected, save to it
                if (saveDialog.showSaveDialog(window) ==
                    JFileChooser.APPROVE_OPTION &&
                    saveDialog.getSelectedFile() != null &&
                    ( ! saveDialog.getSelectedFile().exists() ||
                     javax.swing.JOptionPane.showConfirmDialog(null,
                                                               "Overwrite existing file?", "Overwritehttp://www.harmonyhomestead.com/index_files/tip1.htm", javax.swing.JOptionPane.OK_CANCEL_OPTION,
                                                               javax.swing.JOptionPane.QUESTION_MESSAGE) != javax.swing.JOptionPane.CANCEL_OPTION)) {
                    // write the file
                    try {
                        ((org.biolegato.core.plugintypes.DataFormat) saveDialog.getFileFilter()).writeFile(saveDialog.getSelectedFile(), getAllData());
                    } catch (Throwable e) {
                        e.printStackTrace();
                    }
                }
            }

        }));

        ////////////////////////////////
        //****************************//
        //* READ IN THE CUSTOM MENUS *//
        //****************************//
        ////////////////////////////////

        /*************************************************************
         * ittearates through the Plugins searching for menu formats *
         *************************************************************/
        for (PluginWrapper plugin : PluginLoader.getPlugins(MenuType.class)) {
            plugin.smethod("loadMenu");
        }

        ///////////////////////////////////////////
        //***************************************//
        //* ADD THE DEFAULT TRAILING MENU ITEMS *//
        //***************************************//
        ///////////////////////////////////////////

        /**************************
         * Add the "About" button *
         **************************/
        addMenuItem("Help", new JMenuItem(new AbstractAction("About...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/



            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed (java.awt.event.ActionEvent evt) {
                aboutPopup();
            }

        }));	/* Event handler - open about it window */

        /*******************************
         * Add the "Properties" button *
         *******************************/
        addMenuItem("File", new JMenuItem(new AbstractAction("Properties...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/



            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - open canvas properties window */
                properties.showPropertiesWindow(window);
            }

        }));

        /*************************
         * Add the "Exit" button *
         *************************/
        addMenuItem("File", new JMenuItem(new AbstractAction("Exit") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/



            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_X));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed (java.awt.event.ActionEvent evt) {
                window.dispose();
            }

        }));	/* Event handler - exit the program */

////////////////
//************//
//* CANVASES *//
//************//
////////////////
        // Load the canvas panes.
        // Populates the tabbed pane with all plugin canvases
        for (PluginWrapper plugin : PluginLoader.getPlugins(
                org.biolegato.core.plugintypes.DataCanvas.class)) {
            // adds the valid canvas
            currentCanvas =
            (org.biolegato.core.plugintypes.DataCanvas) plugin.create();
            if (currentCanvas != null) {
                canvasPane.addChangeListener(currentCanvas);
                canvasPane.add(currentCanvas.getTabName(), currentCanvas);
                if (currentCanvas.getTabName() != null &&
                    currentCanvas.getTabName().equalsIgnoreCase(getProperty(
                        "default.canvas"))) {
                    canvasPane.setSelectedComponent(currentCanvas);
                }
            }
        }

///////////////
//***********//
//* DISPLAY *//
//***********//
///////////////

        // center and draw the frame on the screen
        window.pack();
        window.setVisible(true);
        window.setLocationRelativeTo(null);
        window.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    }

    /**
     * Parses the command arguments for flags and data.
     * Currently the following options are available
     * (note aliases are separated by commas, and commands my begin with --, - or /):
     *
     * <table border="1">
     *	<tr><th>Parameter</th>	    <th>Description</th></tr>
     *	<tr><td>help,?</td>         <td>Displays usage information for BioLegato</td></tr>
     *	<tr><td>manpage,man</td>    <td>Displays the manpage entry for BioLegato</td></tr>
     *	<tr><td>optionlist</td>	    <td>Displays the list of options for using BioLegato</td></tr>
     *	<tr><td>version,v</td>	    <td>Displays the version information for BioLegato</td></tr>
     *	<tr><td>pipe,p,in</td>	    <td>Inserts input from standard input into the canvas</td></tr>
     *	<tr><td>debug</td>          <td>Enables BioLegato's debug mode</td></tr>
     *	<tr><td>plugins</td>	    <td>Displays a list of loaded plugins</td></tr>
     *	<tr><td>properties</td>	    <td>Displays a list of BioLegato's properties</td></tr>
     * </table>
     *
     * @param args the command line arguments to parse
     */
    private static void processArgs (String[] args) {
        String value = "";	    // the value of the current argument parameter
        File fileIn = null;	    //
	String argument = null;	    //
        Sequence[] dataIn = null;   //

        // ensure that args is not null
        if (args != null) {
            // itterate through the command arguments
            for (String rawarg : args) {
                // discard null arguments
                if (rawarg != null) {
		    // create a new file object to test if the argument is a file
		    fileIn = new File(rawarg);

                    // if the argument is a file name then read the file; otherwise,
                    // parse the argument
                    if (fileIn != null && fileIn.exists() && fileIn.isFile() &&
                        fileIn.canRead()) {
                        addData(org.biolegato.core.plugintypes.DataFormat.auto(
                                fileIn));
                    } else if (rawarg.startsWith("/") || rawarg.startsWith(
                            "-")) {
			// copy the command line argument
			argument = rawarg;

                        // trim the argument
                        if (argument.startsWith("--") && argument.length() > 2) {
                            argument = argument.substring(2);
                        } else if ((argument.startsWith("/") ||
                                    argument.startsWith("-")) &&
                                   argument.length() > 1) {
                            argument = argument.substring(1);
                        }

                        // If there is an equals '=' sign then the argument has a value.  Set the value
                        // variable to the arguments' value for further parsing
                        if (argument.indexOf('=') > 0) {
                            value = argument.substring(argument.indexOf('=') + 1);
                            argument = argument.substring(0, argument.indexOf(
                                    '='));
                        }

                        // make the argument lower case for better parsing
                        argument = argument.toLowerCase().trim();

                        // process the argument
                        if ("help".equals(argument) || "h".equals(argument) ||
                            "?".equals(argument)) {
                            ////////////
                            //********//
                            //* HELP *//
                            //********//
                            ////////////
                            // show BioLegato's usage
                            System.out.println(
                                    "Usage: biolegato [options] [files]\n" +
                                    "Use --optionlist  to see a detailed list of options\n" +
                                    "Use --manpage     to see BioLegato's manpage");
                            System.exit(0);
                        } else if ("optionlist".equals(argument)) {
                            ///////////////////
                            //***************//
                            //* OPTION LIST *//
                            //***************//
                            ///////////////////
                            // show BioLegato's list of options
                            System.out.println(
                                    "+-------------+------------------------------------------------------+\n" +
                                    "| Option      | Description                                          |\n" +
                                    "+-------------+------------------------------------------------------+\n" +
                                    "| help,?      | Displays usage information for BioLegato             |\n" +
                                    "| manpage,man | Displays the manpage entry for BioLegato             |\n" +
                                    "| optionlist  | Displays the list of options for using BioLegato     |\n" +
                                    "| version,v   | Displays the version information for BioLegato       |\n" +
                                    "| pipe,p,in   | Inserts input from standard input into the canvas    |\n" +
                                    "| debug       | Enables BioLegato's debug mode                       |\n" +
                                    "| plugins     | Displays a list of loaded plugins                    |\n" +
                                    "| properties  | Displays a list of BioLegato's properties            |\n" +
                                    "+-------------+------------------------------------------------------+\n");
                            System.exit(0);
                        } else if ("manpage".equals(argument) ||
                                   "man".equals(argument)) {
                            ///////////////
                            //***********//
                            //* MANPAGE *//
                            //***********//
                            ///////////////
                            // show BioLegato's usage
                            System.out.println(
                                    "NAME\n" +
                                    "    Bio Legato - A customizable GUI for running programs\n" +
                                    "\n" +
                                    "VERSION\n" +
                                    "    Version " + VERSION + "\n" +
                                    "\n" +
                                    "SYNOPSIS\n" +
                                    "    biolegato [options] [files]\n" +
                                    "\n" +
                                    "DESCRIPTION\n" +
                                    "    BioLegato is a customizable GUI for running programs.\n" +
                                    "    Its initial intent is to be a replacement for GDE; however\n" +
                                    "    with its large plugin API, it may be customized to run more\n" +
                                    "    than just CLI programs.\n" +
                                    "\n" +
                                    "    With the versitility of BioLegato's plugin interface, it\n" +
                                    "    supports a wide range of file formats which may be added to\n" +
                                    "    at any time through addition of plugins.\n" +
                                    "\n" +
                                    "OPTIONS\n" +
                                    "  NOTE: All command line parameters are case insensitive\n" +
                                    "        and may optionally begin with -, --, or / to prevent\n" +
                                    "        confusion with filenames\n" +
                                    "\n" +
                                    "    help,h,?\n" +
                                    "        Displays usage information for BioLegato\n" +
                                    "    manpage,man\n" +
                                    "        Displays the manpage entry for BioLegato (this screen)\n" +
                                    "    optionlist\n" +
                                    "        Displays the list of options for using BioLegato\n" +
                                    "    version,v\n" +
                                    "        Displays the version information for BioLegato\n" +
                                    "    pipe,p,in\n" +
                                    "        Inserts input from standard input into the canvas\n" +
                                    "    debug\n" +
                                    "        Enables BioLegato's debug mode\n" +
                                    "    plugins\n" +
                                    "        Displays a list of loaded plugins\n" +
                                    "    properties\n" +
                                    "        Displays a list of BioLegato's properties\n" +
				    "\n" +
                                    "USAGE EXAMPLES\n" +
                                    "    biolegato\n" +
                                    "\n" +
                                    "    biolegato insequence.gb\n" +
                                    "\n" +
                                    "    biolegato --debug\n" +
                                    "    biolegato /debug\n" +
                                    "    biolegato -debug\n" +
                                    "\n" +
                                    "    biolegato --plugins --properties --debug\n" +
                                    "\n" +
                                    "    biolegato --plugins /properties -debug insequence.gb\n" +
                                    "\n" +
				    "ENVIRONMENT VARIABLES\n" +
				    "    BioLegato searches for the following environment variables.  If they don't exist BioLegato sets them to defaults.\n" +
				    "\n" +
				    "      Variable             Default value\n" +
				    "        BIOLEGATO_HOME       <the path of the current working directory>\n" +
				    "\n" +
				    "PROPERTIES\n" +
				    "    BioLegato supports the followind properties:\n" +
				    "\n" +
				    "      Property             Description\n" +
				    "        debug                determines whether or not to display debug information\n" +
				    "        plugins              sets the plugin directory\n" +
				    "        temp                 sets the temporary files directory\n" +
				    "        font.size            determines the default font size for objects in BioLegato\n" +
				    "        font.type            determines the default font type for objects in BioLegato\n" +
				    "        shell.name           determines the shell to execute\n" +
				    "        shell.parameter      determines the parameter for directing the shell to execute a command\n" +
				    "        pwd.properties       determines whether or not to read the properties file located in the current working directory\n" +
				    "        user.properties      determines whether or not to read the user's propery files (located in the user's home directory\n" +
				    "        user.plugins         determines whether or not to read the user's plugin files (located in the user's home directory\n" +
				    "        default.canvas       determines the default canvas to show on startup\n" +
				    "        default.fileformat   determines the default file format for open/save dialogs\n" +
				    "        GDE.menu             determines whether or not to read the GDE menu files (backwards compatibility mode)\n" +
				    "        GDE.help.viewer      determines which file to feed the help file to\n" +
				    "        GDE.help.path        the location to search for the help files\n" +
				    "        undo                 whether or not to enable undo support\n" +
				    "        undo.size            whether or not to enable undo support\n" +
				    "\n" +
				    "    This properties class will read properties files from the directory containing BioLegato, the user directory and finally\n" +
				    "    the directory BioLegato was launched from.  Please note that you can disable reading of properties files\n" +
				    "    from the user directory via. \"user.properties\"; likewise you can disable reading properties files in the directory BioLegato\n" +
				    "    was launched from via. \"pwd.properties\"" +
				    "\n" +
				    "    NOTE: for path properties BioLegato will replace all $'s with the appropriate environment variables if set.\n" +
                                    "\n" +
                                    "SUPPORTED PLUGINS\n" +
                                    "    Currently BioLegato supports the following main types of Plugins:\n" +
                                    "\n" +
                                    "        Canvases\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.DataCanvas\n" +
                                    "        File formats\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.DataFormat\n" +
                                    "        Menu types\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.MenuType\n" +
                                    "        Properties\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.PropertiesExtension\n" +
                                    "\n" +
                                    "    For more information about plugins, please consult the BioLegato API\n" +
                                    "\n" +
                                    "FILES\n" +
                                    "    biolegato\n" +
                                    "        Script to run BioLegato\n" +
                                    "    biolegato.jar\n" +
                                    "        BioLegato's core java code\n" +
                                    "    changelog.txt\n" +
                                    "        BioLegato's revision history\n" +
                                    "    plugins/GDE_Menu.jar" +
                                    "        Provides backward compatibility with GDE menus\n" +
                                    "    plugins/GDE_anvas.jar" +
                                    "        Provides GDE style multiple alignment canvas\n" +
                                    "    plugins/GDEFile.class" +
                                    "        Provides internal file support for GDE files\n" +
                                    "    plugins/GDEFlatFile.class" +
                                    "        Provides internal file support for GDE flatfiles\n" +
                                    "    plugins/GenBankFile2008.class" +
                                    "        Provides internal file support for GenBank files (2008 standard copliant)\n" +
                                    "    plugins/FastA.class" +
                                    "        Provides internal file support for FastA files\n" +
                                    "\n" +
                                    "FILE FORMATS\n" +
                                    "    All file formats in BioLegato are supported through plugins.\n" +
                                    "    Below is a list of file formats supported by a default\n" +
                                    "    installation of BioLegato (with all standard plugins):\n" +
                                    "\n" +
                                    "        BioLegato flatfiles\n" +
                                    "        FastA files\n" +
                                    "        GDE flatfiles\n" +
                                    "        GDE format files\n" +
                                    "        GenBank files (2008 standard compliant)\n" +
                                    "\n" +
                                    "BUGS\n" +
                                    "    There are currently no known bugs in this version of BioLegato\n" +
                                    "\n" +
                                    "    Please report all bugs to: Graham Alvare <alvare@cc.umanitoba.ca>" +
                                    "\n" +
                                    "AUTHORS\n" +
                                    "    Dr. Brian Fristensky\n" +
                                    "    Department of Plant Science\n" +
                                    "    University of Manitoba\n" +
                                    "    Winnipeg, MB  Canada R3T 2N2\n" +
                                    "\n" +
                                    "    Email: frist@cc.umanitoba.ca\n" +
                                    "    Web: http://home.cc.umanitoba.ca/~frist\n" +
                                    "\n" +
                                    "    Graham Alvare\n" +
                                    "    Department of Plant Science\n" +
                                    "    University of Manitoba\n" +
                                    "    Winnipeg, MB  Canada R3T 2N2\n" +
                                    "\n" +
                                    "    Email: alvare@cc.umanitoba.ca\n" +
                                    "    Web: http://home.cc.umanitoba.ca/~alvare\n");
                            System.exit(0);
                        } else if ("version".equals(argument) ||
                                   "v".equals(argument)) {
                            ///////////////
                            //***********//
                            //* VERSION *//
                            //***********//
                            ///////////////
                            // show BioLegato's version
                            System.out.println("BioLegato v" + VERSION);
                            System.exit(0);
                        } else if ("pipe".equals(argument) || "p".equals(argument)
				|| "in".equals(argument)) {
                            ////////////
                            //********//
                            //* PIPE *//
                            //********//
                            ////////////
                            // allow pipelining from stdin
                            try {
                                addData(org.biolegato.core.plugintypes.DataFormat.auto(
                                        System.in));
                            } catch (Throwable ex) {
                                BLMain.error(
                                        "An error occurred reading canvas data from System.in",
                                             "processArgs");
                            }
                        } else if ("debug".equals(argument)) {
                            /////////////
                            //*********//
                            //* DEBUG *//
                            //*********//
                            /////////////
                            // enable debug information
                            properties.setProperty("debug", "true");
                        } else if ("plugins".equals(argument)) {
                            ///////////////
                            //***********//
                            //* PLUGINS *//
                            //***********//
                            ///////////////
                            // show BioLegato's loaded plugins
                            System.out.println("(Current plugins path: " +
                                               getProperty("plugins") + ")");
                            System.out.println("-- listing plugins loaded --");
                            String[] pluginList =
                                     PluginLoader.getPluginHash().keySet().toArray(
                                    new String[0]);
                            for (String pluginName : pluginList) {
                                System.out.println("Plugin: " + pluginName);
                            }
                            System.out.println("-- end of plugin list --");
                        } else if ("properties".equals(argument)) {
                            System.out.println("**********");
                            System.out.println("PROPERTIES");
                            System.out.println("**********");
                            //////////////////
                            //**************//
                            //* PROPERTIES *//
                            //**************//
                            //////////////////
                            // show BioLegato's current settings
                            properties.list(System.out);
                        } else {
                            BLMain.error("Unknown argument: " + rawarg,
                                         "processArgs");
                        }
                    }
                }
            }
        }
        // if debug is set, display the command
        if ("true".equalsIgnoreCase(getProperty("debug"))) {
            BLMain.message("Command line arguments read successfully",
                           "processArgs");
        }
    }

////////////////////////////
//************************//
//* PROPERTIES FUNCTIONS *//
//************************//
////////////////////////////
    /**
     * Changes properties for BioLegato.
     *
     * @param key the property to change.
     * @param value the new value for the property.
     */
    public static void setProperty (String key, String value) {
        properties.setProperty(key, value);
    }

    /**
     * Retrieves individual settings for BioLegato.
     * This is used to obtain values of properties in BioLeagato
     *
     * @param property the property key to retrieve the value for
     * @return the property value corresponding to the key parameter
     */
    public static String getProperty (String property) {
        return properties.getProperty(property);
    }
    
//////////////////////
//******************//
//* DATA FUNCTIONS *//
//******************//
//////////////////////
    /**
     * Obtains the current contents of the clipboard (null if empty).
     *
     * @return the current contents of the clipboard.
     */
    public static Sequence[] getClipboard () {
	String string = "";
	Sequence[] clipboard = null;
	Sequence[] result = null;
        Transferable t = Toolkit.getDefaultToolkit().getSystemClipboard().getContents(null);
    
        try {
            if (t != null) {
		if (t.isDataFlavorSupported(SeqDoc.seqDocFlavour)) {
		    clipboard = ((SeqDoc)t.getTransferData(SeqDoc.seqDocFlavour)).toArray();
		    
		    // clone clipboard sequences
		    result = new Sequence[clipboard.length];
		    for (int count = 0; count < clipboard.length; count++) {
			result[count] = new Sequence(clipboard[count]);
		    }
		} else if (t.isDataFlavorSupported(DataFlavor.stringFlavor)) {
		    string = (String)t.getTransferData(DataFlavor.stringFlavor);
		    result = DataFormat.auto(new BufferedReader(new StringReader(string)));
		}
            }
        } catch (UnsupportedFlavorException e) {
        } catch (IOException e) {
        }
	return result;
    }
    
    /**
     * Changes the current contents of the clipboard.
     *
     * @param copy the new content for the clipboard.
     */
    public static void setClipboard (Sequence[] copy) {
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new SeqDoc(copy), null);
    }
    
    /**
     * Returns the root sequence document for BioLegato.
     *
     * @return the root sequence document.
     */
    public static SeqDoc getSeqDoc () {
        return mainSeqDoc;
    }

    /**
     * Returns the current/selected data in the canvas.
     *
     * @return the current data for usage by commands
     * @throws BadLocationException any exceiptions related to obtaining the data
     */
    public static Sequence[] getData () throws BadLocationException {
        return ((org.biolegato.core.plugintypes.DataCanvas) canvasPane.getSelectedComponent()).getData();
    }

    /**
     * Returns the complete contents of the data in the canvas.
     *
     * @return the complete contents of the data in the canvas
     * @throws BadLocationException any location exceptions thrown by the underlying Document class
     */
    public static Sequence[] getAllData () throws BadLocationException {
        return getSeqDoc().toArray();
    }

    /**
     * Adds the data to the canvas.
     *
     * @param dataAdd the data to add to the canvas.
     * @throws BadLocationException any exceptions thrown by the underlying document class
     */
    public static void addData (Sequence[] dataAdd) {
        // ensure that the array to add data from is not a null object
        if (dataAdd != null) {
	    // add the sequences to the sequence document
	    for (Sequence seq : dataAdd) {
		getSeqDoc().addSequence(getSeqDoc().getLineCount(), seq);
	    }
        }
    }

////////////////////////
//********************//
//* WINDOW FUNCTIONS *//
//********************//
////////////////////////
    /**
     * Provides access for other classes to the main program window JFrame object.
     *
     * @return the JFrame object corresponding to the main program window.
     */
    public static JFrame getJFrame() {
	return window;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     *
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading (String name) {
        JMenu heading = menuHeadings.get(name);	// the JMenu object to add

        // check if the heading already exists in the menu. (null indicates that the menu heading is new)
        if (heading == null) {
            // create a new menu heading object
            heading = new JMenu(name);

            // set the mnemonic
            if (name != null && name.length() >= 1 &&
                ((name.charAt(0) >= 'a' && name.charAt(0) <= 'z') ||
                 (name.charAt(0) >= 'A' && name.charAt(0) <= 'Z'))) {
                heading.setMnemonic(name.charAt(0));
            }

            // add the heading
            menuHeadings.put(name, heading);
            menu.add(heading);
        }

        // return the menu heading object
        return heading;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     *
     * @param order the position to place the menu tag
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading (int order, String name) {
        JMenu heading = addMenuHeading(name);	// the hading to add the item to.

        // ensure that the menu heading is in the correct order
        if (menu.getComponentIndex(heading) != order) {
            menu.remove(heading);
            menu.add(heading, order);
        }
        return heading;
    }

    /**
     * Adds a menu item (JMenuItem) to a menu heading (JMenu) within the window (ProgramWindow)
     * <p>If the heading which was entered does not exist, this function will create it;
     *	hence why addMenuHeading returns the heading corresponding to the name entered if a
     *	menu heading with that name already exists.</p>
     *
     * @param headingName is the name of the menu heading to add the menu item to.
     * @param menuItem is the menu item to add to the heading.
     */
    public static void addMenuItem (String headingName, JMenuItem menuItem) {
        // enforce that the menu heading exists and then add the menu item to it
        addMenuHeading(headingName).add(menuItem);
    }

    /**
     * Displays an about popup for BioLegato.
     */
    public static void aboutPopup () {
        JOptionPane.showMessageDialog(window,
                                      "BioLegato version " +
                                      VERSION +
                                      "\nby Graham Alvare and Brian Fristensky\nUniveristy of Manitoba 2008-2010",
                                      "About BioLegato",
                                      JOptionPane.QUESTION_MESSAGE);
    }

    /**
     * Adds a properties listener to the properties object.
     *
     * @param key the key of the properties to listen to.
     * @param listener the listener to add to the object.
     */
    public static void addPropertiesListener (String key, PropertiesListener listener) {
        properties.addPropertiesListener(key, listener);
    }

/////////////////////////
//*********************//
//* GENERAL FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Ensures that the command will be executed properly as a shell command
     * <p>This function generates a command list for execution.  The command list will contain
     *	the appropriate shell for the current operating system, followed by the "execution-argument",
     *	(whatever flag is required to tell the shell that the rest of the commandline should be executed
     *	by the shell), followed by the command to execute (the variable cmd)</p>
     * <p>Operating system values obtained from http://lopica.sourceforge.net/os.html</p>
     *
     * @param cmd the command string to execute
     * @return if successful a Process object for the executed command.
     */
    public static Process safeExecute (String cmd) {
        Process result = null;                          // the result of the function call
        StringBuffer message = new StringBuffer();	// used for printing debug information
        String[] execute = new String[] {cmd};          // default case - run the command by itself

        // builds the execution array
        if (properties.getProperty("shell.name") != null) {
            // if there is a shell for the current operating system, execute the command through the shell
            if (properties.getProperty("shell.parameter") != null) {
                execute =
                new String[] {properties.getProperty("shell.name"),
                              properties.getProperty("shell.parameter"), cmd};
            } else {
                execute = new String[] {properties.getProperty("shell.name"),
                                        cmd};
            }
        }

        // if debug is set, display the command
        if ("true".equalsIgnoreCase(getProperty("debug"))) {
            for (String parameter : execute) {
                // NOTE: append is faster than + or concat operators
                message.append(parameter + " ");
            }
            // relay the command string the message system
            BLMain.message(message.toString(), "safeExecute");
        }

        // obtain the process object for the command
        try {
            result = Runtime.getRuntime().exec(execute);
        } catch (Throwable e) {
            e.printStackTrace();
        }

        // return the resulting process object
        return result;
    }

    /**
     * Runs simple shell commands.
     * Reroutes all output to the console.
     *
     * @param cmd the command string to run
     * @param data the data to use as standard input (System.in)
     */
    public static void shellCommand (String cmd, String data) {
        int result = -65535;			    // the result of running the command
        final Process process = safeExecute(cmd);   // the process object obtained on execution

        // ensure the process object is not null
        if (process != null) {
            try {
                // passes any data to the program via standard in
                if (data != null) {
                    (new OutputStreamWriter(process.getOutputStream())).write(
                            data);
                }

                // this thread object prints the output from the program
                new Thread() {

                    @Override
                    public void run () {
                        String line = "";
                        BufferedReader reader = null;
                        try {
                            reader = new BufferedReader(new InputStreamReader(
                                    process.getInputStream()));
                            while ((line = reader.readLine()) != null) {
                                System.out.println(line);
                            }
                        } catch (IOException ioe) {
                            ioe.printStackTrace();
                        }
                    }

                }.start();
                // this thread object prints the error output from the program
                new Thread() {

                    @Override
                    public void run () {
                        String line = "";
                        BufferedReader reader = null;
                        try {
                            reader = new BufferedReader(new InputStreamReader(
                                    process.getErrorStream()));
                            while ((line = reader.readLine()) != null) {
                                System.err.println(line);
                            }
                        } catch (IOException ioe) {
                            ioe.printStackTrace();
                        }
                    }

                }.start();

                // display the command's result if debug is enabled
                if ("true".equalsIgnoreCase(getProperty("debug"))) {
                    BLMain.message("BioLegato: Command executed successfully, returned: " +
                                   process.waitFor(), "shellCommand");
                } else {
                    process.waitFor();
                }
            } catch (Throwable e) {
                // if there are any errors, print them to the error prompt
                BLMain.error("BioLegato: error executing command: " + cmd,
                             "shellCommand");
                e.printStackTrace();
            }
        }
    }
    
    /**
     * Sends an error message to BioLegato's standard err.
     * Alias for error(message, null);
     *
     * @param message the error message to send.
     */
    public static void error (String message) {
        error(message, null);
    }

    /**
     * Sends an error message to BioLegato's standard err.
     *
     * @param message the error message to send.
     * @param location the location the error occurred.
     */
    public static void error (String message, String location) {
        // print the error to the error stream
        System.err.println(NAME + ((location != null &&  ! "".equals(location.trim()))
                                   ? " (" + location + ")" : "") + ": ERROR --- " + message);
    }

    /**
     * Sends an warning message to BioLegato's standard err.
     * Alias for warning(message, null);
     *
     * @param message the warning message to send.
     */
    public static void warning (String message) {
        warning(message, null);
    }

    /**
     * Sends an warning message to BioLegato's standard err.
     *
     * @param message the warning message to send.
     * @param location the location the error occurred.
     */
    public static void warning (String message, String location) {
        // prints the warning to BioLegato's error stream
        System.err.println(NAME + ((location != null &&  ! "".equals(location.trim()))
                                   ? " (" + location + ")" : "") + ": WARNING --- " + message);
    }

    /**
     * Sends a message to BioLegato's standard out.
     * Alias for message(message, null);
     *
     * @param message the message to send.
     */
    public static void message (String message) {
        message(message, null);
    }

    /**
     * Sends a message to BioLegato's standard out.
     *
     * @param message the message to send.
     * @param location the location the message was sent from.
     */
    public static void message (String message, String location) {
        // prints the warning to BioLegato's error stream
        System.out.println(NAME + ((location != null &&  ! "".equals(location.trim()))
                                   ? " (" + location + ")" : "") + ": " + message);
    }

    /**
     * Reads the complete contents of a BufferedReader into a string.
     *
     * @param reader the BufferedReader to read.
     * @return the contents of the BufferedReader.
     * @throws IOException throws any exceptions from the read operation.
     */
    public static String readStream (BufferedReader reader) throws IOException {
        String line = reader.readLine();            // the current line to read
        StringBuffer result = new StringBuffer();   // the result of the operation

        // read in every line from the stream
        // NOTE: append is faster than + or concat operators
        while (line != null) {
            // don't forget to add the \n back to the line
            result.append(line).append("\n");

            // itterate to the next line
            line = reader.readLine();
        }

        // return the resulting string
        return result.toString();
    }

    /**
     * Checks if a character array is all digits.
     *
     * @param test the character array to test.
     * @return true if the array only contains digits.
     */
    public static boolean testNumber(char[] test) {
	boolean result = false;

	if (test.length > 0) {
	    result = true;
	    for (int count = 0; result && count < test.length; count++) {
		result &= Character.isDigit(test[count]);
	    }
	}
	return result;
    }
    
    /**
     * Replaces $VARIABLE name with environment variables.
     * This function provides functionality similar to BASH.
     *
     * NOTE: if the variable is $BIOLEGATO_HOME, and this is not set, 
     *	the variable will default to the current working directory.
     *
     * @param change the string to modify.
     * @return the modified string.
     */
    public static String envreplace(String change) {
	int start = 0;
	int end = -1;
	String replace = null;
	String variable = "";
	
	if (change != null) {
	    while ((start = change.indexOf('$', start)) >= start && start > -1) {
		for (end = start + 1; end < change.length() && (change.charAt(end) == '_' || Character.isLetter(change.charAt(end))); end++);
		    /* ITTERATE */

		// get the information to perform the string replacement.
		variable = change.substring(start + 1, end);
		replace = System.getenv(variable);
		
		// ensure BIOLEGATO_HOME is set properly.
		if (variable.equalsIgnoreCase("BIOLEGATO_HOME")) {
		    if (replace == null || "".equals(replace.trim()) || !new File(replace).exists()) {
			replace = BLMain.CURRENT_DIR;
		    } else {
			replace = new File(replace).getAbsolutePath();
		    }
		}
		
		// perform the string replacement.
		if (replace != null) {
		    change = change.substring(0, start) + replace + change.substring(end);
		} else {
		    start++;
		}
	    }
	}
	return change;
    }
}
