/*
 * SeqDocTest.java
 * JUnit based test
 *
 * Created on March 15, 2010, 2:50 PM
 */

package org.biolegato.core.data.seqdoc;

import junit.framework.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.Arrays;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.data.sequence.SequenceListener;
import java.util.LinkedList;
import java.util.Stack;
import org.biolegato.core.main.BLMain;

/**
 *
 * @author alvare
 */
public class SeqDocTest extends TestCase {
    final Sequence abc = new Sequence("ABC");
    final Sequence def = new Sequence("DEF");
    final Sequence ghi = new Sequence("GHI");
    final Sequence jkl = new Sequence("JKL");
    final Sequence mnopqr = new Sequence("MNOPQR");
    final Sequence stuvwxyz = new Sequence("STUVWXYZ");
    final Sequence[] emptySequenceArray = new Sequence[0];
    final Sequence[][] dataset = {
	new Sequence[]{abc},
	new Sequence[] {abc, def, ghi},
	new Sequence[]{stuvwxyz, abc, mnopqr, def, ghi, jkl},
	new Sequence[]{abc, mnopqr, def, stuvwxyz, ghi, jkl},
	new Sequence[]{abc, mnopqr, def, ghi, jkl},
	new Sequence[]{abc, mnopqr, def, stuvwxyz, ghi, abc, jkl},
	new Sequence[]{abc, abc, abc, mnopqr, def, stuvwxyz, ghi, abc, jkl},
	new Sequence[]{mnopqr, def, ghi, jkl, stuvwxyz},
	new Sequence[]{stuvwxyz, def, ghi, jkl, mnopqr},
	new Sequence[]{stuvwxyz}
    };

    public SeqDocTest(String testName) {
	super(testName);
    }

    @Override
    protected void setUp() throws Exception {
    }

    @Override
    protected void tearDown() throws Exception {
    }

    public static Test suite() {
	TestSuite suite = new TestSuite(SeqDocTest.class);

	return suite;
    }

//////////////////
//**************//
//* TESTS *//
//**************//
//////////////////
    
    public void testInsert() {
	final SeqDoc instance = new SeqDoc();
	final SeqDocListenerImpl listener = new SeqDocListenerImpl();

	// test listener and SeqDoc status before test
	assertEmpty("Start of test, however SeqDoc is not empty", instance);
	assertReset("Start of test, however SeqDocListenerImpl has errord values", listener);

	// test that reset function works
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl", listener);

	// test arbitrary listener calls
	listener.sequenceAdded(null, -1, null);
	assertAddedOnly("Arbitrary test call (null, -1, null)", listener, null, -1, null);
	listener.resetListener();

	// test pseudo listener calls
	listener.sequenceAdded(instance, 3, def);
	assertAddedOnly("Arbitrary test call ({}, 3, abc)", listener, instance, 3, def);
	listener.resetListener();

	// test arbitrary listener calls
	listener.sequenceChanged(null, -1, null, null);
	assertChangedOnly("Arbitrary test call (null, -1, null, null)", listener, null, -1, null, null);
	listener.resetListener();

	// test pseudo listener calls
	listener.sequenceChanged(instance, 3, abc, "sequence");
	assertChangedOnly("Arbitrary test call ({}, 3, abc, \"sequence\")", listener, instance, 3, abc, "sequence");
	listener.resetListener();

	// prepare for seqDoc listening
	instance.addListener(listener);
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl before starting SeqDoc test", listener);

	// test insertions on empty Sequence document
	assertYield("insert(0, 0, \"\") on {}", instance, instance.insert(0, 0, ""));
	assertYield("insert(0, 0, null) on {}", instance, instance.insert(0, 0, (String) null));
	assertNoYield("insert(0, 0, \"ABC\") on {}", instance, instance.insert(0, 0, "ABC"));
	assertNoYield("insert(1, 1, \"ABC\") on {}", instance, instance.insert(1, 1, "ABC"));
	assertNoYield("insert(-1, -1, \"ABC\") on {}", instance, instance.insert(-1, -1, "ABC"));
	
	// adding a sequence to the start of the sequence document
	assertNoYield("insert(0, -1, {abc})", instance,
	    instance.insert(0, -1, new Sequence[] {abc}), emptySequenceArray);
	assertReset("insert(0, -1, {abc})", listener);
	listener.resetListener();

	// test adding two sequences to an empty document
	assertYield("insert(0, 0, {def, ghi}) on {}", instance,
	    instance.insert(0, 0, new Sequence[] {def, ghi}), new Sequence[] {def, ghi});
	// no listener test at this moment
	listener.resetListener();

	// test inserting a sequence in the middle of a non-empty line
	assertYield("insert(2, 0, {jkl}) on {def, ghi})", instance,
	    instance.insert(2, 0, new Sequence[] {jkl}), new String[] {"DEJKLF", "GHI"});
	assertChangedOnly("insert(2, 0, {jkl}) on {def, ghi})", listener, instance, 0, instance.getLine(0), "sequence");
	listener.resetListener();

	// test inserting a sequence in the middle of the last line of a non-empty document
	assertYield("insert(2, 1, {mnopqr, abc}) on {\"DEJKLF\", \"GHI\"}", instance,
	    instance.insert(2, 1, new Sequence[] {mnopqr, abc}), new String[] {"DEJKLF", "GHMNOPQRI", "ABC"});
	// no listener test at this moment
	listener.resetListener();

	// try inserting a string at the beginning of a non-empty document
	assertYield("insert(0, 0, \"123\") on {\"DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(0, 0, "123"), new String[] {"123DEJKLF", "GHMNOPQRI", "ABC"});
	assertChangedOnly("insert(0, 0, \"123\") on {\"DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", listener, instance, 0, instance.getLine(0), "sequence");
	listener.resetListener();
	
	// try inserting a string in the middle of the first line of a non-empty document
	assertYield("insert(2, 0, {\"abc\") on {\"123DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(2, 0, "abc"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertChangedOnly("insert(2, 0, \"abc\") on {\"123DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", listener, instance, 0, instance.getLine(0), "sequence");
	listener.resetListener();

	// try inserting a string past the last line of a document
	assertNoYield("insert(0, 3, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(0, 3, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertReset("insertion beyond last line", listener);
	listener.resetListener();

	// try inserting a string past the length of any line in a document
	assertNoYield("insert(13, 0, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(13, 0, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertNoYield("insert(10, 1, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(10, 1, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertNoYield("insert(4, 2, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(4, 2, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertReset("insertion beyond line length test", listener);
	listener.resetListener();

	// try inserting a string past the entire bounds of a document
	assertNoYield("insert(3, 3, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(3, 3, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABC"});
	assertReset("insertion beyond document bounds", listener);
	listener.resetListener();
	
	// try inserting a string past the entire bounds of a document
	assertYield("insert(3, 2, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", instance,
	    instance.insert(3, 2, "xyz"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABCxyz"});
	assertChangedOnly("insert(3, 2, \"xyz\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABC\"}", listener, instance, 2, instance.getLine(2), "sequence");
	listener.resetListener();
	
	// try inserting a string at negative co-ordinates
	assertNoYield("insert(-1, 0, \"pqr\")", instance,
	    instance.insert(-1, 0, "pqr"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABCxyz"});
	assertNoYield("insert(0, -1, \"pqr\")", instance,
	    instance.insert(0, -1, "pqr"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABCxyz"});
	assertNoYield("insert(-1, -1, \"pqr\")", instance,
	    instance.insert(-1, -1, "pqr"), new String[] {"12abc3DEJKLF", "GHMNOPQRI", "ABCxyz"});
	assertReset("insertion at negative x, y", listener);
	listener.resetListener();
	
	// insert a string in the middle of the last line of a document
	assertYield("insert(2, 1, \"pqr\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABCxyz\"}", instance,
	    instance.insert(2, 1, "pqr"), new String[] {"12abc3DEJKLF", "GHpqrMNOPQRI", "ABCxyz"});
	assertChangedOnly("insert(2, 1, \"pqr\") on {\"12abc3DEJKLF\", \"GHMNOPQRI\", \"ABCxyz\"}", listener, instance, 1, instance.getLine(1), "sequence");
	listener.resetListener();
	
	// insert a string at the end of the last line of a document
	assertYield("insert(2, 2, \"456\") on {\"12abc3DEJKLF\", \"GHpqrMNOPQRI\", \"ABCxyz\"}", instance,
	    instance.insert(2, 2, "456"), new String[] {"12abc3DEJKLF", "GHpqrMNOPQRI", "AB456Cxyz"});
	assertChangedOnly("insert(2, 2, \"456\") on {\"12abc3DEJKLF\", \"GHpqrMNOPQRI\", \"ABCxyz\"}", listener, instance, 2, instance.getLine(2), "sequence");
	listener.resetListener();
    }

    /**
     * Test of delete method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testDeleteXYW() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{stuvwxyz});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{stuvwxyz, abc, def, ghi, mnopqr});
	final SeqDocListenerImpl listener = new SeqDocListenerImpl();

	// test listener and SeqDoc status before test
	assertReset("Start of test, however SeqDocListenerImpl has errord values", listener);

	// test that reset function works
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl", listener);

	// test arbitrary listener calls
	listener.sequenceRemoved(null, -1, null);
	assertRemovedOnly("Arbitrary test call (null, -1, null)", listener, null, -1, null);
	listener.resetListener();

	// test pseudo listener calls
	listener.sequenceRemoved(instance0, 3, def);
	assertRemovedOnly("Arbitrary test call ({}, 3, abc)", listener, instance0, 3, def);
	listener.resetListener();

	// prepare for seqDoc listening
	instance0.addListener(listener);
	instance1.addListener(listener);
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl before starting SeqDoc test", listener);


	// test negative co-ordinate deletions
	assertTrue(instance0.delete(0, 0, 0));
	assertFalse(instance0.delete(-1, 0, 0));
	assertFalse(instance0.delete(0, -1, 0));
	assertFalse(instance0.delete(0, 0, -1));
	assertFalse(instance0.delete(-1, -1, -1));
	//assertEmpty(instance0);
	
	// test deletions on empty documents
	assertNoYield("delete(2, 0, 0) on {}", instance0, instance0.delete(-1, 0, 0));
	assertNoYield("delete(2, 0, 0) on {}", instance0, instance0.delete(0, -1, 0));
	assertNoYield("delete(2, 0, 0) on {}", instance0, instance0.delete(0, 0, -1));
	assertNoYield("delete(2, 0, 0) on {}", instance0, instance0.delete(-1, -1, -1));
	//assertEmpty(instance0);

	// test deletions from an empty
	
	// test invalid deletions from a single line sequence document
	
	// test valid deletions from a single line sequence document
	assertDeleteXYW(instance1, 2, 0, 0, listener, new String[] {"STUVWXYZ"});
	assertDeleteXYW(instance1, 2, 0, 2, listener, new String[] {"STWXYZ"});
	assertDeleteXYW(instance1, 3, 0, 1, listener, new String[] {"STWYZ"});
    }

    /**
     * Test of delete method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testDeleteXYWH() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{stuvwxyz});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{stuvwxyz, abc, def, ghi, mnopqr});
	final SeqDocListenerImpl listener = new SeqDocListenerImpl();

	// test listener and SeqDoc status before test
	assertEmpty("Start of test, however instance 0 is not empty", instance0);
	assertReset("Start of test, however SeqDocListenerImpl has errord values", listener);

	// test that reset function works
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl", listener);

	// test arbitrary listener calls
	listener.sequenceRemoved(null, -1, null);
	assertRemovedOnly("Arbitrary test call (null, -1, null)", listener, null, -1, null);
	listener.resetListener();

	// test pseudo listener calls
	listener.sequenceRemoved(instance0, 3, def);
	assertRemovedOnly("Arbitrary test call ({}, 3, abc)", listener, instance0, 3, def);
	listener.resetListener();

	// prepare for seqDoc listening
	instance0.addListener(listener);
	instance1.addListener(listener);
	instance2.addListener(listener);
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl before starting SeqDoc test", listener);
	
	assertFalse(instance0.delete(0, 0, 0, 0));
	assertFalse(instance0.delete(-1, 0, 0, 0));
	assertFalse(instance0.delete(0, -1, 0, 0));
	assertFalse(instance0.delete(0, 0, -1, 0));
	assertFalse(instance0.delete(0, 0, 0, -1));
	assertFalse(instance0.delete(-1, -1, -1, -1));
	assertEmpty("Invalid results from deletion on an empty sequence", instance0);
	assertReset("deletion on an empty sequence", listener);
	
	assertDeleteXYWH(instance1, 2, 0, 0, 0, listener, new String[]{"STUVWXYZ"});
	assertDeleteXYWH(instance1, 2, 0, 2, 0, listener, new String[]{"STWXYZ"});
	assertDeleteXYWH(instance1, 3, 0, 1, 0, listener, new String[]{"STWYZ"});
    }

    /**
     * Test of removeSequence method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testAddSequence() {
	final SeqDoc instance = new SeqDoc();
	final SeqDocListenerImpl listener = new SeqDocListenerImpl();

	// test listener and SeqDoc status before test
	assertEmpty("Start of test, however SeqDoc is not empty", instance);
	assertReset("Start of test, however SeqDocListenerImpl has errord values", listener);

	// test that reset function works
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl", listener);

	// test arbitrary listener calls
	listener.sequenceAdded(null, -1, null);
	assertAddedOnly("Failed to set appropriate values for the sequence added call using (null, -1, null)", listener, null, -1, null);
	listener.resetListener();

	// prepare for seqDoc listening
	instance.addListener(listener);
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl before starting SeqDoc test", listener);

	// insert sequence at negative position of an empty SeqDoc
	assertAddSequenceFail(instance, -1, abc, listener);

	// insert sequence at a position beyond the length of an empty SeqDoc
	assertAddSequenceFail(instance, 1, abc, listener);

	// insert sequence at appropriate position of an empty SeqDoc
	assertAddSequence(instance, 0, abc, listener, new Sequence[] {abc});

	// insert sequence at the start of a non-empty SeqDoc
	assertAddSequence(instance, 0, def, listener, new Sequence[] {def, abc});

	// insert sequence at a negative index of a non-empty SeqDoc
	assertAddSequenceFail(instance, -1, ghi, listener, new Sequence[] {def, abc});

	// insert sequence in at the end of a non-empty SeqDoc
	assertAddSequence(instance, 2, ghi, listener, new Sequence[] {def, abc, ghi});

	// insert sequence beyond the length of a non-empty SeqDoc
	assertAddSequenceFail(instance, 6, jkl, listener, new Sequence[] {def, abc, ghi});

	// insert sequence near the end of a non-empty SeqDoc
	assertAddSequence(instance, 2, jkl, listener, new Sequence[] {def, abc, jkl, ghi});
    }
	
    /**
     * Test of removeSequence method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testRemoveSequence() {
	final SeqDoc instance = new SeqDoc(new Sequence[]{def, abc, jkl, ghi});
	final SeqDocListenerImpl listener = new SeqDocListenerImpl();

	// test listener and SeqDoc status before test
	assertReset("Start of test, however SeqDocListenerImpl has errord values", listener);

	// test that reset function works
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl", listener);

	// test arbitrary listener calls
	listener.sequenceRemoved(null, -1, null);
	assertRemovedOnly("Arbitrary test call (null, -1, null)", listener, null, -1, null);
	listener.resetListener();

	// test pseudo listener calls
	listener.sequenceRemoved(instance, 3, def);
	assertRemovedOnly("Arbitrary test call ({}, 3, abc)", listener, instance, 3, def);
	listener.resetListener();

	// prepare for seqDoc listening
	instance.addListener(listener);
	listener.resetListener();
	assertReset("Failed to reset SeqDocListenerImpl before starting SeqDoc test", listener);

	assertContents("Invalid contents at start of removeSequence test", instance, new Sequence[] {def, abc, jkl, ghi});
	assertNoYield("removeSequence(-1) on {def, abc, jkl, ghi}", instance, instance.removeSequence(-1), new Sequence[] {def, abc, jkl, ghi});
	assertReset("removeSequence(-1) on {def, abc, jkl, ghi}", listener);
	
	assertNoYield("removeSequence(4) on {def, abc, jkl, ghi}", instance, instance.removeSequence(4), new Sequence[] {def, abc, jkl, ghi});
	assertReset("removeSequence(4) on {def, abc, jkl, ghi}", listener);
	
	assertYield("removeSequence(3) on {def, abc, jkl, ghi}", instance, instance.removeSequence(3), new Sequence[] {def, abc, jkl});
	assertRemovedOnly("removeSequence(3) on {def, abc, jkl, ghi}", listener, instance, 3, ghi);
	listener.resetListener();
	
	assertYield("removeSequence(1) on {def, abc, jkl}", instance, instance.removeSequence(1), new Sequence[] {def, jkl});
	assertRemovedOnly("removeSequence(1) on {abc}", listener, instance, 1, abc);
	listener.resetListener();
	
	assertYield("removeSequence(0) on {def, jkl}", instance, instance.removeSequence(0), new Sequence[] {jkl});
	assertRemovedOnly("removeSequence(0) on {def, jkl}", listener, instance, 0, def);
	listener.resetListener();
	
	assertYield("removeSequence(0) on {jkl}", instance, instance.removeSequence(0), emptySequenceArray);
	assertRemovedOnly("removeSequence(0) on {jkl}", listener, instance, 0, jkl);
	listener.resetListener();
	
	assertNoYield("removeSequence(1) on {}", instance, instance.removeSequence(1), emptySequenceArray);
	assertReset("removeSequence(1) on {}", listener);
	
	assertNoYield("removeSequence(0) on {}", instance, instance.removeSequence(0), emptySequenceArray);
	assertReset("removeSequence(0) on {}", listener);
	
	assertNoYield("removeSequence(-1) on {}", instance, instance.removeSequence(-1), emptySequenceArray);
    	assertReset("removeSequence(-1) on {}", listener);
}

    /**
     * Test of getLine method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetMethods() {
	assertEmpty("Empty SeqDoc object fails to produce empty results", new SeqDoc());
	
	for (int index = 0; index < dataset.length; index++) {
	    assertContents("GetLine failed for dataset[" + index + "]", new SeqDoc(dataset[index]), dataset[index]);
	    assertLineLength("GetLineLength failed for dataset[" + index + "]", new SeqDoc(dataset[index]), dataset[index]);
	}
    }

    /**
     * Test of getLongestLine method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testGetLongestLine() {
	SeqDoc instance0 = new SeqDoc();
	SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{abc, mnopqr, def, ghi, jkl});
	SeqDoc instance3 = new SeqDoc(new Sequence[]{abc, mnopqr, def, stuvwxyz, ghi, jkl});
	SeqDoc instance4 = new SeqDoc(new Sequence[]{mnopqr, def, ghi, jkl, stuvwxyz});
	SeqDoc instance5 = new SeqDoc(new Sequence[]{stuvwxyz, def, ghi, jkl, mnopqr});
	SeqDoc instance6 = new SeqDoc(new Sequence[]{stuvwxyz});

	assertEquals(0, instance0.getLongestLine());
	assertEquals(3, instance1.getLongestLine());
	assertEquals(6, instance2.getLongestLine());
	assertEquals(8, instance3.getLongestLine());
	assertEquals(8, instance4.getLongestLine());
	assertEquals(8, instance5.getLongestLine());
	assertEquals(8, instance6.getLongestLine());
    }

    /**
     * Test of indexOf method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testIndexOf() {
	SeqDoc instance0 = new SeqDoc(new Sequence[]{});
	SeqDoc instance1 = new SeqDoc(new Sequence[]{abc});
	SeqDoc instance2 = new SeqDoc(new Sequence[]{abc, mnopqr, def, ghi, jkl});
	SeqDoc instance3 = new SeqDoc(new Sequence[]{mnopqr, def, stuvwxyz, ghi, jkl});
	SeqDoc instance4 = new SeqDoc(new Sequence[]{mnopqr, ghi, jkl, abc, stuvwxyz, def});
	SeqDoc instance5 = new SeqDoc(new Sequence[]{abc, stuvwxyz, def, ghi, jkl, mnopqr});
	SeqDoc instance6 = new SeqDoc(new Sequence[]{stuvwxyz});

	assertEquals(-1, instance0.indexOf(abc));
	assertEquals(-1, instance0.indexOf(def));
	assertEquals(0, instance1.indexOf(abc));
	assertEquals(-1, instance1.indexOf(def));
	assertEquals(0, instance2.indexOf(abc));
	assertEquals(2, instance2.indexOf(def));
	assertEquals(-1, instance3.indexOf(abc));
	assertEquals(1, instance3.indexOf(def));
	assertEquals(3, instance4.indexOf(abc));
	assertEquals(5, instance4.indexOf(def));
	assertEquals(0, instance5.indexOf(abc));
	assertEquals(2, instance5.indexOf(def));
	assertEquals(-1, instance6.indexOf(abc));
	assertEquals(-1, instance6.indexOf(def));
    }

    /**
     * Test of getTransferDataFlavors method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testTransferDataFlavors() throws UnsupportedFlavorException, IOException {

	    SeqDoc instance0 = new SeqDoc();

	    assertTrue(instance0.isDataFlavorSupported(DataFlavor.stringFlavor));
	    assertTrue(instance0.isDataFlavorSupported(SeqDoc.seqDocFlavour));
	    
	    assertEquals(instance0, instance0.getTransferData(SeqDoc.seqDocFlavour));
	    assertEquals(instance0.toString(), instance0.getTransferData(DataFlavor.stringFlavor));
    }

    /**
     * Test of toArray method, of class org.biolegato.core.data.seqdoc.SeqDoc.
     */
    public void testToArray() {
	final Sequence[] array = new Sequence[] {abc, def, ghi, jkl};
	final Sequence[] result = (new SeqDoc(array)).toArray();

	assertEquals("Improper array length", array.length, result.length);
	
	for (int count = 0; count < array.length; count++) {
	    assertEquals("Entry @ " + count + " failed", array[count], result[count]);
	}
    }

//////////////////
//**************//
//* ASSERTIONS *//
//**************//
//////////////////
    
    private void assertDeleteXYW(SeqDoc instance, int x, int y, int w, SeqDocListenerImpl listener, String[] result) {
	String message = "delete(" + x + ", " + y + ", " + w + ") on {" + seqdocPrint(instance) + "}";
	assertYield(message, instance, instance.delete(x, y, w), result);
	if (w != 0) {
	    assertFalse(message + " (listener response failure - added = true)", listener.wasAdded());
	    assertTrue(message + " (listener response failure - changed = false)", listener.wasChanged());
	    assertFalse(message + " (listener response failure - removed = true)", listener.wasRemoved());
	}
	listener.resetListener();
    }
    
    private void assertDeleteXYWH(SeqDoc instance, int x, int y, int w, int h, SeqDocListenerImpl listener, String[] result) {
	String message = "delete(" + x + ", " + y + ", " + w + ", " + h + ") on {" + seqdocPrint(instance) + "}";
	assertYield(message, instance, instance.delete(x, y, w, h), result);
	assertFalse(message + " (listener response failure - added = true)", listener.wasAdded());
	if (h != 0) {
	    assertTrue(message + " (listener response failure - removed = false)", listener.wasRemoved());
	}
	if (w != 0) {
	    assertTrue(message + " (listener response failure - changed = false)", listener.wasChanged());
	}
	listener.resetListener();
    }

    private void assertAddSequence(SeqDoc instance, int index, Sequence seq, SeqDocListenerImpl listener, Sequence[] result) {
	String message = "addSequence(" + index + ", " + (seq != null ? "\"" + seq.get("sequence") + "\"" : "null") + ") on {" + seqdocPrint(instance) + "}";
	assertYield(message, instance, instance.addSequence(index, seq), result);
	assertAddedOnly(message, listener, instance, index, seq);
	listener.resetListener();
    }
    
    private void assertAddSequenceFail(SeqDoc instance, int index, Sequence seq, SeqDocListenerImpl listener) {
	String message = "addSequence(" + index + ", " + (seq != null ? "\"" + seq.get("sequence") + "\"" : "null") + ") on {" + seqdocPrint(instance) + "}";
	assertNoYield(message, instance, instance.addSequence(index, seq));
	assertReset(message, listener);
    }
    
    private void assertAddSequenceFail(SeqDoc instance, int index, Sequence seq, SeqDocListenerImpl listener, Sequence[] result) {
	String message = "addSequence(" + index + ", " + (seq != null ? "\"" + seq.get("sequence") + "\"" : "null") + ") on {" + seqdocPrint(instance) + "}";
	assertNoYield(message, instance, instance.addSequence(index, seq), result);
	assertReset(message, listener);
    }
    
    private void assertNoYield(String operation, SeqDoc instance, boolean functionResult) {
	assertFalse("Failure: " + operation + " should fail", functionResult);
	assertEmpty("Wrong contents from " + operation + " = false", instance);
    }
    
    private void assertNoYield(String operation, SeqDoc instance, boolean functionResult, String[] result) {
	assertFalse("Failure: " + operation + " should fail", functionResult);
	assertContents("Wrong contents from " + operation + " = false", instance, result);
    }
    
    private void assertNoYield(String operation, SeqDoc instance, boolean functionResult, Sequence[] result) {
	assertFalse("Failure: " + operation + " should fail", functionResult);
	assertContents("Wrong contents from " + operation + " = false", instance, result);
    }
    
    private void assertYield(String operation, SeqDoc instance, boolean functionResult) {
	assertTrue("Failure: " + operation + " shouldn\'t fail", functionResult);
	assertEmpty("Wrong contents from " + operation + " = true", instance);
    }
    
    private void assertYield(String operation, SeqDoc instance, boolean functionResult, String[] result) {
	assertTrue("Failure: " + operation + " shouldn\'t fail", functionResult);
	assertContents("Wrong contents from " + operation + " = true", instance, result);
    }
    
    private void assertYield(String operation, SeqDoc instance, boolean functionResult, Sequence[] result) {
	assertTrue("Failure: " + operation + " shouldn\'t fail", functionResult);
	assertContents("Wrong contents from " + operation + " = true", instance, result);
    }
    
    private void assertEmpty(String message, SeqDoc instance) {
	assertContents(message, instance, emptySequenceArray);
	assertLineLength(message, instance, emptySequenceArray);
    }
    
    private void assertContents (String message, SeqDoc instance, Sequence[] check) {
	assertEquals(message + " (length failure)", check.length, instance.getLineCount());
	assertNull(message + " (contains negative index)", instance.getLine(-1));
	for (int index = 0; index < check.length; index++) {
	    assertEquals(message + " (invalid data @ index " + index + ")", check[index], instance.getLine(index));
	}
	assertNull(message + " (SeqDoc contains data past length)", instance.getLine(check.length));
	assertNull(message + " (SeqDoc contains data past length)", instance.getLine(check.length + 1));
    }

    private void assertContents (String message, SeqDoc instance, String[] check) {
	assertEquals(message + " (length failure)", check.length, instance.getLineCount());
	assertNull(message + " (contains negative index)", instance.getLine(-1));
	for (int index = 0; index < check.length; index++) {
	    assertNotNull(message + " (missing index " + index + ")", instance.getLine(index));
	    assertNotNull(message + " (missing sequence data for " + index + ")", instance.getLine(index).get("sequence"));
	    assertEquals(message + " (invalid data @ index " + index + ")", check[index], instance.getLine(index).get("sequence").toString());
	}
	assertNull(message + " (SeqDoc contains data past length)", instance.getLine(check.length));
	assertNull(message + " (SeqDoc contains data past length)", instance.getLine(check.length + 1));
    }
    
    private void assertReset(String message, SeqDocListenerImpl listener) {
	assertFalse(message + " (added status not false)", listener.wasAdded());
	assertFalse(message + " (changed status not false)", listener.wasChanged());
	assertFalse(message + " (removed status not false)", listener.wasRemoved());
	asssertListenerData(message, listener, null, -1, null, null);
    }
    
    private void assertAddedOnly(String call, SeqDocListenerImpl listener, SeqDoc source, int index, Sequence seq) {
	String message = "Failed to set appropriate listener status for sequence added: " + call;
	assertTrue(message + " (added status not true)", listener.wasAdded());
	assertFalse(message + " (changed status not false)", listener.wasChanged());
	assertFalse(message + " (removed status not false)", listener.wasRemoved());
	asssertListenerData(message, listener, source, index, seq, null);
    }

    private void assertChangedOnly(String call, SeqDocListenerImpl listener, SeqDoc source, int index, Sequence seq, String key) {
	String message = "Failed to set appropriate listener status for sequence changed: " + call;
	assertFalse(message + " (added status not false)", listener.wasAdded());
	assertTrue(message + " (changed status not true)", listener.wasChanged());
	assertFalse(message + " (removed status not false)", listener.wasRemoved());
	asssertListenerData(message, listener, source, index, seq, key);
    }
    
    private void assertRemovedOnly(String call, SeqDocListenerImpl listener, SeqDoc source, int index, Sequence seq) {
	String message = "Failed to set appropriate listener status for sequence removed: " + call;
	assertFalse(message + " (added status not false)", listener.wasAdded());
	assertFalse(message + " (changed status not false)", listener.wasChanged());
	assertTrue(message + " (removed status not true)", listener.wasRemoved());
	asssertListenerData(message, listener, source, index, seq, null);
    }
    
    private void asssertListenerData(String message, SeqDocListenerImpl listener, SeqDoc source, int index, Sequence seq, String key) {
	assertEquals(message + " (source assertion failure)", source, listener.getSource());
	assertEquals(message + " (index assertion failure)", index, listener.getIndex());
	assertEquals(message + " (sequence assertion failure)", seq, listener.getSequence());
	assertEquals(message + " (modified key assertion failure)", key, listener.getKey());
    }

    private void assertLineLength(String message, SeqDoc instance, Sequence[] data) {
	assertEquals(0, instance.getLineLength(-1));
	for (int index = 0; index < data.length; index++) {
	    assertEquals(data[index].get("sequence").toString().length(), instance.getLineLength(index));
	}
	assertEquals(0, instance.getLineLength(data.length));
	assertEquals(0, instance.getLineLength(data.length + 1));
    }

    private String seqdocPrint(SeqDoc instance) {
	String result = "";
	Sequence[] array = instance.toArray();
	
	if (instance == null) {
	    result = "null";
	} else {
	    for (Sequence seq : array) {
		if (!"".equals(result)) {
		    result += ", ";
		}
		if (seq == abc) {
		    result += "abc";
		} else if (seq == def) {
		    result += "def";
		} else if (seq == ghi) {
		    result += "ghi";
		} else if (seq == jkl) {
		    result += "jkl";
		} else if (seq == mnopqr) {
		    result += "mnopqr";
		} else if (seq == stuvwxyz) {
		    result += "stuvwxyz";
		} else if (seq != null) {
		    result += "\"" + seq.get("sequence") + "\"";
		} else {
		    result += "null";
		}
	    }
	    result = "{" + result + "}";
	}
	return result;
    }
    
    /**
     * Generated implementation of abstract class org.biolegato.core.data.seqdoc.SeqDocListener. Please fill dummy bodies of generated methods.
     */
    private class SeqDocListenerImpl implements SeqDocListener {
	private boolean added = false;
	private boolean changed = false;
	private boolean removed = false;
	private SeqDoc source = null;
	private int index = -1;
	private Sequence seq = null;
	private String key = null;

	public void sequenceAdded(SeqDoc seqDoc, int i, Sequence sequence) {
	    added = true;
	    source = seqDoc;
	    index = i;
	    seq = sequence;
	}

	public void sequenceChanged(SeqDoc seqDoc, int i, Sequence sequence, String string) {
	    changed = true;
	    source = seqDoc;
	    index = i;
	    seq = sequence;
	    key = string;
	}

	public void sequenceRemoved(SeqDoc seqDoc, int i, Sequence sequence) {
	    removed = true;
	    source = seqDoc;
	    index = i;
	    seq = sequence;
	}
	
	public boolean wasAdded() {
	    return added;
	}

	public boolean wasChanged() {
	    return changed;
	}

	public boolean wasRemoved() {
	    return removed;
	}
	
	public SeqDoc getSource() {
	    return source;
	}
	
	public int getIndex() {
	    return index;
	}
	
	public Sequence getSequence() {
	    return seq;
	}
	
	public String getKey() {
	    return key;
	}
	
	public void resetListener() {
	    added = false;
	    changed = false;
	    removed = false;
	    source = null;
	    index = -1;
	    seq = null;
	    key = null;
	}
    }
}
