/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.colourmask;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Hashtable;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.colourmap.ColourMap;

/**
 * The menu item to handle importing colour masks
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class ImportFileAction extends Box implements ActionListener {
    /**
     * The window to associate the import with.
     */
    private JDialog window;
    /**
     * The combobox to use for selecting imported colour masks from.
     */
    private JComboBox combobox;
    /**
     * The list of all colour masks loaded in the system.
     */
    protected static Hashtable<String, ColourMap> colourMasks = new Hashtable<String, ColourMap>();
    /**
     * Used for serialization purposes.
     */
    private static final long serialVersionUID = 7526472295622777037L;
    /**
     * The colour mask file reader for GDE's position-based colour mask files.
     */
    private static final CharacterColourMaskReader characterMaskFile = new CharacterColourMaskReader();
    /**
     * The colour mask file reader for BioLegato's character-based colour mask files.
     */
    private static final GDEColourMaskFile gdeMaskFile = new GDEColourMaskFile();
    
    /**
     * Load the default colour masks (this was the only way I was able to get this code to work properly).
     */
    static {
        if (colourMasks.isEmpty()) {
            if (System.getenv("BIOLEGATIO_COLOURMASK_DIR") != null && ! "".equals(System.getenv("BIOLEGATIO_COLOURMASK_DIR"))) {
                load(new File(BLMain.envreplace(System.getenv("BIOLEGATIO_COLOURMASK_DIR"))));
            } else if (BLMain.getProperty("seqcolourdir") != null && ! "".equals(BLMain.getProperty("seqcolourdir"))) {
                load(new File(BLMain.envreplace(BLMain.getProperty("seqcolourdir"))));
            }
        }
    }

    /**
     * Creates a new ImportFileAction object.
     *
     * @param window the window to associate the import with.
     * @param defaultMask the default mask for the combobox.
     */
    public ImportFileAction(JDialog window, final Object defaultMask) {
	super(BoxLayout.LINE_AXIS);

        // set class parameters
	this.combobox = new JComboBox(colourMasks.values().toArray());
        this.window = window;

        // add the currently loaded masks combobox
	combobox.insertItemAt("<default>", 0);
	if (defaultMask == null || "".equals(defaultMask) || "<default>".equals(defaultMask)) {
	    combobox.setSelectedItem("<default>");
	} else {
	    if (!colourMasks.containsValue(defaultMask)) {
		combobox.insertItemAt(defaultMask, 0);
	    }
	    combobox.setSelectedItem(defaultMask);
	}
	add(combobox);

        // add the import file button
        JButton importButton = new JButton("Import file...");
        importButton.addActionListener(this);
	add(importButton);
    }

    /**
     * Performs all of the import file functionality.
     *
     * @param e is ignored by this function.
     */
    public void actionPerformed (ActionEvent e) {
        JFileChooser openDialog = new JFileChooser();   // the JFileChooser for selecting the file to import.
        ColourMap[] importMasks = null;                 // the colour masks that were imported.

        // set the parameters for the open file dialog box
        openDialog.setCurrentDirectory(new File(System.getProperty("user.dir")));
        openDialog.setAcceptAllFileFilterUsed(false);
	
	// add the default file filters
	openDialog.addChoosableFileFilter(characterMaskFile);
	openDialog.addChoosableFileFilter(gdeMaskFile);
	
	// set the default file filter
	openDialog.setFileFilter(characterMaskFile);

        // if a file is selected, open it
        if (openDialog.showOpenDialog(window) == JFileChooser.APPROVE_OPTION && openDialog.getSelectedFile() != null
                && openDialog.getSelectedFile().exists() && openDialog.getSelectedFile().isFile()) {
            if (openDialog.getFileFilter() != null && openDialog.getFileFilter() instanceof ColourMaskReader) {
                addMasks(((ColourMaskReader)openDialog.getFileFilter()).readColourMapFile(openDialog.getSelectedFile()), this.combobox);
	    }
        }
    }

    /**
     * Loads colour masks into biolegato.
     */
    public static void addMasks(ColourMap[] importMasks, JComboBox addCombobox) {
        int number = 0;                                 // number to append to duplicate masks
	
	for (ColourMap mask : importMasks) {
	    if (colourMasks.contains(mask.getName())) {
		while (colourMasks.contains(mask.getName() + " (" + number + ")")) {
		    number++;
		}
		mask.setName(mask.getName() + " (" + number + ")");
	    }
	    if (addCombobox != null) {
		addCombobox.addItem(mask);
	    }
	    colourMasks.put(mask.getName(), mask);
	}
    }

    /**
     * Returns the current selected colour mask for the sequence.
     *
     * @return the currently selected colour mask for the sequence.
     */
    public Object getSelectedItem() {
	return combobox.getSelectedItem();
    }

    /**
     * Loads all of the colour masks contained in a file or directory.
     *
     * @param location the file object to read colour masks from.
     */
    public static void load (File location) {
	if (location.exists() && location.canRead()) {
	    if (location.isDirectory()) {
		for (File file : location.listFiles()) {
		    load(file);
		}
	    } else if (location.isFile() && location.getAbsolutePath().toLowerCase().endsWith(".csv")) {
		ImportFileAction.addMasks(new CharacterColourMaskReader().readColourMapFile(location), null);
	    }
	}
    }
}
