package org.biolegato.gdesupport.canvas.textarea;

/*
 * BLTextArea.java
 *
 * Created on August 18, 2008, 11:10 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.Stack;
import javax.swing.JComponent;
import org.biolegato.core.data.seqdoc.SeqDoc;
import org.biolegato.core.data.seqdoc.SeqDocListener;
import org.biolegato.gdesupport.canvas.*;
import org.biolegato.gdesupport.canvas.colourmap.CharacterColourMap;
import org.biolegato.gdesupport.canvas.colourmap.ColourMap;
import org.biolegato.gdesupport.canvas.colourmap.MonochromeColourMap;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.biolegato.core.data.sequence.Sequence;
import org.biolegato.core.main.BLMain;

/**
 * A general canvas with more functionality support than JTextArea.
 *
 * This canvas was originally created to support the rectangular selection model,
 * and has since supported rectangular selections, different colour schemes,
 * sequence documents, and many other features.
 * Please add more generic feature support as necessary.
 *
 * Also if you wish to add any functionality or handling specific to a plugin,
 * please feel free to subclass this textarea.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class BLTextArea extends BLComponent implements FocusListener, Editable,SeqDocListener, KeyListener, MouseMotionListener, MouseListener {

    /**
     * If true new text will overwrite existing text.
     */
    protected boolean insertMode = false;
    /**
     * Used to track the shift key.
     * If this is false, cancel the current selection on moving using the direction arrows.
     * If this is true, stretch the current selection on moving using the direction arrows.
     */
    protected boolean selectionMove = false;
    /**
     * Used for mouse dragging.
     * If this is true, dragging the mouse stretches the current selection;
     * otherwise dragging the mouse starts a new selection.
     */
    protected boolean selectionMouse = false;
    /**
     * The current row of the caret
     */
    protected int row = 0;
    /**
     * The current column of the caret.
     */
    protected int column = 0;
    /**
     * The start column of the selection (selection X 1)
     */
    private int sx1 = -1;
    /**
     * The start row of the selection (selection Y 1)
     */
    private int sy1 = -1;
    /**
     * The end column of the selection (selection X 2)
     */
    private int sx2 = -1;
    /**
     * The end row of the selection (selection Y 2)
     */
    private int sy2 = -1;
    /**
     * The colour map to use when the textarea is not greyed out.
     */
    protected ColourMap normalMap = defaultColourMap;
    /**
     * The colour map to use when the textarea is greyed out.
     */
    protected ColourMap greyedOutMap = defaultGreyColourMap;
    /**
     * Handles printing text
     */
    protected ColourMap currentMap = normalMap;
    /**
     * The right click menu for the text area.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Linked list to store all cursor listeners
     */
    protected LinkedList<CursorListener> cursorListeners = new LinkedList<CursorListener>();
    /**
     * Linked list to store all mode change listeners
     */
    protected LinkedList<ModeListener> modeListeners = new LinkedList<ModeListener>();
    /**
     * The colour of invalid areas in the textarea.
     */
    protected final Color invalidzone = Color.RED;
    /**
     * Self-reference for inner classes.
     */
    public final BLTextArea blTextAreaSelf = this;
    /**
     * The default colour map for regular text in BLTextAreas
     */
    public static final ColourMap defaultColourMap = new MonochromeColourMap(Color.BLACK, Color.WHITE, Color.WHITE, Color.BLUE);
    /**
     * The default colour map for greyed out text in BLTextAreas
     */
    public static final ColourMap defaultGreyColourMap = new MonochromeColourMap(Color.BLACK, Color.LIGHT_GRAY, Color.DARK_GRAY, Color.WHITE);
    /**
     * This constant is used for serialization purposes.
     */
    public static final long serialVersionUID = 7526472295622777004L;

    /**
     * Creates a new instance of BLTextArea
     */
    public BLTextArea () {
        // add the listeners
        BLMain.getSeqDoc().addListener(this);
        addKeyListener(this);
        addFocusListener(this);
	addMouseListener(this);
        addMouseMotionListener(this);

        // create popup menu
        addPopupMenuItem(new JMenuItem(new AbstractAction("Cut") {

            private static final long serialVersionUID = 7526472295622777029L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.cut();
            }

        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Copy") {

            private static final long serialVersionUID = 7526472295622777030L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.copy();
            }

        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Paste") {

            private static final long serialVersionUID = 7526472295622777031L;

            public void actionPerformed (ActionEvent e) {
                blTextAreaSelf.paste();
            }

        }));

        // set display
        setFont(BLMain.DEFAULT_FONT);
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
        setFocusTraversalKeysEnabled(false);
        refreshSize();
        setDoubleBuffered(true);
        repaint();
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addCursorListener (CursorListener listener) {
        cursorListeners.add(listener);
    }

    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addModeListener (ModeListener listener) {
        modeListeners.add(listener);
    }

    /**
     * Updates/moves the cursor to the new position.
     *
     * @param newColumn the column co-ordinate of the new position.
     * @param newRow the row co-ordinate of the new position.
     */
    public void changePosition (int newColumn, int newRow) {
        final int oldRow = row;
        final int oldColumn = column;
        
        // ensure that the new row and column are valid
        newRow = Math.max(0, Math.min(newRow, (BLMain.getSeqDoc().getLineCount() - 1)));
        newColumn = Math.max(0, Math.min(newColumn, (BLMain.getSeqDoc().getLineLength(newRow))));

        // update the row and column
        this.row = newRow;
        this.column = newColumn;

        // repaint the new and old cursor areas
        repaint(column2X(oldColumn), row2Y(oldRow), column2X(oldColumn), row2Y(oldRow + 1));
        repaint(column2X(column), row2Y(row), column2X(column), row2Y(row + 1));
        
        // update any cursor listeners
        for (CursorListener listener : cursorListeners) {
            listener.cursorChange(this, column, row);
        }
	
	scrollRectToVisible(new Rectangle(column2X(column), row2Y(row), 1, 1));
    }

    /**
     * Returns the SeqDoc associated with the BLTextArea
     *
     * @return the SeqDoc object.
     */
    public SeqDoc getSeqDoc () {
        return BLMain.getSeqDoc();
    }

///////////////////////
//*******************//
//* DISPLAY METHODS *//
//*******************//
///////////////////////
    /**
     * Changes the current colour map for unselected text.
     *
     * @param c the colour map to use
     */
    public void setColourMap (ColourMap c) {
	if (currentMap == normalMap) {
	    currentMap = c;
	    c.setBackground(getBackground());
	}
        normalMap = currentMap;
        repaint();
    }

    /**
     * Changes the current colour map for unselected text.
     *
     * @param c the colour map to use
     */
    public void setGreyColourMap (ColourMap c) {
	if (currentMap == greyedOutMap) {
	    currentMap = c;
	}
        greyedOutMap = currentMap;
        repaint();
    }
    
    /**
     * Changes the background of the canvas.
     *
     * @param c the new colour for the normal canvas background.
     */
    @Override
    public void setBackground (Color c) {
        super.setBackground(c);
        normalMap.setBackground(c);
    }

    /**
     * Used to obtain all of the text selected within the document.
     *
     * @return the text currently selected in the document.
     */
    public Sequence[] getData () {
        // TODO: getSelectedText
        int lineLength;
        String currentSequence = "";
        Sequence current = null;
        LinkedList<Sequence> result = new LinkedList<Sequence>();

        if ( ! isEmptySelection()) {
            for (int count = getMinSY(); count <= getMaxSY(); count ++) {
                lineLength = BLMain.getSeqDoc().getLineLength(count);
                if (getMinSX() < lineLength) {
                    current = BLMain.getSeqDoc().getLine(count);
                    if (current != null) {
                        current = (Sequence) current.clone();
                        currentSequence = current.get("sequence").toString();
                        if (getMaxSX() < lineLength) {
                            currentSequence = currentSequence.substring(0, getMaxSX());
                        }
			if (getMinSX() > 0) {
                            currentSequence = currentSequence.substring(getMinSX());
                        }
			if (!currentSequence.equals(current.get("sequence").toString())) {
			    current.put("sequence", currentSequence);
			}
                        result.add(current);
                    }
                }
            }
        }
        return result.toArray(new Sequence[0]);
    }

    /**
     * Paints the current textarea (this uses the current clip bounds to determine the area to paint.
     *
     * @param gfx the graphics instance to paint the window to.
     */
    @Override
    public void paintComponent (Graphics gfx) {
	try {
	    final Rectangle area = gfx.getClipBounds();   // get the area to paint
	    final int x = area.x;
	    final int y = area.y;
	    final int width = area.width;
	    final int height = area.height;
	    final int startrow = Y2Row(y);
	    final int endrow = Math.min(BLMain.getSeqDoc().getLineCount(), startrow + Y2Row(height) + 1);
	    final int startcol = X2Column(x);
	    final int endcol = startcol + X2Column(width);
	    final int xstart = column2X(startcol);
	    final int ystart = row2Y(startrow);
	    // get the selection start position
	    final int startSelect = getMinSX();
	    final int endSelect = getMaxSX();
	    Sequence currentLine;
	    ColourMap useMap = currentMap;

	    if (gfx != null) {
		// print the background
		gfx.setColor(getBackground());
		gfx.fillRect(xstart, ystart, width, height);

		// print each sequence
		for (int count = startrow; count < endrow; count ++) {
		    // get the data to print
		    currentLine = BLMain.getSeqDoc().getLine(count);

		    // set the current colour map to the default current colour map
		    useMap = currentMap;

		    // use the sequence specific colour masks if available
		    if (hasFocus() && currentLine.containsKey("mask")) {
			useMap = (ColourMap)currentLine.get("mask");
		    }

		    if (isSelectedLine(count) && endSelect >= 0 && startSelect >= 0 && endSelect - startSelect > 0 && endSelect <= endcol) {
			// draw the sequences
			useMap.regularDrawString(this, gfx, xstart, row2Y(count), currentLine, startcol, startSelect);
			useMap.selectDrawString(this, gfx, xstart + Math.round(columnSize() * startSelect), row2Y(count), currentLine, startSelect, endSelect);
			useMap.regularDrawString(this, gfx, xstart + Math.round(columnSize() * endSelect), row2Y(count), currentLine, endSelect, endcol);
		    } else {
			useMap.regularDrawString(this, gfx, xstart, row2Y(count), currentLine, startcol, endcol);
		    }
		}

		// draw the cursor.
		if (hasFocus() && column2X(column) >= x && column2X(column) <= x + width
			&& row2Y(row) >= y && row2Y(row + 1) <= y + height) {
		    gfx.setColor(getForeground());
		    gfx.drawLine(column2X(column), row2Y(row), column2X(column), row2Y(row + 1));
		}
	    }
	} catch (Throwable th) {
	    // don't halt printing other objects on crash
	    th.printStackTrace();
	}
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont(Font font) {
        super.setFont(font);
        refreshSize();
	repaint();
    }
    
////////////////////////
//********************//
//* KEYBOARD METHODS *//
//********************//
////////////////////////
    /**
     * Processes the typing of keys within the text area
     *
     * @param event the KeyEvent for the key typed
     */
    public void keyTyped (KeyEvent event) {
        boolean canInsert = true;
	
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                case KeyEvent.VK_DELETE:
                    break;
                case KeyEvent.VK_ENTER:
                    // now only changes the row (same effect as downkey, except clears selection)
                    if (row + 1 < BLMain.getSeqDoc().getLineCount()) {
                        changePosition(Math.min(BLMain.getSeqDoc().getLineLength(row + 1), column), row + 1);
                        resetSelection();
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    break;
                default:
                    if ( ! isEmptySelection()) {
                        deleteSelection();
                    } else if (insertMode && BLMain.getSeqDoc().getLineCount() > 0
			    && BLMain.getSeqDoc().getLineLength(0) > 0) {
			canInsert = delete(column, row, 1, 0);
                    }
                    if (isEmptySelection() || !insertMode || canInsert) {
			insert(column, row, String.valueOf(event.getKeyChar()));
                        changePosition(column + 1, row);
                    }
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
	event.consume();
    }

    /**
     * Processes key presses within the text area
     *
     * @param event the KeyEvent for the key pressed
     */
    public void keyPressed (KeyEvent event) {
	int backspacecol = 0;
        boolean moved = false;  // used for updating selections

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                    if ( ! isEmptySelection()) {
                        deleteSelection();
                    } else if (row > 0 || column > 0) {
                        if (column <= 0 && row > 0) {
                            backspacecol = BLMain.getSeqDoc().getLineLength(row - 1);
                        }
			if (column - 1 >= 0) {
			    if (delete(column - 1, row, 1, 0)) {
				if (column <= 0 && row > 0) {
				    changePosition(backspacecol, row - 1);
				} else {
				    changePosition(column - 1, row);
				}
			    }
			}
                    }
                    break;
                case KeyEvent.VK_DELETE:
                    if ( ! isEmptySelection()) {
                        deleteSelection();
                    } else if (BLMain.getSeqDoc().getLineCount() > 0
			    && BLMain.getSeqDoc().getLineLength(0) > 0) {
                        delete(column, row, 1, 0);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            if (isEmptySelection()) {
                                moveSelection(column, row);
                                stretchSelection(column, row);
                            }
                            selectionMove = true;
                            break;
                        case KeyEvent.VK_LEFT:
                            if (column > 0) {
                                changePosition(column - 1, row);
                                moved = true;
                            } else if (row > 0) {
                                changePosition(BLMain.getSeqDoc().getLineLength(row), row - 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_RIGHT:
                            if (column + 1 <= BLMain.getSeqDoc().getLineLength(row)) {
                                changePosition(column + 1, row);
                                moved = true;
                            } else if (row + 1 < BLMain.getSeqDoc().getLineCount()) {
                                changePosition(0, row + 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_UP:
                            if (row > 0) {
                                changePosition(Math.min(BLMain.getSeqDoc().getLineLength(row - 1), column), row - 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_DOWN:
                            if (row + 1 < BLMain.getSeqDoc().getLineCount()) {
                                changePosition(Math.min(BLMain.getSeqDoc().getLineLength(row + 1), column), row + 1);
                                moved = true;
                            }
                            break;
                        case KeyEvent.VK_HOME:
                            changePosition(0, row);
                            moved = true;
                            break;
                        case KeyEvent.VK_END:
                            changePosition(BLMain.getSeqDoc().getLineLength(row), row);
                            moved = true;
                            break;
                        case KeyEvent.VK_COPY:
                            copy();
                            break;
                        case KeyEvent.VK_CUT:
                            cut();
                            break;
                        case KeyEvent.VK_PASTE:
                            paste();
                            break;
                        case KeyEvent.VK_INSERT:
                            insertMode =  ! insertMode;
                            for (ModeListener listener : modeListeners) {
                                listener.insertionMode(insertMode);
                            }
                            break;
                        default:
                            if ("true".equalsIgnoreCase(BLMain.getProperty("debug"))) {
                                BLMain.warning("Unhandled key pressed --- getKeyChar() = " + ((int) event.getKeyChar()) + "\tgetKeyCode() = " + event.getKeyCode(), "BLTextArea");
                            }
                            break;
                    }
                    if (column >= BLMain.getSeqDoc().getLineLength(row)) {
                        changePosition(BLMain.getSeqDoc().getLineLength(row), row);
                    }
                    if (moved) {
                        if (selectionMove) {
                            stretchSelection(column, row);
                        } else if (!isEmptySelection()) {
                            resetSelection();
                        }
                        moved = false;
                    }
                    break;
                default:
                    deleteSelection();
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
	event.consume();
    }

    /**
     * Processes key releases within the text area
     *
     * @param event the KeyEvent for the key released
     */
    public void keyReleased (KeyEvent event) {
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = false;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
	event.consume();
    }

/////////////////////
//*****************//
//* MOUSE METHODS *//
//*****************//
/////////////////////
    /**
     * Handles mouse clicks.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set, in which case it stretches the selection).
     *
     * @param event the MouseEvent object corresponding to the click.
     */
    public void mouseClicked (MouseEvent event) {
        if (!event.isPopupTrigger()) {
            // calculate the row and column numbers from the X and Y co-ordinates.
            changePosition(X2Column(event.getX()), Y2Row(event.getY()));

            // update the selection (if necessary)
            if ( ! selectionMove) {
                resetSelection();
            } else {
                stretchSelection(column, row);
            }
            
            // request focus within the current window
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse button presses.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set).
     *
     * @param event the MouseEvent object corresponding to the press.
     */
    public void mousePressed (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(this, event.getX(), event.getY());
        } else if ( ! selectionMove) {
            resetSelection();
        }
    }

    /**
     * Handles mouse button releases.
     * This method sets selectionMouse to false
     *
     * @param event the MouseEvent object corresponding to the release.
     */
    public void mouseReleased (MouseEvent event) {
        selectionMouse = false;
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX(), event.getY());
        }
    }

    /**
     * Handles the mouse entering the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the enter.
     */
    public void mouseEntered (MouseEvent event) {
    }

    /**
     * Handles the mouse exiting the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the exit.
     */
    public void mouseExited (MouseEvent event) {
    }

    /**
     * Handles mouse drags.
     * Updates the selection and caret on drags.
     *
     * @param event the MouseEvent object corresponding to the drag.
     */
    public void mouseDragged (MouseEvent event) {
        // calculate the row and column numbers from the X and Y co-ordinates and update the position
        changePosition(X2Column(event.getX()), Y2Row(event.getY()));
        
        if ( ! selectionMouse) {
            moveSelection(column, row);
            selectionMouse = true;
        } else {
            stretchSelection(column, row);
        }
        requestFocus();
        requestFocusInWindow();
    }

    /**
     * Handles mouse movements.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the movement.
     */
    public void mouseMoved (MouseEvent event) {
    }
    
    /**
     * Refreshes the size of the textarea (for scroll size purposes)
     */
    protected void refreshSize() {
	final int width = (BLMain.getSeqDoc().getLongestLine() + 1) * columnSize();
	final int height = rowSize() * BLMain.getSeqDoc().getLineCount();
	
        Dimension size = new Dimension(width, height);
        setSize(size);
        setPreferredSize(size);
	revalidate();
    }
    
/////////////////////////////
//*************************//
//* CUSTOMIZATION METHODS *//
//*************************//
/////////////////////////////
    /**
     * Adds an item to the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to add.
     */
    public void addPopupMenuItem (JMenuItem act) {
        popup.add(act);
    }

    /**
     * Removes an item from the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to remove.
     */
    public void removePopupMenuItem (JMenuItem act) {
        popup.remove(act);
    }
    
/////////////////////////////////////
//*********************************//
//* EXTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Copies content from the current Editable object.
     */
    public void copy () {
        if ( ! isEmptySelection()) {
            BLMain.setClipboard(getData());
        }
    }

    /**
     * Cuts content from the current Editable object.
     */
    public void cut () {
        if ( ! isEmptySelection()) {
            copy();
            deleteSelection();
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste () {
        deleteSelection();
        insert(column, row, BLMain.getClipboard());
    }

    /**
     * Selects all the sequences within the textarea.
     */
    public void selectAll() {
        moveSelection(0, 0);
        stretchSelection(BLMain.getSeqDoc().getLongestLine(), BLMain.getSeqDoc().getLineCount() - 1);
    }

/////////////////////////////////////
//*********************************//
//* INTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Inserts a string into the textarea's underlying SeqDoc
     * NOTE: this is a wrapper method for <b>insert</b>(<i>x</i>, <i>y</i>, <i>data</i>)
     *
     * @param x the X co-ordinate (column number) to insert the string at.
     * @param y the Y co-ordinate (row number) to insert the string at.
     * @param data the string to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert (int x, int y, String data) {
        return BLMain.getSeqDoc().insert(x, y, data);
    }
    
    /**
     * Inserts an array of sequences into the textarea's underlying SeqDoc
     *
     * @param x the X co-ordinate (column number) to insert the sequences at.
     * @param y the Y co-ordinate (row number) to insert the sequences at.
     * @param data the array of sequences to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert (int x, int y, Sequence[] data) {
	boolean result = true;
	
        for (int count = 0; count < data.length; count++) {
	    if (y + count < BLMain.getSeqDoc().getLineCount()) {
		result &= insert(x, y + count, data[count].get("sequence").toString());
	    } else {
		result &= BLMain.getSeqDoc().addSequence(y + count, data[count]);
	    }
	}
	return result;
    }
    
    /**
     * Deletes characters from the textarea's underlying SeqDoc
     * NOTE: deletions are performed in a sequential manner
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return a boolean corresponding to the result of the deletion (true = successful deletion)
     */
    public boolean delete (final int x, final int y, final int w, final int h) {
        return BLMain.getSeqDoc().delete(x, y, w, h);
    }
    
    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     *
     * @return whether or not the deletion was performed.
     */
    public boolean deleteSelection () {
	final int x = getMinSX();
	final int y = getMinSY();
	final int w = getMaxSX() - x;
	final int h = getMaxSY() - y;
	boolean result = false;
	
        if ( ! isEmptySelection()) {
	    if (delete(x, y, w, h)) {
		resetSelection();
		changePosition(x, y);
		result = true;
	    }
        }
	return result;
    }
    
//////////////////////
//******************//
//* STATUS METHODS *//
//******************//
//////////////////////
    /**
     * Returns the insertion mode status of the textarea.
     *
     * Insertion mode is invoked by hitting the "INSERT" key.
     * While in insertion mode, if you type a character key in front of some text
     * the character key inserted will replace the first character of the text in front
     * of the insertion point.
     *
     * @return whether or not insertion mode is on.
     */
    public boolean getInsertMode () {
        return insertMode;
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Called when a sequence is added to a SeqDoc.
     *
     * @param index the location the sequence was added.
     * @param sequence the sequence added.
     */
    public void sequenceAdded(SeqDoc source, final int index, Sequence sequence) {
        final int x = 0;				// the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();	// the width of the area to repaint
        final int y = row2Y(index);			// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint
	
        refreshSize();					// update the size of the textarea
        repaint(x, y, w, h);				// repaint the modified area of the textarea
    }

    /**
     * Called when a field in a sequence is modified.
     *
     * @param index the location of the sequence.
     * @param sequence the sequence modified.
     * @param key the value modified.
     */
    public void sequenceChanged(SeqDoc source, int index, Sequence sequence, String key) {
        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(index);                 // the Y co-ordinate to start repainting at
        final int h = rowSize();                    // the height of the area to repaint
        
        refreshSize();                              // update the size of the textarea
        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Called when a sequence is removed from a SeqDoc.
     *
     * @param index the location (line number) where the sequence was removed from.
     * @param sequence the sequence removed from the SeqDoc.
     */
    public void sequenceRemoved(SeqDoc source, int index, Sequence sequence) {
        final int x = 0;				// the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();	// the width of the area to repaint
        final int y = row2Y(index);			// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint
        
        refreshSize();					// update the size of the textarea
        repaint(x, y, w, h);				// repaint the modified area of the textarea
    }

    /**
     * Handles activating the text area (changing colours from greyed out to normal).
     *
     * @param e this parameter is currently ignored
     */
    public void focusGained(FocusEvent e) {
        currentMap = normalMap;
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
        repaint();
    }

    /**
     * Handles deactivating the text area (changing colours from normal to greyed out).
     *
     * @param e this parameter is currently ignored
     */
    public void focusLost(FocusEvent e) {
        currentMap = greyedOutMap;
        setForeground(Color.BLACK);
        setBackground(Color.LIGHT_GRAY);
	repaint();
    }
    
/////////////////////////
//*********************//
//* SELECTION METHODS *//
//*********************//
/////////////////////////
    /**
     * Resets the current selection to (0,0 - 0,0)
     */
    public void resetSelection () {
	moveSelection(0,0);
	stretchSelection(0,0);
    }
    
    /**
     * Moves the start of a selection to a new co-ordinate
     *
     * @param newx the new X co-ordinate the selection has been moved to.
     * @param newy the new Y co-ordinate the selection has been moved to.
     */
    public void moveSelection(int newx, int newy) {
        final int x = 0;					// the X co-ordinate to start repainting at
        final int y = row2Y(Math.min(newy, getMinSY()) - 1);    // the width of the area to repaint
        final int w = (int)getSize().getWidth();		// the Y co-ordinate to start repainting at
        final int h = row2Y(Math.max(newy, getMaxSY()) + 1);    // the height of the area to repaint
	
	// update the selection
	sx1 = newx;
	sy1 = newy;
	
	// repaint the affected canvas area
        repaint(x, y, w, h);				// repaint the modified area of the textarea	
    }
    
    /**
     * Stretches the selection by changing the co-ordinate of the selection's end
     *
     * @param newx the new X co-ordinate the selection has been stretched to.
     * @param newy the new Y co-ordinate the selection has been stretched to.
     */
    public void stretchSelection(int newx, int newy) {
        final int x = 0;					// the X co-ordinate to start repainting at
        final int y = row2Y(Math.min(newy, getMinSY()) - 1);    // the width of the area to repaint
        final int w = (int)getSize().getWidth();		// the Y co-ordinate to start repainting at
        final int h = row2Y(Math.max(newy, getMaxSY()) + 1);    // the height of the area to repaint
        
	// update the selection
	sx2 = newx;
	sy2 = newy;
	
	// repaint the affected canvas area
        repaint(x, y, w, h);				// repaint the modified area of the textarea	
    }
    
    /**
     * Tests if the selection shape is empty
     *
     * @return true if the selection is empty
     */
    protected boolean isEmptySelection() {
        return (sx1 == sx2 || sy1 == sy2);
    }

    /**
     * Gets the minimum X co-ordinate of the selection area.
     *
     * @return Math.min(sx1, sx2)
     */
    public int getMinSX () {
        return Math.min(sx1, sx2);
    }

    /**
     * Gets the maximum X co-ordinate of the selection area.
     *
     * @return Math.max(sx1, sx2)
     */
    public int getMaxSX () {
        return Math.max(sx1, sx2);
    }

    /**
     * Gets the minimum Y co-ordinate of the selection area.
     *
     * @return Math.min(sy1, sy2)
     */
    public int getMinSY () {
        return Math.min(sy1, sy2);
    }

    /**
     * Gets the maximum Y co-ordinate of the selection area.
     *
     * @return Math.max(y1, y2)
     */
    public int getMaxSY () {
        return Math.max(sy1, sy2);
    }

    /**
     * Gets the starting X co-ordinate of the selection area.
     *
     * @return sx1
     */
    public int getSX1 () {
        return sx1;
    }

    /**
     * Gets the starting Y co-ordinate of the selection area.
     *
     * @return sy1
     */
    public int getSY1 () {
        return sy1;
    }

    /**
     * Gets the ending X co-ordinate of the selection area.
     *
     * @return sx2
     */
    public int getSX2 () {
        return sx1;
    }

    /**
     * Gets the ending Y co-ordinate of the selection area.
     *
     * @return sy2
     */
    public int getSY2 () {
        return sy2;
    }

    /**
     * This function is used to test if a line is selected.
     * Currently in this class, the test is just if the selection is within the bounds of sy1 and sy2.
     *
     * @param y the Y co-ordinate (or line number) to test.
     * @return true if the line is within the selection area, and should be printed as such.
     */
    public boolean isSelectedLine(int y) {
	return (y >= getMinSY() && y <= getMaxSY());
    }
}
