/*
 * EZRunWindow.java
 *
 * Created on January 5, 2010, 2:06 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.ezmenu;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JScrollPane;
import org.biolegato.core.main.BLMain;
import org.biolegato.core.plugins.PluginLoader;
import org.biolegato.core.plugins.PluginWrapper;
import org.biolegato.core.plugintypes.MenuType;
import org.biolegato.ezmenu.variables.GDEChoiceList;
import org.biolegato.ezmenu.variables.GDEChooser;
import org.biolegato.ezmenu.variables.GDEComboBox;
import org.biolegato.ezmenu.variables.GDEFileChooser;
import org.biolegato.ezmenu.variables.GDESlider;
import org.biolegato.ezmenu.variables.GDETempFile;
import org.biolegato.ezmenu.variables.GDETextField;
import org.biolegato.ezmenu.variables.GDEVariable;
import org.biolegato.ezmenu.variables.GDEWidgetVariable;

/**
 *
 * @author alvare
 */
public class EZRunWindow implements Runnable {

    /**
     * Variable list - used for updating variable status
     */
    private LinkedList<Map<String, Object>> variables;
    /**
     * The current window
     */
    private JDialog runWindow = null;
    /**
     * This variable stores the basic command string.
     * This basic command string will be used to determine the command to run,
     * and will be left unmodified after execution.
     */
    private String command = "";
    /**
     * List of widgets for the window.
     */
    private LinkedList<GDETempFile> files = new LinkedList<GDETempFile>();
    /**
     * List of widgets for the window.
     */
    private HashMap<String, GDEWidgetVariable> widgets = new HashMap<String, GDEWidgetVariable>();
    /**
     * Used in serialization.
     */
    private static final long serialVersionUID = 7526472295622777007L;

    /** Creates a new instance of EZRunWindow */
    public EZRunWindow(String name, String command, Action[] buttons, LinkedList<Map<String, Object>> variables) {
        // set the classes internal variables
        this.command = command;
        this.variables = variables;

        /* THIS BLOCK OF CODE
         * Determines whether the program has widgets that require input.
         * If there are no widgets the program will launch automatically when called.
         */
        boolean isDisplayWindow = false;
        GDEVariable variable = null;
        Container variablePane = new Box(BoxLayout.PAGE_AXIS);

        // create the variables and determine if they are widgets
        for (Map<String, Object> varData : variables) {
            variable = createVariable(varData);
            if (variable instanceof GDETempFile) {
                files.add((GDETempFile) variable);
            } else if (variable instanceof GDEWidgetVariable) {
                // NOTE: this must be lower case to prevent case sensitivity
                variablePane.add((GDEWidgetVariable) variable);
                widgets.put(variable.getName().toLowerCase(), (GDEWidgetVariable) variable);
            }
        }

        if (!widgets.isEmpty() || buttons.length > 0) {
            runWindow = new JDialog(BLMain.getJFrame(), "BioLegato: " + name, true);
            Container panel = null;
            Container windowPane = new Box(BoxLayout.PAGE_AXIS);
            Container buttonPane = new Box(BoxLayout.LINE_AXIS);

            // create the pane and make it scrollable
            runWindow.setContentPane(new JScrollPane(windowPane));
            windowPane.add(variablePane);
            windowPane.add(buttonPane);

            // clean up and display the window
            runWindow.setMinimumSize(new Dimension(50, 50));
            runWindow.setSize(new Dimension(100, 100));
            runWindow.setLocationRelativeTo(BLMain.getJFrame());
            runWindow.setVisible(false);

            buttonPane.add(new JButton(new ActionThread("Run", this)));

            for (Action button : buttons) {
                buttonPane.add(new JButton(button));
            }

            runWindow.pack();
            runWindow.setVisible(true);
            runWindow.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        } else {
            new Thread(this).start();
        }
    }

    /**
     * Used for running the command.
     * <p>
     *	This function gathers all of the parameter settings from the widgets
     *	in the window, then generates and executes the corresponding command
     *	string.
     * </p>
     */
    public void run() {
        String search = "";
        String run = command;

        // update the values of all used variables
        for (Map<String, Object> var : variables) {
            search = var.get("name").toString().toLowerCase();
            if (widgets.containsKey(search)) {
                var.put("default", widgets.get(search).getValue());
            }
        }

        // destroy the current run window
        if (runWindow != null) {
            runWindow.dispose();
        }

        run = replaceArguments(run);

        // execute the program and collect program output
        BLMain.shellCommand(run, "");

        // cleanup execution
        // releases all of the input files
        for (GDETempFile file : files) {
            file.close();
        }
        runWindow = null;
    }

    /**
     * Replaces the variables in the command string with their corresponding values.
     *
     * @param run the command to do the replacements on.
     * @param widgets the widget variables to use for string replacement.
     * @param files the file variables to use for string replacement.
     * @return the altered command string.
     */
    private String replaceArguments(String run) {
        int start = -1;
        int end = -1;
        String name = "";
        String value = "";

        // create the command string (NOTE: variable.toString().replaceAll(Pattern.quote("$"), "\\\\\\$") is used to prevent regex grouping (e.g. $0, etc.))
        while ((start = run.indexOf('$', start)) >= 0) {
            end = start + 1;
            while (end < run.length() && isDigit(run.charAt(end))) {
                end++;
            }
            name = run.substring(start + 1, end).toLowerCase();
            value = "";
            if (widgets.containsKey(name)) {
                value = widgets.get(name).getValue().toString();
                run = (start == 0 ? "" : run.substring(0, start)) + value + (end >= run.length() ? "" : run.substring(end));
                start--;
            } else {
                start++;
            }
        }

        for (GDETempFile file : files) {
            // reset the start and positions
            start = -1;
            end = -1;

            // make sure name is valid
            if (file.getName() != null && !file.getName().equals("")) {
                name = file.getName();
                value = file.getValue().toString();

                // itterate through and find each instance of the name to replace
                while ((start = run.indexOf(name, end + 1)) >= 0) {
                    end = start + name.length();

                    // make sure that the name is not part of a bigger name
                    if ((start == 0 || notDigit(run.charAt(start - 1))) && (end >= run.length() || notDigit(run.charAt(end)))) {
                        run = (start == 0 ? "" : run.substring(0, start)) + value + (end >= run.length() ? "" : run.substring(end));
                        end = start - 1;
                    }
                }
            }
        }
        return run;
    }

    /**
     * Used to create BioLegato variables.
     *
     * @param data the data to create the variable with
     * @return the crated variable
     */
    public final static GDEVariable createVariable(Map data) {
        String type = null;
        GDEVariable result = null;

        if (data.containsKey("type")) {
            type = data.get("type").toString().trim();
            if ("slider".equalsIgnoreCase(type)) {
                result = new GDESlider(data);
            } else if ("choice_menu".equalsIgnoreCase(type)) {
                result = new GDEComboBox(data);
            } else if ("choice_list".equalsIgnoreCase(type)) {
                BLMain.warning("Using deprecated menu widget \"choice_list\"");
                result = new GDEChoiceList(data);
            } else if ("chooser".equalsIgnoreCase(type)) {
                result = new GDEChooser(data);
            } else if ("text".equalsIgnoreCase(type)) {
                result = new GDETextField(data);
            } else if ("tempfile".equalsIgnoreCase(type)) {
                result = new GDETempFile(data);
            } else if ("file_chooser".equalsIgnoreCase(type)) {
                result = new GDEFileChooser(data);
            } else if (type != null) {
                for (PluginWrapper plugin : PluginLoader.getPlugins(GDEWidgetVariable.class)) {
                    if (type.equalsIgnoreCase("" + plugin.smethod("getType"))) {
                        result = (GDEWidgetVariable) plugin.create(new Class[]{java.util.Map.class}, new Object[]{data});
                    }
                }
            }
        }
        if (result == null) {
            BLMain.error("Invalid variable type: " + type + " (check your plugin directory, properties file, and menu files for possible errors)", MenuType.class.getCanonicalName() + ".createVariable");
        }
        return result;
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is not a digit, otherwise false.
     */
    private boolean notDigit(char test) {
        return ((test < '0' || test > '9') && (test < 'a' || test > 'z') && (test < 'A' || test > 'Z'));
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is a digit, otherwise false.
     */
    private boolean isDigit(char test) {
        return ((test >= '0' && test <= '9') || (test >= 'a' && test <= 'z') || (test >= 'A' && test <= 'Z'));
    }
}
