/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.colourmap;

import org.biolegato.gdesupport.canvas.textarea.BLTextArea;
import java.awt.Color;
import java.awt.Graphics;
import org.biolegato.core.data.sequence.Sequence;

/**
 * The superclass of all colour maps.
 *
 * Colour maps are used to provide different colour schemas to canvases.
 * To implement a new schema, create a subclass that implements all of the methods
 * you wish to override of the default schema.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public abstract class ColourMap {

    /**
     * The current default foreground colour of normal unselected text
     */
    protected Color foreground = Color.BLACK;
    /**
     * The current default background colour of normal unselected text
     */
    protected Color background = Color.WHITE;
    /**
     * The current default foreground colour of normal selected text
     */
    protected Color foregroundSelect = Color.WHITE;
    /**
     * The current default background colour of normal selected text
     */
    protected Color backgroundSelect = Color.BLUE;
    
    /**
     * The name of the colour map.
     */
    private String name = "";
    
    /**
     * The abstract constructor for colour maps.
     */
    public ColourMap () {
    }

    /**
     * Draws an unselected sequence while mapping the appropriate colours to each character rendered.
     *
     * The sequence passed to this function need not be modified, as all of the math and string
     * manipulation required for handling selected text is done in the drawString function.
     *
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param seq the sequence to draw.
     */
    public abstract void regularDrawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, Sequence seq, int pstart, int pend);

    /**
     * Draws a selected sequence while mapping the appropriate colours to each character rendered.
     *
     * The sequence passed to this function need not be modified, as all of the math and string
     * manipulation required for handling selected text is done in the drawString function.
     * 
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param seq the sequence to draw.
     */
    public void selectDrawString (BLTextArea ta, Graphics gfx, int xstart, int ystart, Sequence seq, int pstart, int pend) {
	char[] drawArray = safesubstring(seq, pstart, pend).toCharArray();
        drawChars(ta, gfx, xstart, ystart, drawArray, 0, drawArray.length, foregroundSelect, backgroundSelect);
    }

    /**
     * This is a function to help ensure proper spacing.
     * All of the drawing should be done through this function.
     *
     * @param ta the textarea to use for obtaining metrics from.
     * @param gfx the graphics object to draw the string to.
     * @param xstart the x co-ordinate to start drawing from.
     * @param ystart the y co-ordinate to start drawing from.
     * @param array the character to draw.
     * @param index the index to draw the characters in the array.
     * @param length the length of characters to draw in the array.
     * @param foreground the foreground colour of the string.
     * @param background the background colour of the string.
     */
    protected void drawChars (BLTextArea ta, Graphics gfx, int xstart, int ystart, char[] array, int index, int length, Color foreground, Color background) {
        int xstartMod = 0;                          // the amount to offset xstart so each printed character is centered.
	int end = index + length;
        
        gfx.setColor(background);
        gfx.fillRect(xstart, ystart, length * ta.columnSize(), ta.rowSize());
        gfx.setFont(ta.getFont());
        gfx.setColor(foreground);
	
	if (end <= array.length) {
	    for (int column = index; column < end; column++) {
		xstartMod = (ta.columnSize() - ta.getFontMetrics(ta.getFont()).charWidth(array[column])) / 2;
		gfx.drawChars( array, column, 1, xstart + xstartMod, ystart + ta.rowSize() - 5);
		xstart += ta.columnSize();
	    }
	}
    }
    
    /**
     * Ensures all substrings do not excees the start or end of the sequence
     *
     * @param pstart the start position of the substring
     * @param pend the end position of the substring
     */
    public String safesubstring(Sequence seq, int pstart, int pend) {
	String sequencetxt = seq.get("sequence").toString();
	pstart = Math.max(0, Math.min(sequencetxt.length(), pstart));
	pend = Math.max(pstart, Math.min(sequencetxt.length(), pend));
	return sequencetxt.substring(pstart, pend);
    }

    /**
     * Returns the current default foreground colour for painting portions of the canvas without text.
     *
     * @return the background colour.
     */
    public Color getForeground () {
        return background;
    }

    /**
     * Changes the current default foreground colour for painting portions of the canvas without text.
     *
     * @param c the new background colour.
     */
    public void setForeground (Color c) {
        background = c;
    }

    /**
     * Returns the current default background colour for painting portions of the canvas without text.
     *
     * @return the background colour.
     */
    public Color getBackground () {
        return background;
    }

    /**
     * Changes the current default background colour for painting portions of the canvas without text.
     *
     * @param c the new background colour.
     */
    public void setBackground (Color c) {
        background = c;
    }

    /**
     * Returns the name of the colour map.
     * 
     * @return the name of the colour map.
     */
    public String getName() {
        return name;
    }

    /**
     * Changes the name of the colour map.
     * 
     * @param string the new name for the colour map.
     */
    public void setName (String string) {
        name = string;
    }
    
    /**
     * Returns the name of the colour map.
     * 
     * @return the name of the colour map.
     */
    public String toString() {
        return name;
    }
}
