//=====================================================================
// File:    HighestPeakLocate.java
// Class:   HighestPeakLocate
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

/**
 * This class can be used to find peaks in lanes. It will simply find the
 * highest value in the specified range. It will then add this value to 
 * the list of peaks in the given lane. However, it the peak already exists,
 * same locaton, then it will not be added. The highest value is found
 * using methods in the lane. No existing peaks will be cleared if a new
 * peak is found in the region. Also, the area of the peak is NOT 
 * calculated and is instead set to 0.
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public class HighestPeakLocate extends PeakLocate
{
  /**
   * Creats a new peak locating class that finds the highest value
   * in a range and calls it a peak.
   */
  public HighestPeakLocate()
  {
    name = "Find Highest";
    descript = "Finds the highest value in a region and records it as a peak.";
    helpFile = "highpeak.html";
  }

  /**
   * Gives the name of this peak location function
   *
   * @return the name
   */
  public String getName()
  {
    return name;
  }

  /**
   * Gives a brief description of the peak location function/algorithm.
   *
   * @return a sentence description
   */
  public String getDescription()
  {
    return descript;
  }

  /**
   * Gives the help file for the Highest peak location function
   *
   * @return an html or plaintest file with help info. <code>null</code>
   * if this file does not exist.
   */
  public String getHelpFile()
  {
    return helpFile;
  }

  /**
   * Gives the options, but since there aren't any, this is always 
   * <code>null</code>. All of the information that is needed is passed
   * into the <code>findPeaks</code> method. These are merely implemented
   * to satisfy the requirements of the parent class.
   *
   * @return <code>null</code>, always
   */
  public Option[] getOptions()
  {
    return null;
  }

  /**
   * Since this operations does not have any options, the method does not
   * do anything. It is implemented merely to satisfy the requirements of
   * the abstarct parent class that it is inheretid from.
   */
  public void setOptions(Option[] opts)
  {
  }

  /**
   * Finds a peak in each lane and saves it in the lane. The peak
   * will be between (inclusive) minSize and maxSize. The list of peaks
   * will not be cleared. Instead, peaks will simply be added to whatever
   * already exists, with the exception of duplicate peaks, which will not
   * be added. The algorithm used is described above in the class description.
   * 
   * @param lanes     the list of lanes that the algorithm should look
   *                  for peaks in.
   * @param minSize   the minimum size of any peak. If the value 
   *                  <code>MIN</code> is given, then the smallest size 
   *                  in the lane will be used.
   * @param maxSize   the maximum size of any peak. If the value
   *                  <code>MAX</code> is given, then the largest size
   *                  in the lane will be used.
   */
  public void findPeak(DataList lanes, double minSize, double maxSize)
  {
    Lane ln;
    TracePoint pt;
    Peak peak;
    DataList peaks;
    double loc; // the location in bp.

    // go through all of the lanes
    for(int i=0; i < lanes.size(); i++)
      {
     ln = (Lane)lanes.dataAt(i);

     // check the values
     if(minSize == MIN)
       minSize = ln.getMinSize();
     if(maxSize == MAX)
       maxSize = ln.getMaxSize();

     pt = ln.getMaxPoint(minSize, maxSize);

     // just make up numbers for the area
     loc = ln.getSizeFunction().getSize((int) pt.location);
     peak = new Peak(loc, pt.height, 0);

     // Make sure it doesn't already exist and then add it to the
     // list
     peaks = ln.getPeaks();
     if(peaks.find(loc).location == -1)
       {
         peaks.include(peak);
       }
      }
  }

}
