//=====================================================================
// File:    NoSize.java
// Class:   NoSize
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import AFLPcore.Option;

/**
 * This class will not provide any sizing. However, it can be useful if one
 * wants to show how different lanes ran. It works on the assumption,
 * which is obviously false, that size = scan#. Therefore, this sizing
 * function provides a way to get the scan numbers easily and without
 * using different calls at other points in a program. Since it simply 
 * assumes that they are equal, it does not need to be initialized, nor
 * does it have any options.
 *
 * @see SizeFunction
 * @see LocalSouthern
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public class NoSize extends SizeFunction
{
  /**
   * Create a new NoSize object, which will just use scan numbers as the
   * size.
   */
  public NoSize()
  {
    name = "Don't Size";
    descript = "Simply uses the scan numbers so no sizing is performed.";
    helpFile = "nosize.html";
  }

  /**
   * Gives the name describing this sizing function.
   *
   * @return the name
   */
  public String getName()
  {
    return name;
  }

  /**
   * Gives a brief description of this sizing function.
   *
   * @return a one sentence explination.
   */
  public String getDescription()
  {
    return descript;
  }

  /**
   * Gives the help file that describes the function in more detail
   *
   * @return a plaintext of html help file.
   */
  public String getHelpFile()
  {
    return helpFile;
  }

  /**
   * There are no options for this sizing function.
   *
   * @return <code>null</code>
   */
  public Option[] getOptions()
  {
    return null;
  }

  /**
   * Since there are no options, this method doesn't do much. In fact,
   * it does absolutely nothing, but it is required by the abstract class
   * <code>Operation</code>, which this class is a child of.
   */
  public void setOptions(Option[] opts)
  {
  }

  /**
   * Since this class does not really perform any sizing and instead just
   * assumes that size=scan#, there is no need to initialize it with 
   * known points. Therefore, this method does not do anything to this class.
   *
   * @param standardPeaks  a list of peaks
   */
  public void init(DataList standardPeaks)
  {
  }

  /**
   * Gives the scan number back as the size. This has the affect of not
   * sizing the lane.
   *
   * @param scan  the scan number to determine the size for.
   *
   * @return the scan number <code>scan</code>
   */
  public final double getSize(int scan)
  {
    return (double)scan;
  }

  /**
   * Gives the input value back as the scan number since the size is the
   * scan number in this case.
   *
   * @param size  the size in bp, which is also the scan number.
   *
   * @return the scan number, which is also <code>size</code>
   */
  public final int getScan(double size)
  {
    return (int)size;
  }

  /**
   * This class simply interprests the size number as the scan number,
   * so the maximum scan number is not really needed.
   *
   * @param max the maximum scan number, ignored
   */
  public void setMaxScan(int max)
  {
  }

  /**
   * Reads in the properties of this class from the specified input stream.
   * The stream data should have been created by <code>write</code>. This
   * will retrieve this classes state from the input stream. All of the 
   * current data in this class will be replaced.
   *
   * @param in  the input stream with the data for the class.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     reading from the stream.
   */
  public void read(DataInputStream in) throws IOException
  {
    // Nothing to restore really.
  }

  /**
   * Writes all of the information this class needs to store in order
   * to be recreated. This will be used for things like storing the
   * class in a file. Therefore, the write should output enough information
   * so that <code>read</code> can recreate the essential properties of this
   * class.
   *
   * @param out  the destination to write the data to.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public void write(DataOutputStream out) throws IOException
  {
    // since we don't store anything in this class, there is nothing to
    // write out
  }

}
