package ca.ucalgary.seahawk.util;

import org.biomoby.shared.MobyPrefixResolver;
import org.biomoby.shared.parser.MobyTags;

import org.w3c.dom.*;

import java.net.URL;
import java.util.*;
import javax.xml.parsers.*;

/**
 * Holds info to limit the display of information from a document to some
 * combination of a regex match and an XPath.  This info is especially useful
 * for creating a Taverna filter processor if the browsing history is 
 * exported as a workflow.  The regex and the XPath vector are mutable
 * so GUI components can update the values on the fly while editing.
 */

public class FilterSearch{

    private URL docURL;
    private boolean caseSensitivity = false;
    private boolean inverseSelection = false;
    private StringBuffer filterRegex;
    private Vector<XPathOption> xpathOptions;
    private int xpathSelected = 0;
    public static final String SELECT_ALL_XPATH = "//* | //*/@*";

    /**
     * A c-tor to manually populate the filter with a specific set of filter criteria.  
     * No autopoulation of options is done.
     */
    public FilterSearch(String regex, XPathOption xpath, boolean case_sensitive, boolean inverse_selection){
	filterRegex = new StringBuffer(regex);
	xpathOptions = new Vector<XPathOption>();
	xpathOptions.add(xpath);
	setCaseSensitivity(case_sensitive);
	setSelectionInversed(inverse_selection);
    }

    /**
     * Autopopulates the possible search field values based on the incoming doc
     *
     * @param docToFilter the document that will be used to determine what XPaths can be suggested
     * @param docBuilder the xml parser to use on the document
     *
     * @throws Exception if the URL given could not be loaded
     */
    public FilterSearch(URL docToFilter, DocumentBuilder docBuilder) throws Exception{
	docURL = docToFilter;
	filterRegex = new StringBuffer();
	xpathOptions = new Vector<XPathOption>();
	xpathOptions.add(new XPathOption(SELECT_ALL_XPATH, "entire response"));
	Document domDoc = docBuilder.parse(docURL.openStream());
	// Get the moby jobs
	NodeList jobs = domDoc.getDocumentElement().getElementsByTagNameNS(MobyPrefixResolver.MOBY_XML_NAMESPACE, MobyTags.MOBYDATA);
	// Get the unique element names within the responses
	HashMap<String,String> memberNames = new HashMap<String,String>();
	HashMap<String,String> elementNames = new HashMap<String,String>();
	for(int i = 0; i < jobs.getLength(); i++){
	    Element job = (Element) jobs.item(i);
	    NodeList dataElements = job.getElementsByTagName("*");
	    for(int j = 0; j < dataElements.getLength(); j++){
		Element data = (Element) dataElements.item(j);
		String articleName = data.getAttribute(MobyTags.ARTICLENAME);
		if(!data.getLocalName().equals(MobyTags.COLLECTION) &&
		   !data.getLocalName().equals(MobyTags.SIMPLE) &&
		   !elementNames.containsKey(data.getLocalName())){ //todo: add nested element xpaths?
		    if(data.getTextContent().trim().length() != 0){  // has text content
			elementNames.put("//"+data.getLocalName(), "element " + data.getLocalName());
		    }
		    else if(data.getAttribute(MobyTags.OBJ_ID).length() > 0){  // or an id
			elementNames.put("//*/@id", "ID");
		    }
		}
		if(articleName != null && articleName.trim().length() != 0 && 
		   !memberNames.containsKey(articleName)){
		    memberNames.put("//*[@"+MobyTags.ARTICLENAME+"='"+articleName+"']", "field "+articleName);
		}
	    }
	}

	for(Map.Entry<String,String> option: memberNames.entrySet()){
	    xpathOptions.add(new XPathOption(option.getKey(), option.getValue()));
	}
	for(Map.Entry<String,String> option: elementNames.entrySet()){
	    xpathOptions.add(new XPathOption(option.getKey(), option.getValue()));
	}
    }

    public boolean getSelectionInversed(){
	return inverseSelection;
    }

    public void setSelectionInversed(boolean inversed){
	inverseSelection = inversed;
    }

    public boolean getCaseSensitivity(){
	return caseSensitivity;
    }

    public void setCaseSensitivity(boolean sensitivity){
	caseSensitivity = sensitivity;
    }

    public void setSelectedXPath(int cursor){
	xpathSelected = cursor;
    }

    public XPathOption getSelectedXPath(){
	return xpathOptions.elementAt(xpathSelected);
    }

    public StringBuffer getFilterRegex(){
	return filterRegex;
    }

    public Vector<XPathOption> getXPathOptions(){
	return xpathOptions;
    }

    public String toString(){
	return filterRegex.toString()+"\t"+getSelectedXPath().getXPath()+"\t"+
	    getSelectedXPath().getDesc()+"\t"+caseSensitivity+"\t"+inverseSelection;
    }

    public boolean equals(Object o){
        if(o == null || !(o instanceof FilterSearch)){
          return false;
        }
        FilterSearch other = (FilterSearch) o;
	//System.err.println("Comparing "+this+"\n\nand\n\n"+o);
        return filterRegex.toString().equals(other.filterRegex.toString()) &&
               getSelectedXPath().getXPath().equals(other.getSelectedXPath().getXPath()) &&
               caseSensitivity == other.caseSensitivity &&
	       inverseSelection == other.inverseSelection;
    }
}
