package ca.ucalgary.services.util;

import java.io.*;
import java.net.URL;
import java.util.*;

/**
 * A class representing the contents of an ACD file (EMBOSS program parameter specification).
 * For more details on the format, please see http://emboss.sourceforge.net/developers/acd/
 */

public class ACDFile{
    private List<Map<String,String>> application;
    private List<Map<String,String>> input;
    private List<Map<String,String>> required;
    private List<Map<String,String>> additional;
    private List<Map<String,String>> advanced;
    private List<Map<String,String>> output;

    // Used for ACD parsing
    public final static String BLOCK_TYPE_KEY = "_blocktype";
    public final static String BLOCK_NAME_KEY = "_blockname";

    /**
     * Reads an EMBOSS ACD file and populates data structures representing the sections of the file
     * (see the getXXXSection() methods).
     *
     * @throws Exception if the file cannot be read or parsed
     */
    public ACDFile(File acdFile) throws Exception{
	if(!acdFile.exists()){
	    throw new Exception("The given ACD path (" + acdFile.toString() + 
				") does not exist on the filesystem");
	}
	if(!acdFile.isFile()){
	    throw new Exception("The given ACD path (" + acdFile.toString() + 
				") exists, but is not a file");
	}
	
	parse(acdFile.toURI().toURL().openStream());
    }

    public ACDFile(URL acdURL) throws Exception{
	parse(acdURL.openStream());
    }

    public ACDFile(InputStream is) throws Exception{
	parse(is);
    }

    private void parse(InputStream is) throws Exception{
	LineNumberReader reader = new LineNumberReader(new InputStreamReader(is));

	StringBuffer applicationSection = new StringBuffer();
	StringBuffer inputSection = new StringBuffer();
	StringBuffer requiredSection = new StringBuffer();
	StringBuffer additionalParamsSection = new StringBuffer();
	StringBuffer advancedParamsSection = new StringBuffer();
	StringBuffer outputSection = new StringBuffer();

    	for(String line = reader.readLine(); line != null; line = reader.readLine()){

	    // Application information
	    if(line.matches("^\\s*application\\s*:\\s*\\S+\\s*\\[")){
		applicationSection.append(line+"\n");
		for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			applicationSection.append(line+"\n");
			break;
		    }
		    applicationSection.append(line+"\n");
		}		
	    }
	    
	    // Input definition
	    else if(line.matches("^\\s*section\\s*:\\s*input\\s*\\[")){
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			break;
		    }
		}
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*endsection\\s*:\\s*input")){
			break;
		    }
		    inputSection.append(line+"\n");
		}
	    }

	    else if(line.matches("^\\s*section\\s*:\\s*required\\s*\\[")){
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			break;
		    }
		}
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*endsection\\s*:\\s*required")){
			break;
		    }
		    requiredSection.append(line+"\n");
		}
	    }
	    
	    else if(line.matches("^\\s*section:\\s*additional\\s*\\[")){
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			break;
		    }
		}
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*endsection\\s*:\\s*additional")){
			break;
		    }
		    additionalParamsSection.append(line+"\n");
		}
	    }

	    else if(line.matches("^\\s*section\\s*:\\s*advanced\\s*\\[")){
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			break;
		    }
		}
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*endsection\\s*:\\s*advanced")){
			break;			
		    }
		    advancedParamsSection.append(line+"\n");
		}
	    }

	    else if(line.matches("^\\s*section\\s*:\\s*output\\s*\\[")){
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			break;
		    }
		}
                for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*endsection\\s*:\\s*output")){
			break;
		    }
		    outputSection.append(line+"\n");
		}
	    }
	}  // end for lines in file

	// Populate the sections
	application = acdSectionToTagValuePairs(applicationSection.toString());
	input = acdSectionToTagValuePairs(inputSection.toString());
	required = acdSectionToTagValuePairs(requiredSection.toString());
	additional = acdSectionToTagValuePairs(additionalParamsSection.toString());
	advanced = acdSectionToTagValuePairs(advancedParamsSection.toString());
	output = acdSectionToTagValuePairs(outputSection.toString());

    }

    public List<Map<String,String>> getApplicationSection(){
	return application;
    }
    
    /**
     * Returns a list representing the blocks of the section.  Each block
     * is a map of key/value pairs corresponding to the tag-name/tag-value pairs
     * of the ACD block.  Two hardcoded keys are the block name (BLOCK_NAME_KEY) 
     * such as "sequence", "trim", "clean"
     * and the block type (BLOCK_TYPE_KEY) such as "list", "boolean", "range", etc.
     */
    public List<Map<String,String>> getInputSection(){
	return input;
    }

    public List<Map<String,String>> getRequiredParamsSection(){
	return required;
    }

    public List<Map<String,String>> getAdditionalParamsSection(){
	return additional;
    }

    public List<Map<String,String>> getAdvancedParamsSection(){
	return advanced;
    }

    public List<Map<String,String>> getOutputSection(){
	return output;
    }

    /**
     * A section is made out of 0 or more blocks
     */
    private List<Map<String,String>> acdSectionToTagValuePairs(String acdSection) throws Exception{
	List<Map<String,String>> blocks = new ArrayList<Map<String,String>>();
	LineNumberReader reader = new LineNumberReader(new StringReader(acdSection));

	for(String line = reader.readLine(); line != null; line = reader.readLine()){
	    if(line.matches("^\\s*$")){ // Ignore blank lines
		continue;
	    }   

	    StringBuffer blockText = new StringBuffer();
	    if(line.matches("^\\s*\\S+\\s*:\\s*\\S+\\s*\\[\\s*$")){
		String blockType = line.replaceAll("^\\s*(\\S+)\\s*:\\s*\\S+\\s*\\[\\s*$", "$1");
		String blockName = line.replaceAll("^\\s*\\S+\\s*:\\s*(\\S+)\\s*\\[\\s*$", "$1");
		boolean subsection = line.matches("^\\s*section\\s*:\\s*\\S*\\s*\\[");

		for(line = reader.readLine(); line != null; line = reader.readLine()){
		    if(line.matches("^\\s*]")){
			if(subsection){
			    //System.err.println("Ignoring subsection:\n" + blockText.toString());
			    break; // ignore subsection specifications (MOBY 2ndary params don't hav'em)
			}
			Map<String,String> block = acdBlockToTagValuePairs(blockText.toString());
			block.put(BLOCK_TYPE_KEY, blockType);
			block.put(BLOCK_NAME_KEY, blockName);
			blocks.add(block);
			break;
		    }
		    blockText.append(line+"\n");
		}		
	    }
	    else if(line.matches("^\\s*variable:\\s*\\S+\\s+\".*\"\\s*$")){
		// Ignore variable declarations (for the moment)
		// since EMBOSS will figure out their values for itself,
		// they are not passed in
            }
	    else if(line.matches("^\\s*endsection:\\s*\\S+")){
		// ignore subsection info
            }
	    else{
		throw new Exception("Unexpected content where blocks expected: "+line);
	    }
	}

	return blocks;
    }

    /**
     * Basic unit of parameter specification in an ACD file. e.g.:
     *
     * seqset: sequence [
     *   parameter: "Y"
     *   type: "gapany"
     *   aligned: "Y"
     *   help: "The sequence alignment to be displayed."
     * ]
     */
    private Map<String,String> acdBlockToTagValuePairs(String acdBlock) throws Exception{
	Map<String,String> tagValuePairs = new HashMap<String,String>();
	LineNumberReader reader = new LineNumberReader(new StringReader(acdBlock));

	for(String line = reader.readLine(); line != null; line = reader.readLine()){
	    if(line.matches("^\\s*$")){  // Ignore blank lines
		continue;
	    }

	    String param = line.replaceAll("^\\s*(\\S+)\\s*:\\s*\".*$", "$1");
	    if(param.equals(line)){
		throw new Exception("Expected format (tag: \"value\") not found in ACD file line: " + line);
	    }

	    String value = line.replaceAll("^\\s*\\S+\\s*:\\s*\"(.*)\"$", "$1");
	    if(value.equals(line)){
		value = line.replaceAll("^\\s*\\S+\\s*:\\s*\"(.*)$", "$1");
		for(line = reader.readLine(); line != null; line = reader.readLine()){
		    value += line;
		    if(line.indexOf("\"") != -1){
			value = value.replaceAll("\\s{2,50}", " ");  // Normalize spaces
			value = value.replaceAll("\"\\s*", "");  // Strip ending quote mark
			value = value.replaceAll("\\\\", "\n");  //ACD uses '\' to denote literal new-line
			break;
		    }
		}
		
	    }
	    tagValuePairs.put(param, value);	
	}

	return tagValuePairs;
    }
}
