package embl.ebi.trace;

import java.awt.*;
import java.applet.*;
import java.awt.event.*;
// import java.beans.*;


/**
Now that the sequence trace viewer needs extended user-interactivity,
this class implements a controller in a Model, View, Controller design pattern.
@see Chromatogram
@see ChromatogramApplet
@author Eugen Beuhler
@author Jeremy D. Parsons
*/
public class ChromatogramViewer extends Panel implements ActionListener, AdjustmentListener, ItemListener{
	
	// View objects
	private ChromatogramCanvas chromArea;
	private Scrollbar 	hBar = null;
	private Scrollbar 	vBar = null;
	private Chromatogram 	chromData = null;
	private	Panel		cardPanel = new Panel();
	private	Panel		chromPanel = new Panel();
	private	Panel		controlPanel = new Panel();
	private	CardLayout	myLayout = new CardLayout();
	private	TextArea 	seqBox = new TextArea();
	private	TextArea 	commentBox = new TextArea();
	private	String		visibleView;

	// Controller objects
   	private Button		revButton;
	private String[]	orientationIndicatorStrings = {"Original", "Complement"};
	private Label		orientationIndicator;
	private TextField	database, filename;
	private	Choice		viewChoice;

	static final int	V_BAR_MAX = 70;
	static final int V_BAR_MIN = 1;
	static final int	V_BAR_BLOCK = 10;
	static final int	V_BAR_UNIT = 1;
	static final int	V_BAR_DEFAULT = V_BAR_MAX - V_BAR_BLOCK;
	static final int	V_BAR_VISIBLE = 0;

	static final String	CHROMATOGRAM_VIEW = "Trace";
	static final String	SEQUENCE_VIEW = "Sequence";
	static final String	COMMENT_VIEW = "Comments";
	
	public ChromatogramViewer() {
		super();
		// Main screen is divided into two areas: controls and switchable card views
		this.setLayout(new BorderLayout(0,0));
		this.add("North", controlPanel);
		this.add("Center", cardPanel);
		this.createControls();
		this.createViews();
		}

	void createControls(){
		// First set up the controls
		controlPanel.setLayout(new FlowLayout(FlowLayout.LEFT));

		controlPanel.add(new Label("DB:"));
		database = new TextField(10);
		database.addActionListener(this);
		controlPanel.add(database);

		controlPanel.add(new Label("Acc:"));
		filename = new TextField(10);
		filename.addActionListener(this);
		controlPanel.add(filename);

		revButton = new Button("Complement");
		revButton.addActionListener(this);
		revButton.setForeground(Color.black);
		revButton.setBackground(Color.lightGray);
		controlPanel.add(revButton);

		orientationIndicator = new Label(orientationIndicatorStrings[0]);
		controlPanel.add(orientationIndicator);

		viewChoice = new Choice();
		viewChoice.addItemListener(this);
		viewChoice.add(CHROMATOGRAM_VIEW);
		viewChoice.add(SEQUENCE_VIEW);
		viewChoice.add(COMMENT_VIEW);
		controlPanel.add(viewChoice);

		}

	void createViews() {
		// Now set up the views
		cardPanel.setLayout(myLayout);

		// The sequence text view
		seqBox.setFont(new Font("Monospaced", Font.PLAIN, 12));
		cardPanel.add(SEQUENCE_VIEW, seqBox);

		// The comment text view
		commentBox.setFont(new Font("Monospaced", Font.PLAIN, 12));
		cardPanel.add(COMMENT_VIEW, commentBox);

		// The chromatogram (trace) view
		chromArea = new ChromatogramCanvas();
		chromArea.emptyTrace();
		hBar = new Scrollbar(Scrollbar.HORIZONTAL);
		hBar.addAdjustmentListener(this);
		vBar = new Scrollbar(Scrollbar.VERTICAL);
		vBar.addAdjustmentListener(this);
		vBar.setVisibleAmount(V_BAR_VISIBLE);
		vBar.setValue(V_BAR_DEFAULT);
		chromPanel.setLayout(new BorderLayout(0,0));
		chromPanel.add("Center", chromArea);
		chromPanel.add("South", hBar);
		chromPanel.add("East", vBar);
		cardPanel.add(CHROMATOGRAM_VIEW, chromPanel);
		myLayout.show(cardPanel, CHROMATOGRAM_VIEW);
		visibleView = CHROMATOGRAM_VIEW;
		}
	
	/**
	This method is usually called from the controller above when a new trace
	is ready for display so: get new data and update display.
	*/
	public void loadTrace(Chromatogram chromData){
		this.chromData = chromData;
		chromArea.loadTrace(chromData);
		updateViews();
		vBar.setValue(V_BAR_DEFAULT);
		setAmplitude(V_BAR_DEFAULT);
		}

	public Dimension getMinimumSize() {
		return new Dimension(100,100);
		}

	public void adjustmentValueChanged(AdjustmentEvent evt) {
		// Which scrollbar ?
		if (evt.getSource() == hBar){
			chromArea.setOffset(evt.getValue());
			}
		if (evt.getSource() == vBar){
			int	value = evt.getValue()+ vBar.getVisibleAmount();
			setAmplitude(value);
			}
		chromArea.repaint();
		}
	
	public void itemStateChanged(ItemEvent evt) {
		visibleView = (String)evt.getItem();
		updateViews();
		}

	void updateViews(){
		// Something has changed and the visible view needs repainting
		Dimension dimension = this.getSize();
		dimension.width = (dimension.width <= getMinimumSize().width ? getMinimumSize().width : dimension.width);
		if (chromData == null){
			seqBox.setText("Not loaded a chromatogram yet\n");
			}
		else	{
			database.setText(chromData.getDatabaseName());
			filename.setText(chromData.getAccession());
			if (visibleView.equals(CHROMATOGRAM_VIEW)){
				chromArea.emptyTrace();
				chromArea.loadTrace(chromData);
				fitScrollBars(dimension.width);
				int realMaximum = hBar.getMaximum() - hBar.getVisibleAmount();
				if (chromArea.getOffset() > realMaximum){
					chromArea.setOffset(realMaximum);
					}
				chromArea.repaint();
				}
			else	{
				if (visibleView.equals(SEQUENCE_VIEW)){
					// provide for wrapped text in sequence display
					int charWidth = (seqBox.getFontMetrics(seqBox.getFont())).stringWidth("A");
					int numChars = (dimension.width - 40) / charWidth;
					StringBuffer buf = new StringBuffer(chromData.getSequence());
					int insertPoint = numChars - 1;
					try	{
						while (insertPoint < buf.length()) {
							buf.insert(insertPoint, "\n");
							insertPoint += numChars;
							}
						}
					catch (StringIndexOutOfBoundsException err) {
						System.err.println("Sequence string out of bounds!");
						}
					seqBox.setText(buf.toString());
					}
				else	{
					if (visibleView.equals(COMMENT_VIEW)){
						commentBox.setText(chromData.getComments());
						}
					}
				}
			}
		myLayout.show(cardPanel, visibleView);
		}
	
	// Handle size change
	public synchronized void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		updateViews();
		controlPanel.repaint();
		}

	public synchronized void fitScrollBars(int width) {
		if (chromData != null){	
			// hBar.setMaximum(chromData.getTraceLength() - width);
			hBar.setMaximum(chromData.getTraceLength());
			hBar.setMinimum(0);
			hBar.setVisibleAmount(width);
			hBar.setBlockIncrement(width - 1);
			hBar.setUnitIncrement((width / 50) + 1);

			vBar.setMaximum(V_BAR_MAX);
			vBar.setMinimum(V_BAR_MIN);
			//vBar.setVisibleAmount(V_BAR_BLOCK);
			setAmplitude(vBar.getValue());
			vBar.setBlockIncrement(V_BAR_BLOCK);
			vBar.setUnitIncrement(V_BAR_UNIT);
			/*
			System.err.println("DEBUG: Scrollbar values= " + 
				hBar.getMaximum() + " " +
				hBar.getMinimum() + " " +
				hBar.getVisibleAmount() + " " +
				hBar.getBlockIncrement() + " " +
				hBar.getUnitIncrement() + " " +
				"offset= " + chromArea.getOffset() + " " +
				"width= " + width
				);
			*/
			}
		}

	public void setAmplitude(int scrollPos){
		float magnifier = (float)(V_BAR_MAX - scrollPos) / 10.0F;
		chromArea.setMagnifier(magnifier);
		/* System.err.println("DEBUG: V Scrollbar value= " + scrollPos 
			+ " magnifier= " + magnifier);
		*/
		}

	public void setOrientationIndicator(boolean originalOrientation) {
		// This method should be called every time the sequence flips
		orientationIndicator.setText(originalOrientation ? orientationIndicatorStrings[0] : orientationIndicatorStrings[1]);
		repaint();
		}

	public void actionPerformed(ActionEvent event) {
		Object eventSource = event.getSource();
		setCursor(new Cursor(Cursor.WAIT_CURSOR));
		if (eventSource == revButton){
			if (chromData != null){
				chromData.complement();
				this.setOrientationIndicator(chromData.getOriginalOrientation());
				}
			}
		if (eventSource == database) {
			this.newTrace(database.getText(), filename.getText());
			}
		else	{
			if (eventSource == filename) {
				this.newTrace(database.getText(), filename.getText());
				}
			}
		updateViews();
		setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		}
	/**
	This method is called when the user interface has created an event
	(ActionEvent) and the submitted user request needs to be checked
	before being passed oon up to the real controller.
	*/	
	public void newTrace(String database, String traceName){
		String	fullPathname = null;

		// Check that both parts are set
		if (database.equals("") || traceName.equals("")){
			return;
			}
		chromArea.emptyTrace();
		chromData = null;
		chromArea.setOffset(0); // Start at the begining
		hBar.setValue(0);
		// Send the parent an event to say that a new sequence is wanted
		TraceRequestEvent realRequest = new TraceRequestEvent(this, database, traceName);
		((TraceRequestListener)this.getParent()).traceRequested(realRequest);
		}
}
